/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to create and use a simple volumeData and volumeSkin node.
author : Jerome Hummel
August 2002
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeSkin.h>
#include <LDM/nodes/SoDataRange.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>

//main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;
    
  // Initialize of VolumeViz extension
  SoVolumeRendering::init();

  //Volume Size
  // Note this is the same (tiny) volume used in the simpleVolume example
  const int dimX = 10;
  const int dimY = 10;
  const int dimZ = 10;

  // Create data :
  // Note: 0 < data[i] < number of entries in the colorMap.
  // If each data value has the same value A the volume will be unicolor
  // with the color described by the Ath entries of the colorMap.
  // This volume data will use every color in the colorMap
  unsigned char *data = new unsigned char[dimX * dimY * dimZ];
  for( int i=0; i<dimX * dimY * dimZ; i++ ) {
    data[i] = i%256;
  }

  // Node to hold the volume data
  // In the first call we specify the voxel dimensions and the actual data.
  // In the second call we specify the 3D extent of the volume in modeling coords.
  SoVolumeData* pVolData = new SoVolumeData();
  pVolData->data.setValue( SbVec3i32(dimX, dimY, dimZ),SbDataType::UNSIGNED_BYTE, 8, data, SoSFArray3D::NO_COPY_AND_DELETE);
  pVolData->extent.setValue( SbBox3f((float)-dimX, (float)-dimY, (float)-dimZ, (float)dimX, (float)dimY, (float)dimZ) );

  // If necessary, specify the actual range of the data values.
  //    By default VolumeViz maps the entire range of the voxel data type
  //    (e.g. 0..65535 for unsigned short) into the colormap.  This works
  //    great for byte (8 bit) voxels, but not so well for 16 bit voxels
  //    and not at all for floating point voxels. So it's not actually
  //    necessary for this data set, but shown here for completeness.
  //    NOTE: Min/max values are stored in the header for LDM format
  //    files, but for other formats the getMinMax query can take a
  //    long time because VolumeViz has to examine every voxel.
  SoDataRange *pRange = new SoDataRange();
  int voxelSize = pVolData->getDataSize();
  if (voxelSize > 1) {
    double minval, maxval;
    pVolData->getMinMax( minval, maxval );
    pRange->min = minval;
    pRange->max = maxval;
  }

  // Use a predefined colorMap with the SoTransferFunction
  SoTransferFunction* pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::STANDARD;

  // Node in charge of drawing the volume
  SoVolumeSkin *pVolRender = new SoVolumeSkin;

  // Assemble the scene graph
  // Note: SoVolumeSkin must appear after the SoVolumeData node.
  SoSeparator *root = new SoSeparator;
  root->ref();
  root->addChild( pVolData );
  root->addChild( pRange );
  root->addChild( pTransFunc );
  root->addChild( pVolRender );

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("Volume Skin");
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}


