/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to create and use a simple volume transform and volumeRender node.
author : David Beilloin
september 2008
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <DialogViz/SoDialogVizAll.h>

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoRotation.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoFontStyle.h>
#include <Inventor/nodes/SoCallback.h>

#include <Inventor/STL/iostream>
#include <Inventor/STL/sstream>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoVolumeGroup.h>
#include <LDM/nodes/SoDataSetId.h>

#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/nodes/SoMultiDataSeparator.h>
#include <LDM/manips/SoROIManip.h>

// Custom volume transform
#include "volumeTransformer.h"

// Forward declarations
Widget buildInterface(Widget);

// Global variables
#define DIALOG_FILENAME "$OIVHOME/examples/source/VolumeViz/volumeTransform/volumeTransform_gui.iv"
#define VOLUME_FILENAME "$OIVHOME/examples/data/VolumeViz/3DHEAD.ldm"
#define NB_VOLUME 2
#define TRANSFORM_TILE_CACHE 16

SoSwitch *testSwitch = NULL;
volumeTransformer *volumerTransformers[NB_VOLUME];

// The volume Data
SoVolumeData* pVolData;

SoText2* g_infoText;

/*---------------------------------------------------------------------------*/
SoSeparator*
displayInfo()
{
  // Informations
  SoSeparator *infoSep = new SoSeparator ;

  SoOrthographicCamera *cam = new SoOrthographicCamera ;
  infoSep->addChild(cam) ;
  cam->viewportMapping = SoOrthographicCamera::LEAVE_ALONE ;
  cam->position.setValue(SbVec3f(0.005f,0.005f,1)) ;
  cam->height = 0.01f ;

  SoPickStyle *pickStyle = new SoPickStyle ;
  pickStyle->style = SoPickStyle::UNPICKABLE ;
  infoSep->addChild(pickStyle) ;

  SoSeparator *textBgSep = new SoSeparator;
  SoMaterial *textBgMat = new SoMaterial;
  textBgMat->transparency = 0.7f;
  textBgMat->diffuseColor = SbColor(0,0,0);

  SoCube *textBg = new SoCube;
  textBg->width = 1.0f;
  textBg->height = 0.0008f;
  textBgSep->addChild(textBgMat);
  textBgSep->addChild(textBg);

  SoLightModel *lModel = new SoLightModel ;
  lModel->model = SoLightModel::BASE_COLOR ;
  infoSep->addChild(lModel) ;

  SoFontStyle *fontInfo = new SoFontStyle ;
  fontInfo->name = "Courier New" ;
  fontInfo->size = 14 ;
  fontInfo->style = SoFontStyle::BOLD;
  fontInfo->renderStyle = SoFontStyle::TEXTURE;
  infoSep->addChild(fontInfo) ;

  SoBaseColor *infoColor = new SoBaseColor ;
  infoColor->rgb.setValue(SbColor(1, 1, 1.f)) ;
  infoSep->addChild(infoColor) ;

  SoTranslation *transInfo = new SoTranslation ;
  transInfo->translation.setValue(0.00025f, 0.00975f, 0.) ;
  infoSep->addChild(transInfo) ;

  g_infoText = new SoText2 ;
  g_infoText->string.set1Value(0, "LDM Memory used: 0 MB") ;
  infoSep->addChild(textBgSep);
  infoSep->addChild(g_infoText) ;

  return infoSep ;
}

SoSeparator*
singleSliceTransform(const SbString &/*filename*/, const int nbVolume)
{
  SbString caseName = "Transform_";
  caseName += SbString(nbVolume);
  caseName += "slice(s)";

  // Root separator
  SoSeparator* volGroup = new SoSeparator;
  volGroup->setName(caseName);

  for (int nbsub=0; nbsub<nbVolume; nbsub++)
  {
    // DataSetId need to be different to uniquely identifies each data set.
    SoDataSetId* dsId = new SoDataSetId;
    dsId->id = nbsub+1;

    SbVec3i32 dimensions = pVolData->data.getSize();
    SoROIManip* pROIManip = new SoROIManip();
    pROIManip->box.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
    pROIManip->subVolume.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
    pROIManip->constrained = TRUE;
    pROIManip->boxOn = FALSE;
    pROIManip->subVolume.setValue(
      0,nbsub*(dimensions[1]-1)/nbVolume,0,
      dimensions[0]-1,(nbsub+1)*(dimensions[1]-1)/nbVolume,dimensions[2]-1
      );


    // Use a predefined colorMap with the SoTransferFunction
    SoTransferFunction* pTransFunc = new SoTransferFunction;
    pTransFunc->predefColorMap = SoTransferFunction::STANDARD;
    pTransFunc->minValue = 50;
    pTransFunc->transferFunctionId.connectFrom( &dsId->id );

    // Node in charge of drawing the slice
    SoOrthoSlice* pOrthoSlice = new SoOrthoSlice;
    pOrthoSlice->sliceNumber.setValue(dimensions[2]/2);

    // Assemble the scene graph
    SoSeparator* subroot = new SoSeparator;
    // Original volumeRender
    subroot->addChild( dsId );
    subroot->addChild( pVolData );
    subroot->addChild( pROIManip );
    subroot->addChild( volumerTransformers[nbsub]);
    subroot->addChild( pTransFunc );
    subroot->addChild( pOrthoSlice );

    volGroup->addChild(subroot);
  }
  return volGroup;
}

SoSeparator*
multiSliceTransform(const SbString &/*filename*/, const int nbVolume)
{
  SbString caseName = "MultiData_transform_";
  caseName += SbString(nbVolume);
  caseName += "slice(s)";

  // Root separator
  SoSeparator* volGroup = new SoSeparator;
  volGroup->setName(caseName);

  for (int nbsub=0; nbsub<nbVolume; nbsub++)
  {
    SoDataSetId* dsId = new SoDataSetId;

    // Id
    dsId->id = nbsub+1;

    // Use a predefined colorMap with the SoTransferFunction
    SoTransferFunction* pTransFunc = new SoTransferFunction;
    pTransFunc->predefColorMap = SoTransferFunction::STANDARD;
    pTransFunc->minValue = 50;
    pTransFunc->transferFunctionId.connectFrom( &dsId->id );

    // allocate our own VolumeTransform as we need to change the volumeTransformId
    volumeTransformer * pVolTransform = new volumeTransformer;
    pVolTransform->volumeTransformId.setValue(nbsub+1);
    pVolTransform->cacheSize.setValue(TRANSFORM_TILE_CACHE);
    pVolTransform->scaleFactor.connectFrom( &(volumerTransformers[nbsub]->scaleFactor));

    volGroup->addChild( dsId );
    volGroup->addChild( pVolData );
    volGroup->addChild( pVolTransform );
    volGroup->addChild( pTransFunc );
  }

  SbVec3i32 dimensions = pVolData->data.getSize();
  SoROIManip* pROIManip = new SoROIManip();
  pROIManip->box.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  pROIManip->subVolume.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  pROIManip->constrained = TRUE;
  pROIManip->boxOn = FALSE;
  pROIManip->subVolume.setValue(
    0,dimensions[1]-1,0,
    dimensions[0]-1,dimensions[1]-1,dimensions[2]-1
    );
  volGroup->addChild(pROIManip);

  // Node in charge of drawing the volume
  SoOrthoSlice* pOrthoSlice = new SoOrthoSlice;
  pOrthoSlice->sliceNumber.setValue(pVolData->data.getSize()[2]/2);

  volGroup->addChild( pOrthoSlice );

  return volGroup;
}


SoSeparator*
singleVolumeRenderTransform(const SbString &/*filename*/, const int nbVolume)
{
  SbString caseName = "Transform_";
  caseName += SbString(nbVolume);
  caseName += "volRender(s)";

  // Root separator
  SoSeparator* sep = new SoSeparator;
  sep->setName(caseName);

  SoVolumeGroup* volGroup = new SoVolumeGroup;
  sep->addChild(volGroup);

  for (int nbsub=0; nbsub<nbVolume; nbsub++)
  {
    // DataSetId need to be different to uniquely identifies each data set.
    SoDataSetId* dsId = new SoDataSetId;
    dsId->id = nbsub+1;

    SbVec3i32 dimensions = pVolData->data.getSize();
    SoROIManip* pROIManip = new SoROIManip();
    pROIManip->box.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
    pROIManip->subVolume.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
    pROIManip->constrained = TRUE;
    pROIManip->boxOn = FALSE;
    pROIManip->subVolume.setValue(
      0,nbsub*(dimensions[1]-1)/nbVolume,0,
      dimensions[0]-1,(nbsub+1)*(dimensions[1]-1)/nbVolume,dimensions[2]-1
      );


    // Use a predefined colorMap with the SoTransferFunction
    // Use a transparent colorMap to activate volumeGroup specific rendering
    SoTransferFunction* pTransFunc = new SoTransferFunction;
    pTransFunc->predefColorMap = SoTransferFunction::SEISMIC;
    pTransFunc->minValue = 50;
    pTransFunc->transferFunctionId.connectFrom( &dsId->id );

    // Node in charge of drawing the volume
    SoVolumeRender* pVolRender = new SoVolumeRender;

    volumerTransformers[nbsub]->volumeTransformId.connectFrom( &dsId->id );

    // Assemble the scene graph
    SoSeparator* subroot = new SoSeparator;
    // Original volumeRender
    subroot->addChild( dsId );
    subroot->addChild( pVolData );
    subroot->addChild( pROIManip );
    subroot->addChild( volumerTransformers[nbsub]);
    subroot->addChild( pTransFunc );
    subroot->addChild( pVolRender );

    volGroup->addChild(subroot);
  }
  return sep;
}

SoSeparator*
multiVolumeRenderTransform(const SbString &/*filename*/, const int nbVolume)
{
  SbString caseName = "MultiData_transform_";
  caseName += SbString(nbVolume);
  caseName += "volRender(s)";

  // Root separator
  SoSeparator* sep = new SoSeparator;
  sep->setName(caseName);

  SoVolumeGroup* volGroup = new SoVolumeGroup;
  sep->addChild(volGroup);

  for (int nbsub=0; nbsub<nbVolume; nbsub++)
  {
    SoDataSetId* dsId = new SoDataSetId;

    // Id
    dsId->id = nbsub+1;

    // Use a predefined colorMap with the SoTransferFunction
    SoTransferFunction* pTransFunc = new SoTransferFunction;
    pTransFunc->predefColorMap = SoTransferFunction::SEISMIC;
    pTransFunc->minValue = 50;
    pTransFunc->transferFunctionId.connectFrom( &dsId->id );

    // allocate our own VolumeTransform as we need to change the volumeTransformId
    volumeTransformer * pVolTransform = new volumeTransformer;
    pVolTransform->volumeTransformId.setValue(nbsub+1);
    pVolTransform->cacheSize.setValue(TRANSFORM_TILE_CACHE);
    pVolTransform->scaleFactor.connectFrom( &(volumerTransformers[nbsub]->scaleFactor));

    volGroup->addChild( dsId );
    volGroup->addChild( pVolData );
    volGroup->addChild( pVolTransform );
    volGroup->addChild( pTransFunc );
  }

  SbVec3i32 dimensions = pVolData->data.getSize();
  SoROIManip* pROIManip = new SoROIManip();
  pROIManip->box.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  pROIManip->subVolume.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  pROIManip->constrained = TRUE;
  pROIManip->boxOn = FALSE;
  pROIManip->subVolume.setValue(
    0,dimensions[1]-1,0,
    dimensions[0]-1,dimensions[1]-1,dimensions[2]-1
    );
  volGroup->addChild(pROIManip);

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;
  volGroup->addChild( pVolRender );

  return sep;
}

//main function
int main(int argc, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  SoPreferences::setBool("OIV_PERFCOUNTER_ENABLE",TRUE);

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();
  SoDialogViz::init();

  // Initialize our custom volume transform node
  volumeTransformer::initClass();

  const char *filename = (argc >= 2 ? argv[1] : VOLUME_FILENAME);

  // The volume Data
  pVolData = new SoVolumeData();
  pVolData->fileName = filename;

  SoMultiDataSeparator *rootSep = new SoMultiDataSeparator;
  rootSep->ref();

  SoPerspectiveCamera* cam = new SoPerspectiveCamera ;
  rootSep->addChild(cam);

  SoRotation *pRotation = new SoRotation;
  pRotation->rotation.setValue(0.0f, 0.0f , 3.14f, 0.0f);
  rootSep->addChild(pRotation);

  // Create volumeTransformer
  for (int i=0;i<NB_VOLUME;++i)
  {
    volumerTransformers[i] = new volumeTransformer;
    volumerTransformers[i]->scaleFactor.setValue( 2.0*(i+1)/(double)NB_VOLUME );
    volumerTransformers[i]->cacheSize.setValue(TRANSFORM_TILE_CACHE);
  }

  testSwitch = new SoSwitch;
  testSwitch->addChild(singleVolumeRenderTransform(filename,1));
  testSwitch->addChild(singleVolumeRenderTransform(filename,NB_VOLUME));
  testSwitch->addChild(multiVolumeRenderTransform(filename,NB_VOLUME));

  testSwitch->addChild(singleSliceTransform(filename,1));
  testSwitch->addChild(singleSliceTransform(filename,NB_VOLUME));
  testSwitch->addChild(multiSliceTransform(filename,NB_VOLUME));

  rootSep->addChild(testSwitch);
  testSwitch->whichChild.setValue(4);

  rootSep->addChild(displayInfo());

  // Set up viewer:
  // Create a simple user interface to move the slice
  Widget parent = buildInterface(myWindow);

  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(parent);
  myViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
  myViewer->setSceneGraph(rootSep);
  myViewer->setTitle("Volume transform rendering");
  myViewer->viewAll();
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();

  delete myViewer;
  rootSep->unref();

  double bandwidthCPU;
  volumeTransformer::getStats(&bandwidthCPU);
  fprintf(stderr,"   CPU performance: <%.3f> MBytes/sec\n",bandwidthCPU);

  volumeTransformer::exitClass();
  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}


////////////////////////////////////////////////////////////////////////

// DialogViz auditor class to handle user input
class myAuditorClass : public SoDialogAuditor
{
  void dialogComboBox     (SoDialogComboBox* cpt);
  void dialogRealSlider(SoDialogRealSlider* cpt);
};

// Auditor method for combobox input
void
myAuditorClass::dialogComboBox(SoDialogComboBox* cpt)
{
  // change current test
  if (cpt->auditorID.getValue() == "configuration") {
    int value = cpt->selectedItem.getValue();
    testSwitch->whichChild = value;
  }
}

// Auditor method for slider input
void
myAuditorClass::dialogRealSlider(SoDialogRealSlider* cpt)
{
  // Move slice
  if (cpt->auditorID.getValue() == "scalefactor0")
  {
    float value = cpt->value.getValue();
    volumerTransformers[0]->scaleFactor.setValue((double)value);
  }
  else if (cpt->auditorID.getValue() == "scalefactor1")
  {
    float value = cpt->value.getValue();
    volumerTransformers[1]->scaleFactor.setValue((double)value);
  }
}


////////////////////////////////////////////////////////////////////////
// Build user interface with embedded viewer

Widget
buildInterface(Widget window)
{
  SoInput myInput;
  if (! myInput.openFile( DIALOG_FILENAME )) {
    fprintf( stderr, "ERROR opening dialogviz file '%s'\n", DIALOG_FILENAME );
    return NULL;
  }

  SoGroup *myGroup = SoDB::readAll( &myInput );
  if (! myGroup) {
    fprintf( stderr, "ERROR reading dialogviz file '%s'\n", DIALOG_FILENAME );
    return NULL;
  }

  SoTopLevelDialog *myTopLevelDialog = (SoTopLevelDialog *)myGroup->getChild( 0 );

  // Create and register auditor to handle user input
  myAuditorClass *myAuditor = new myAuditorClass;
  myTopLevelDialog->addAuditor(myAuditor);

  // Initialize scale factor numbers slider to match actual initial value
  SoDialogRealSlider *slider0 =
    (SoDialogRealSlider *)myTopLevelDialog->searchForAuditorId(SbString("scalefactor0"));
  if (slider0)
      slider0->value = (float)volumerTransformers[0]->scaleFactor.getValue();
  SoDialogRealSlider *slider1 =
    (SoDialogRealSlider *)myTopLevelDialog->searchForAuditorId(SbString("scalefactor1"));
  if (slider1)
      slider1->value = (float)volumerTransformers[1]->scaleFactor.getValue();


  // Build dialog
  SoDialogCustom *customNode = (SoDialogCustom *)myTopLevelDialog->searchForAuditorId(SbString("Viewer"));
  myTopLevelDialog->buildDialog( window, customNode != NULL );
  myTopLevelDialog->show();

  // setup available test
  SoDialogComboBox* testList = (SoDialogComboBox *)myTopLevelDialog->searchForAuditorId(SbString("configuration"));
  for (int i=0; i<testSwitch->getNumChildren();++i)
    testList->items.set1Value(i,testSwitch->getChild(i)->getName().getString());
  testList->selectedItem.setValue(testSwitch->whichChild.getValue());

  return customNode ? customNode->getWidget() : window;
}


