/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (MMM yyyy)
**=======================================================================*/

#define BUILDING_DLL 1
#include "volumeTransformer.h"


#include <Inventor/stats/SoPerfCounterManager.h>
#include <Inventor/stats/SoPerfAccumCounter.h>
#include <Inventor/SbElapsedTime.h>

#include <Inventor/actions/SoCallbackAction.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/actions/SoGetBoundingBoxAction.h>
#include <Inventor/actions/SoPickAction.h>
#include <Inventor/actions/SoWriteAction.h>

#include <Inventor/devices/SoBufferObject.h>
#include <Inventor/devices/SoGLBufferObject.h>
#include <Inventor/devices/SoCpuBufferObject.h>
#include <Inventor/algorithms/SoArithmetic.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <LDM/SoLDMTopoOctree.h>

#include <ScaleViz/SoScaleViz.h>

using std::vector;

SO_NODE_SOURCE(volumeTransformer)

SoArithmetic  *volumeTransformer::s_cpuArithmeticInterface = NULL;

SoPerfCounter *volumeTransformer::s_TimeCounterCPU = NULL;
SoPerfCounter *volumeTransformer::s_DataCounterCPU = NULL;


//-----------------------------------------------------------------------------
volumeTransformer::volumeTransformer()
{
  SO_NODE_CONSTRUCTOR( volumeTransformer );
  SO_NODE_ADD_FIELD( scaleFactor, (1.0) );
}

//-----------------------------------------------------------------------------
volumeTransformer::~volumeTransformer()
{
}


//-----------------------------------------------------------------------------
void
volumeTransformer::initClass()
{
  SO__NODE_INIT_CLASS(volumeTransformer, "volumeTransformer", SoVolumeTransform);

  SoPerfCounterManager::registerCounter( new SoPerfAccumCounter("VOLUME_TRANSFORMER_TIME_CPU", "CPU Time spent in volumeTransformer","VolumeViz",SoPerfCounter::LOW,false) );
  SoPerfCounterManager::registerCounter( new SoPerfAccumCounter("VOLUME_TRANSFORMER_DATA_CPU", "CPU data processed in volumeTransformer","VolumeViz",SoPerfCounter::LOW,false) );

  SoPerfCounterManager *m_perf = SoPerfCounterManager::getInstance();
  if (m_perf)
  {
    s_TimeCounterCPU  = m_perf->getCounter("VOLUME_TRANSFORMER_TIME_CPU");
    s_DataCounterCPU  = m_perf->getCounter("VOLUME_TRANSFORMER_DATA_CPU");
  }

  // Ok we can init the algorithms classes
  SoArithmetic::initClass();
  s_cpuArithmeticInterface = new SoArithmetic;
}

//-----------------------------------------------------------------------------
void
volumeTransformer::exitClass()
{
  SO__NODE_EXIT_CLASS(volumeTransformer);

  delete s_cpuArithmeticInterface;
  SoArithmetic::exitClass();
}

//-----------------------------------------------------------------------------
SbBool
volumeTransformer::isValid(SoState *state, const SoLDM::DataSetIdPair& p,
                           SoBufferObject *bufferObject ,
                           const SoLDMTileID &tileID)
{
  // check our own condition
  if ( scaleFactor.getValue() == 1.0 )
    return FALSE;

  // check inherited condition then
  return SoVolumeTransform::isValid(state, p, bufferObject , tileID );
}

//-----------------------------------------------------------------------------
void
volumeTransformer::apply( SoState *state, const SoLDM::DataSetIdPair& p,
                          SoBufferObject *bufferObject ,
                          const SoLDMTileID &tileID)
{
  // Do some parameters check.
  if ( bufferObject == NULL )
    return;

  if ( p.first == NULL )
    return;

  SoCpuBufferObject* cpuBufferObject=new SoCpuBufferObject;
  cpuBufferObject->ref();
  bufferObject->map(cpuBufferObject,SoBufferObject::READ_WRITE);
  volumeTransformer::apply( state, p, cpuBufferObject , tileID);
  bufferObject->unmap(cpuBufferObject);
  cpuBufferObject->unref();

}

//-----------------------------------------------------------------------------
void
volumeTransformer::apply( SoState *, const SoLDM::DataSetIdPair& p,
                          SoCpuBufferObject *bufferObject,
                          const SoLDMTileID &/*tileID*/)
{
  SoDataSet* ds = p.first;
  SoDataSet::DataType datatype = ds->getDataType();

  SbElapsedTime computeTime;

  bufferObject->getContext()->bind();

  s_cpuArithmeticInterface->scale( bufferObject, datatype, bufferObject, datatype, (float)scaleFactor.getValue() );

  bufferObject->getContext()->unbind();

  if (s_TimeCounterCPU && s_DataCounterCPU)
  {
    s_TimeCounterCPU->increment((SoPerfCounter::SoPerfCounterValue) computeTime.getElapsed());
    s_DataCounterCPU->increment((SoPerfCounter::SoPerfCounterValue) bufferObject->getSize());
  }

}

//-----------------------------------------------------------------------------
void
volumeTransformer::getStats(double *bandwidthCPU)
{
  *bandwidthCPU  = 0.0;

  // get CPU perf
  if (s_TimeCounterCPU && s_DataCounterCPU && s_TimeCounterCPU->getValue()!=0.0)
  {
    *bandwidthCPU = (s_DataCounterCPU->getValue()/s_TimeCounterCPU->getValue())/(1024.*1024.);
    s_DataCounterCPU->reset();
    s_TimeCounterCPU->reset();
  }

}
