/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Alain Dumesny (MMM yyyy)
** Modified by : David Mott (MMM yyyy)
**=======================================================================*/


#ifndef _SO_QT_GLWIDGET_
#define _SO_QT_GLWIDGET_

#include <Inventor/Qt/SoQtComponent.h>

#include <Inventor/Qt/OivQtCompat.h>
#include <Inventor/devices/SoGLFormat.h>

#include <Inventor/Gui/SoGuiGLWidget.h>
#include <Inventor/nodes/SoFullSceneAntialiasing.h>

#include <Qt>

// FSAA
typedef std::set<int, std::less<int> > SoSamplesList;

#if !defined(_WIN32)
class SoQtRemoteRenderInfo;
#endif // _WIN32

#include <QPointer>
#include <QSurfaceFormat>

/*
 * Defines used when specifying the glModes flag to the constructor.
 * (used instead of the glx.h defines which do overlap)
 */
#define SO_GLX_RGB (1<<0)
#define SO_GLX_DOUBLE (1<<1)
#define SO_GLX_ZBUFFER (1<<2)
#define SO_GLX_STEREO (1<<4)
#define SO_GLX_STENCIL (1<<5)

class QVBoxLayout;
class QCursor;
class SoGLContext;

class SoQGLContextProxy;
class SoQGLWidgetProxy;
class QGLFormat;
class QGLContext;
class QOpenGLWindow;

/**
 * @VSGEXT Component for OpenGL rendering.
 *
 * @ingroup Qt
 *
 * @DESCRIPTION
 *   This abstract base class provides a C++ wrapper around an OpenGL widget.
 *   It allows OpenGL rendering to be performed within a widget and is
 *   used by the SoQtRenderArea. SoQtGLWidget uses a parent window with two separate
 *   OpenGL widgets (one for single and one for double buffering), with
 *   routines to return the appropriate windows.
 *
 *   Subclasses only need to redefine the redraw() routine for rendering and
 *   processEvent() routine if they are interested in receiving Q events.
 *
 * @SEE_ALSO
 *    SoQtComponent,
 *    SoQtRenderArea
 *
 *
 */
class INVENTORQT_API SoQtGLWidget : public SoQtComponent
{
  Q_OBJECT

public:

   /** FloatColorBufferSize */
   enum FloatColorBufferSize
   {
    /**
     * 16-bit rendering per component.
     */
    FLOAT_16_COLOR_BUFFER = SoGuiGLWidget::FLOAT_16_COLOR_BUFFER,
    /**
     * 32-bit rendering per component.
     */
    FLOAT_32_COLOR_BUFFER = SoGuiGLWidget::FLOAT_32_COLOR_BUFFER
  };

  /** 
   * Returns the color map size.
   */
  int getColorMapSize();

  /** 
   * Sets the color map size.
   */
  void setColorMapSize( int size );

  /**
   * Gets the normal GL window (window system identifier of the widget), which is
   * needed as an argument to glXMakeCurrent() (on Xt)
   * or SbGlContextHelper::makeCurrent() (on Windows) when drawing in the normal
   * planes.
   *
   * Note: This should @B not @b be cached by users because it will change as
   * single/double buffering changes.
   */
  virtual WindowQt getNormalWindow();

  /**
   * Gets the current context, which is
   * needed as an argument to glXMakeCurrent() (on Xt)
   * or SbGlContextHelper::makeCurrent() (on Windows) when drawing in the normal
   * planes.
   *
   * Note: This should @B not @b be cached by users because it will change as OpenGL format changes.
   */
  QOpenGLContext* getQOpenGLContext();

  /**
   * Gets the current context, which is
   * needed as an argument to glXMakeCurrent() (on Xt)
   * or SbGlContextHelper::makeCurrent() (on Windows) when drawing in the normal
   * planes.
   *
   * Note: This should @B not @b be cached by users because it will change as OpenGL format changes.
   */
  SoGLContext* getNormalSoContext();

  /**
   * Gets the current normal widget. @BR
   * Since Open Inventor 8.0, this widget is a QWidget.
   * More information about this class can be found on-line in the Qt
   * documentation.
   *
   * Note: This should @B not @b be cached by users because it will change as OpenGL format changes.
   */
  QWidget* getNormalWidget();

  /** 
   * Returns the QSurfaceFormat currently used.
   */
  QSurfaceFormat getQSurfaceFormat();

  /** 
   * Sets the visual/pixel format for the normal window. @BR
   * This allows the user to create all possible visuals supported by OpenGL. The
   * OpenGL drivers or accelerated hardware may or may not support advanced features
   * such as alpha channel or stereographic viewing. If you request some features that
   * the driver/hardware does not provide when you set the @B QGLFormat @b, you will get
   * a rendering context with the nearest subset of features.
   *
   * The methods for setting the visual are virtual so that derived classes
   * can know when the visual is changing.
   */
  virtual void setNormalVisual( QSurfaceFormat vis );

  /**
   * Sets the visual/pixel format for the normal window. @BR
   * This allows the user to create all possible visuals supported by OpenGL. The
   * OpenGL drivers or accelerated hardware may or may not support advanced features
   * such as alpha channel or stereographic viewing. If you request some features that
   * the driver/hardware does not provide when you set the @B QGLFormat @b, you will get
   * a rendering context with the nearest subset of features.
   *
   * The methods for setting the visual are virtual so that derived classes
   * can know when the visual is changing.
   */
  virtual void setNormalVisual( const SoGLFormat& format );

  /**
   * This shows the component.
   */
  virtual void show();

  /**
   * This hides the component.
   */
  virtual void hide();

  /**
   * Routine that dynamically changes between single and double buffering. Default
   * is double buffer off. (The SoQtRenderArea subclass makes it double buffer by
   * default.)
   */
  virtual void setDoubleBuffer( SbBool onOrOff );

  /**
   * Returns whether double buffering is on or off.
   */
  SbBool isDoubleBuffer() { return m_guiGLWidget->isDoubleBuffer(); }

  /**
   * Included for portability only.
   */
  void  setBorder( SbBool onOrOff );

  /**
   * Included for portability only.
   */
  int getBorderSize() { return m_guiGLWidget->getBorderSize(); }

  /**
   * Included for portability only.
   */
  SbBool isBorder() const { return m_guiGLWidget->isBorder(); }

  /**
   * Sets drawing to the front buffer. Controls drawing to the front buffer when
   * an obscured portion of the window is exposed. Default: TRUE.
   */
  void setDrawToFrontBufferEnable( SbBool enableFlag );

  /**
   * Queries drawing to the front buffer.
   */
  SbBool isDrawToFrontBufferEnable() const { return m_guiGLWidget->isDrawToFrontBufferEnable(); }

  /**
   * Makes the normal rendering context the current context. 
   * Equivalent to makeNormalCurrent() call.
   */
  virtual SbBool bindNormalContext();

  /**
   * unbind the current context (previously bind with bindNormalContext );
   */
  virtual SbBool unbindNormalContext();

  /**
   * Swaps the normal front and back buffers. 
   */
  virtual SbBool swapNormalBuffers();

  /**

   * Sets the current cursor. If you are using a viewer class, the viewer will
   * automatically change the cursor depending on the viewer mode. These changes will
   * override the cursor set with this method unless you also call
   * SoQtGLWidget::setCursorEnabled(FALSE).
   */
  virtual void setCursor( const QCursor& newCursor );

  /**
   * Returns the current cursor.
   */
  QCursor getCursor() const;

  /** 
    * @copydoc SoSceneManager::setAntialiasing(const float, const SoSceneManager::AntialiasingMode)
    */
  void setAntialiasing(const float quality, const SoSceneManager::AntialiasingMode mode = SoSceneManager::AUTO);

  /** 
    * @copydoc SoSceneManager::setAntialiasing(SoAntialiasingParameters*)
    */
  void setAntialiasing(SoAntialiasingParameters* advancedParameters);

  /** 
    * @copydoc SoSceneManager::getAntialiasingQuality()
    */
  float getAntialiasingQuality() const;

  /** 
    * @copydoc SoSceneManager::getAntialiasingMode()
    */
  SoSceneManager::AntialiasingMode getAntialiasingMode() const;

  /** 
    * @copydoc SoSceneManager::getAntialiasingParameters
    */
  SoAntialiasingParameters* getAntialiasingParameters() const;

  /**
   * Enables/disables floating point rendering using 16- or 32-bit components.
   * If TRUE, Open Inventor will automatically render to a floating point color
   * buffer.
   *
   * To determine if floating point rendering was successfully enabled,
   * use #getFloatingColorBuffer.
   *
   * Using floating point rendering can improve image quality, particularly when
   * many objects are being blended.
   */
  void setFloatingColorBuffer(SbBool enable,
                              FloatColorBufferSize size = FLOAT_16_COLOR_BUFFER);

  /**
   * Returns TRUE if floating point rendering is used and its precision.
   */
  void getFloatingColorBuffer(SbBool& enable, FloatColorBufferSize& size);

  /**
   * Returns TRUE if floating point rendering is available.
   */
  static SbBool isFloatingColorBufferSupported();

  /** @copydoc SoWinGLWidget::saveSnapshot
   */
  bool saveSnapshot( const SbString& filename, bool overwrite = true );

SoINTERNAL public:
#if !defined(_WIN32) && !defined(__APPLE__) //UNIX
  // WARNING : Workaround only available for Amira/Avizo
  // These methods should be removed when the GraphicConfigs will be implemented for Qt.
  void setAttribList( const int* newAttribList );
  int* getAttribList() const;
#endif // UNIX

  SoGuiGLWidget* getGuiGLWidget() const;

  // Return the internal QWindow used to do the OpenGL rendering
  QWindow* getInternalQWindow();

  // These variables are  public for retro compatibility purposes only.
  // Do not use them directly but call their accessors instead;
  SbBool drawToFrontBuffer;

  // These two function are used by SoGui, to call the system dependent version of
  // these functions. data is a SoWinGLWidget handle.
  static SbBool swapNormalBuffersCB( void *data );
  static SbBool bindNormalCurrentCB( void *data );
  static SbBool unbindNormalCurrentCB( void *data );
  static void* getNormalWindowCB( void *data );
  static void* getCurrentDisplayCB(void *data);

  /** Return pixel ratio of this widget. Some devices, like Retina screen, may contains several pixel per point.
   * Qt coordinates, like widget size, move event position, etc... are generally returned in points.
   * Coordinates must be multiplied by this value to be converted in pixels.
   * Note that Retina screen can be simulated on windows by setting QT_DEVICE_PIXEL_RATIO=2
   * See http://doc.qt.io/qt-5/highdpi.html for details. */
  float getPixelRatio();

  /**
   * This function request an update of QWindow used to render OIV's OpenGL
   * @param immediatUpdate false => use Qt event loop to request update, true => force immediat rendering
   */
  void updateInternalOpenGLWindow( bool immediatUpdate );

 SoEXTENDER public:

  // Starting with version 3.0, subclasses should no longer make direct
  // calls to glXMakeCurrent, glXSwapBuffers and glFlush.
  // For RemoteRendering we need to know when MakeCurrent is called
  // (marking the beginning of a frame) and when SwapBuffers or glFlush
  // is called (marking the end of a frame).
  // Subclasses should always use these generic methods.
  void   flush();

#if !defined(__APPLE__) && !defined(_WIN32) //UNIX
  // Pbuffers are a limited resource.  By default use of Pbuffers for
  // remote rendering is enabled, but it might make sense to disable
  // it for less important windows (like UI components drawn with OpenGL).
  // Must be called after creating the widget but before it is realized.
  void setRemoteRenderPbufferEnable( SbBool enable )
    { m_remoteRenderPbufferEnable = enable; }

  SbBool isRemoteRenderPbufferEnable() const
    { return m_remoteRenderPbufferEnable; }

  float getLowResPercent() {return m_lowResPercent;}
#endif

protected:

  SoQtGLWidget( QWidget* parent = NULL,
                const char* name = NULL,
                SbBool buildInsideParent = TRUE,
                int glModes = SO_GLX_RGB,
                SbBool buildNow = TRUE,
                SbBool sync = TRUE );

  SoQtGLWidget( QWidget* parent,
                const char* name,
                SbBool buildInsideParent,
                int glModes,
                SbBool buildNow,
                SbBool sync,
                SoGuiGLWidget* guiGLWidget );

#if defined(__linux__)
  SoQtGLWidget( QWidget* parent,
                const char* name,
                SbBool buildInsideParent,
                int glModes,
                SbBool buildNow,
                SbBool sync,
                SbBool connectionType );

  SoQtGLWidget( QWidget* parent,
                const char* name,
                SbBool buildInsideParent,
                int glModes,
                SbBool buildNow,
                SbBool sync,
                SbBool connectionType,
                SoGuiGLWidget* guiGLWidget );
#endif

  virtual ~SoQtGLWidget();

  ////////////////////////////////////////////////////////////////////////
  // EVENTS

  virtual void resize( int w, int h );

  virtual void closeEvent( QCloseEvent* );

  virtual void focusInEvent( QFocusEvent* );

  virtual void focusOutEvent( QFocusEvent* );

  virtual void showEvent( QShowEvent* );

  virtual void hideEvent( QHideEvent* );

  virtual void moveEvent( QMoveEvent* );

  virtual void keyPressEvent( QKeyEvent* );

  virtual void keyReleaseEvent( QKeyEvent* );

  virtual void mouseMoveEvent( QMouseEvent* );
  
  virtual void mousePressEvent( QMouseEvent* );

  virtual void mouseReleaseEvent( QMouseEvent* );

  virtual void mouseDoubleClickEvent( QMouseEvent* );

  virtual void wheelEvent( QWheelEvent* );

  bool eventFilter( QObject* object, QEvent* event );

  // subclasses can use this routine as the event handler for any
  // devices they wish to get input from.
  static void eventHandler( QWidget* w, SoQtGLWidget* glw, QEvent* qe, bool* b );

  // This function is overloaded by the render area to set the window element
  virtual void onFocusEvent(SbBool hasFocus);

  ////////////////////////////////////////////////////////////////////////


  // subclasses MUST redefine redraw() to draw in the normal bit planes.
  // processEvent() should be defined if X events are being
  // received (see eventMask).
  virtual void redraw() = 0;

  virtual void processEvent( QEvent* anyevent );

  void performPaint();

  // subclasses can redefine these to do something useful.
  // initGraphic() is called whenever a GLX window gets created
  // sizeChanged() is called whenever the window changes size
  // widgetChanged() is called whenever the widget is changed (i.e. at
  //    initialization or after switching from single->double buffer)
  virtual void initGraphic();
  virtual void sizeChanged( const SbVec2s& size );
  virtual void posChanged( const SbVec2i32&, const SbVec2i32& ) {};
  virtual void widgetChanged( QWidget* w );

  // sets/gets the size of the glx widget(s) in pixels - Note this size could be
  // different from the SoWinComponent::getSize() method which return
  // the size of the component, not necessary the same as the glx widget
  // window (because of extra stuff like the decoration in the viewers or pixel ratio).
  void  setGlxSize( SbVec2s newSize );
  const SbVec2s& getGlxSize() const { return m_guiGLWidget->getGlxSize(); }

  // set/get stereo buffering visual. This routine (like setDoubleBuffer)
  // can be called interactively, althought slower since a new window
  // with the needed visual is always created on the fly.
  void setStereoBuffer( SbBool flag );
  SbBool isStereoBuffer() { return (m_guiGLWidget->getGLModes() & SO_GLX_STEREO); }

  // returns TRUE if main window is in rgb mode (FALSE if color index)
  SbBool isRGBMode() { return (m_guiGLWidget->getGLModes() & SO_GLX_RGB); }

#if defined(__linux__)
  void allocCell( Display* display, Colormap cmap, XColorPointer colorcells,
                  int ncolors, int colorMapSize );
#endif

  SbBool waitForExpose;

  QWidget* buildWidget( QWidget* parent );
  void changeCursor( QCursor newCursor );
  QWidget* getGlxMgrWidget() { return m_containerWidget; }

  virtual void onExpose();

  // ----- RemoteRender additions (v3.0) -----
  // FSAA
  SbBool m_initialFsaaValueChanged;

#if !defined(_WIN32) && !defined(__APPLE__)
  enum RemoteRenderEnabled  // Possible values for remote rendering
  {
    OFF,                      // Never use remote rendering
    ON,                       // Always try to use remote rendering
    AUTO                      // Use if remote does not support GLX (default)
  };
  RemoteRenderEnabled   m_remoteRenderMode;  // See possible values above
  SoQtRemoteRenderInfo* m_remoteRenderInfo;  // Opaque type for internal state
  SbBool                m_remoteRenderPbufferEnable; // Can we use a Pbuffer?
  SbBool                m_lowResRender;      //Is Low Resolution Render Mode in use?
  float                 m_lowResLowPercent;  //lowest value of m_lowResPercent
  float                 m_lowResHighPercent; //highest value of m_lowResPercent
  float                 m_lowResPercent;    //percent of the window sent thru the network

  void remoteRenderInit();                // Init member vars
  void remoteRenderCleanup();             // Free allocated resources
  void remoteRenderGinitCB( QWidget* );     // Create pixmap and other stuff
  GLXContext remoteRenderLowResRenderGinitCB( QWidget* ); //Same for LowResRender configuration : return Remote Context
  void remoteRenderResizeCB();            // Resize pixmap
  void remoteRenderSendImage();           // Send rendered image to remote display
  void remoteRenderSendLowResRender();    // send low Resolution image to remote display
  SbGlContextHelper::VisualInfo remoteRenderChooseVisual();        // Find visual for remote window
  void remoteRenderFatalError( const char* methodName, const char* message );

  void setLowResLowPercent( float percent ) { m_lowResLowPercent = percent; }
  void setLowResHighPercent( float percent ) { m_lowResHighPercent = percent; }
  float getLowResLowPercent() const { return m_lowResLowPercent; }
  float getLowResHighPercent() const { return m_lowResHighPercent;}

  virtual void setLowResPercent( float newLowres ) { m_lowResPercent = newLowres; }
#endif

private:

  void constructorCommon( SbBool buildNow, SbBool connectionType );
  SoGuiGLWidget* m_guiGLWidget; //Implementation class for SoXxGLWidget

  QPointer<QWidget> m_containerWidget;
  QPointer<QVBoxLayout> m_containerLayout;
  QPointer<QWidget> m_internalWidget;
  QPointer<QWindow> m_internalWindow;

  SoQGLContextProxy* m_context;

  // Flags used when the stereo is set from environment variables.
  bool m_stereoActiveDelayed; 
  bool m_stereoBufferFlag;

#if defined(__linux__)
  SbBool directConnection;
#endif

  QCursor currentCursor;
  int* attribList;

  SbBool windowResized;

  // set of windows, color maps, etc...
  void buildNormalGLXWidget( SoGLFormat* format = NULL );

  virtual void initializeGL();

  void destroyNormalWindows();
  void destroyGLXWidget( SbBool normalCall );

  void destroyNormalWidget( SbBool normalCall );

  void notifyProxyDestruction( QWindow* window );

  friend class SoQGLWidgetProxy;

  // callbacks from glx widget
  static void ginitCB( QWidget* w, SoQtGLWidget* glw, void* v );
  static void exposeCB( QWidget* w, SoQtGLWidget* glw, void* v );
  static void resizeCB( QWidget* w, SoQtGLWidget* glw, void* v );

  static SbBool onGLFormatChangeCallback( SoGLFormat& format, void* userData );

  SbBool remoteRenderGinitCB_pbuffer( QWidget* );

  // Members used to synchronize first rendering (in case of OIV_SYNC_QTWIDGET_FIRSTRENDER == TRUE)
  bool m_firstRenderSync;
  static std::map<SoQtGLWidget*, bool> s_blackRendering;
  static std::map<SoQtGLWidget*, bool> s_widgetRendered;

private Q_SLOTS:
  void destroyedQtContext();
};

#endif // _SO_QT_GLWIDGET_

