/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Alain Dumesny (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2018 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SB_COLOR_H_
#define  _SB_COLOR_H_

#include <Inventor/SbLinear.h>


//////////////////////////////////////////////////////////////////////////////
//
//  Class: SbColor
//
//  3D vector used to represent an RGB color. Each component of the vector is 
// a floating-point number between 0.0 and 1.0 .
//
//////////////////////////////////////////////////////////////////////////////

class SbColor;

/**
 * Color vector class.
 * 
 * @ingroup Basics
 * 
 * @DESCRIPTION
 *   This class is used to represent an RGB color. Each component of the vector is a
 *   floating-point number between 0.0 and 1.0. There are routines to convert back
 *   and forth between RGB and HSV.
 * 
 * @SEE_ALSO
 *    SbColorRGBA, SbVec3f
 * 
 * [OIVJAVA-WRAPPER-CLASS BASIC_TYPE{true},LAYOUTED_ARRAY_TYPE{float,3}]
 */
class INVENTORBASE_API SbColor : public SbVec3f {
 public:
  /**
   * Default constructor.  The color value is not initialized.
   */
  SbColor();

  /**
   * Constructor that takes a vector value.
   */
  SbColor(const SbVec3f vec3f) 
    { setValue(vec3f.getValue()); }

  /**
   * Constructor that takes an array of 3 floats.
   */
  SbColor(const float rgb[3])
    { setValue(rgb); }

  /**
   * Constructor that takes 3 floats.
   */
  SbColor(float r, float g, float b)
    { setValue(r, g, b); }

  //
  // HSV routines. Those are 3 floats containing the Hue, Saturation and
  // Value (same as brightness) of the color.
  //

  /**
   * Sets value of color vector from 3 HSV (Hue, Saturation, and Value) components.
   * Value is the same as brightness of the color.
   */
  SbColor &   setHSVValue(float h, float s, float v);

  /**
   * Sets value of color vector from array of 3 HSV components
   */
  SbColor &   setHSVValue(const float hsv[3])
    { return setHSVValue(hsv[0], hsv[1], hsv[2]); }

  /**
   * Returns 3 individual HSV components
   */
  void        getHSVValue(float &h, float &s, float &v) const;

  /**
   * Returns an array of 3 HSV components
   */
  void        getHSVValue(float hsv[3]) const
    { getHSVValue(hsv[0], hsv[1], hsv[2]); }
    
  //
  // RGBA Packed integer color routines. The color format expressed in 
  // hexadecimal is 0xrrggbbaa, where
  //      aa  is the alpha value
  //      bb  is the blue value
  //      gg  is the green value
  //      rr  is the red value
  // RGBA component values range from 0 to 0xFF (255).
  //


  /**
  * Set RGB values as unsigned char. RGB values range from 0 to 0xFF (255)
  */
  void setRGBValue(unsigned char r, unsigned char g, unsigned char b)
  { setValue(r / 255.f, g / 255.f, b / 255.f); }
    
  /**
   * Sets value of color vector from an RGBA packed color value. The packed color
   * format expressed in hexadecimal is 0xrrggbbaa, where
   * 
   * - rr is the red value
   * 
   * - gg is the green value
   * 
   * - bb is the blue value
   * 
   * - aa is the alpha value
   * 
   * RGBA component values range from 0 to 0xFF (255). The returned transparency value
   * is a floating point value between 0.0 (opaque) and 1.0 (completely transparent).
   * It is derived from the alpha component of the RGBA color.
   */
  SbColor &setPackedValue(uint32_t orderedRGBA, float& transparency);

  /**
   * Returns an RGBA packed color value, derived from the color vector and the passed
   * transparency value. The alpha component is set to (1.0 - transparency) * 255,
   * resulting in a hex value between 0 and 0xFF. If transparency not specified,
   * alpha is set to 0xFF (opaque). The color format expressed in hexadecimal is 0xrrggbbaa.
   */
  uint32_t getPackedValue(float transparency = 0.0) const;

  /**
  * Returns an RGBA packed color value, derived from the color vector and the passed
  * transparency value. The alpha component is set to (1.0 - transparency) * 255,
  * resulting in a hex value between 0 and 0xFF. If transparency not specified,
  * alpha is set to 0xFF (opaque).
  *
  * This method takes "endianness" in account and will return a value according to
  * the processor architecture. For example, on "little-endian" processors like Intel,
  * the color value returned is actually 0xaabbggrr.  See the note in SoVolumeData
  * about constructing RGBA voxel values programmatically.
  */
  uint32_t getPackedValueEndiannessOrder(float transparency = 0.0) const;

#if SoDEPRECATED_BEGIN(9500)

  SoDEPRECATED_METHOD(9500, "Use getPackedValueendiannessOrder(float transparency = 0.0) const instead.")
  uint32_t getPackedValueInternal(float transparency = 0.0) const {
    return getPackedValueEndiannessOrder(transparency);
  };
#endif /** @DEPRECATED_END */

};

#endif /* _SB_COLOR_H_ */


