/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2018 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SB_COLORRGBA_H_
#define  _SB_COLORRGBA_H_

#include <Inventor/SbLinear.h>


//////////////////////////////////////////////////////////////////////////////
//
//  Class: SbColorRGBA
//
//  4D vector used to represent an RGBA color. Each component of the vector is 
//  a floating-point number between 0.0 and 1.0 .
//
//////////////////////////////////////////////////////////////////////////////

class SbColorRGBA;

/**
 * ColorRGBA vector class.
 * 
 * @ingroup Basics
 * 
 * @DESCRIPTION
 *   This class is used to represent an RGBA color. Each component of the vector is a
 *   floating-point number between 0.0 and 1.0. RGBA stands for red, green, blue and
 *   alpha.  Alpha is the inverse of "transparency", in other words 0 is fully
 *   transparent and 1 is fully opaque.
 *
 *   Values may be specified in the HSV (hue, saturation and value) color space, but
 *   are stored internally as RGBA values.
 *
 * @SEE_ALSO
 *    SbColor, SbVec4f
 * 
 * [OIVJAVA-WRAPPER-CLASS BASIC_TYPE{true},LAYOUTED_ARRAY_TYPE{float,4}]
 */
class INVENTORBASE_API SbColorRGBA : public SbVec4f {
 public:
  /**
   * Default constructor.  The color value is not initialized.
   */
  SbColorRGBA();

  /**
   * Constructor that takes an RGBA vector value.
   */
  SbColorRGBA(const SbVec4f vec4f) 
    { setValue(vec4f.getValue()); }

  /**
   * Constructor that takes an RGBA value as an array of 4 floats.
   */
  SbColorRGBA(const float rgba[4])
    { setValue(rgba); }

  /**
   * Constructor that takes an RGBA value as 4 floats.
   */
  SbColorRGBA(float r, float g, float b, float a)
    { setValue(r, g, b, a); }

  /**
  * Constructor that takes an RGBA value as a packed color.
  */
  SbColorRGBA (uint32_t orderedRGBA);

  /**
   * Sets value of color vector from an RGBA packed color value. The packed color
   * format expressed in hexadecimal is 0xrrggbbaa, where
   * 
   * - rr is the red value
   * 
   * - gg is the green value
   * 
   * - bb is the blue value
   * 
   * - aa is the alpha value
   * 
   * RGBA component values range from 0 to 0xFF (255).
   *
   */
  SbColorRGBA &setPackedValue(uint32_t orderedRGBA);

  /**
   * Returns an RGBA packed color value, derived from the color vector.
   * The packed color format expressed in hexadecimal is 0xrrggbbaa.
   */
  uint32_t   getPackedValue() const;

  /**
   * Sets value of color vector from an HSV (Hue, Saturation, and Value)
   * plus Alpha color value.  All components must be in the range 0 to 1.
   * 
   * Value is the same as brightness of the color.
   */
  SbColorRGBA &   setHSVAValue(float h, float s, float v, float a);

  /**
   * Sets value of color vector from an HSV (Hue, Saturation, and Value)
   * plus Alpha color value.  All components must be in the range 0 to 1.
   */
  SbColorRGBA &   setHSVAValue(const float hsva[4])
    { return setHSVAValue(hsva[0], hsva[1], hsva[2], hsva[3]); }

  /**
   * Returns color value in the HSV (hue, saturation and value) plus
   * Alpha color space.
   */
  void getHSVAValue(float &h, float &s, float &v, float& a) const;

  /**
   * Returns color value in the HSV (hue, saturation and value) plus
   * Alpha color space.
   */
  void getHSVAValue(float hsva[4]) const
    { getHSVAValue(hsva[0], hsva[1], hsva[2], hsva[3]); }

SoINTERNAL public:

  /** initialize class. */
  static void initClass();

  /** Finalize class. */
  static void exitClass();

  /**
  * Sets value of color vector from an RGBA packed color value. The packed color
  * format expressed in hexadecimal is 0xaabbggrr, where
  *
  * - rr is the red value
  *
  * - gg is the green value
  *
  * - bb is the blue value
  *
  * - aa is the alpha value
  *
  * RGBA component values range from 0 to 0xFF (255).
  *
  */
  SbColorRGBA &setLEPackedValue(uint32_t littleEndianRGBA);

};

#endif /* _SB_COLORRGBA_H_ */


