/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Thibaut Andrieu (Mar 2015)
**=======================================================================*/

#ifndef  _SB_DELEGATE_H_
#define  _SB_DELEGATE_H_

#include <Inventor/SbDelegatePtrFunImpl.h>
#include <Inventor/SbDelegateMemberFunImpl.h>
/*! \cond PRIVATE */
namespace inventor
{
namespace helper
{

/**
 * This class represents a "function pointer". It can encapsulate a pointer to
 * a static function or a member function.
 * WARNING ReturnType must have a default constructor.
 */
template<typename ReturnType, typename ArgType>
SoINTERNAL class SbDelegate
{
public:

  /** Default contructor */
  SbDelegate()
  {
  }

  /** Copy constructor */
  SbDelegate( const SbDelegate<ReturnType, ArgType>& other )
  : m_delegateImpl ( other.m_delegateImpl->clone() )
  {
  }

  /**
   * Create an SbDelegate from a static function.
   * ex:
   * void myFun(int i) { }
   * SbDelegate<void, int> delegate(myFun)
   */
  SbDelegate(ReturnType (*f)(ArgType))
  {
    if ( f != NULL )
      m_delegateImpl = new inventor::impl::SbDelegatePtrFunImpl<ReturnType, ArgType>(f);
    else
      m_delegateImpl = NULL;
  }

  /**
   * Create an SbDelegate from a member function.
   * ex:
   * MyClass {
   *   void myFun(int i) { }
   * }
   * MyClass myClass;
   *
   * SbDelegate<void, int> delegate(myClass, &MyClass::myFun)
   */
  template<typename ClassType>
  SbDelegate(ClassType& a, ReturnType (ClassType::*f)(ArgType))
  {
    if ( f != NULL )
      m_delegateImpl = new inventor::impl::SbDelegateMemberFunImpl<ReturnType, ArgType, ClassType>(a, f);
    else
      m_delegateImpl = NULL;
  }

  /**
   * Call the function associated to this delegate. If delegate is empty (no function),
   * Return ReturnType default constructor.
   */
  ReturnType operator()(ArgType arg) const
  {
    if ( m_delegateImpl.ptr() != NULL )
      return m_delegateImpl->call(arg);
    else
      return ReturnType();
  }

  /**
   * Return true if function encapsulated by this delegate is the given function.
   * An empty delegate is equal to NULL function
   */
  bool equal(ReturnType (*f)(ArgType)) const
  {
    if ( m_delegateImpl.ptr() == NULL )
      return (f == NULL);

    inventor::impl::SbDelegatePtrFunImpl<ReturnType, ArgType>* delegatePtrFun = dynamic_cast< inventor::impl::SbDelegatePtrFunImpl<ReturnType, ArgType>* >(m_delegateImpl.ptr());
    if ( delegatePtrFun != NULL )
      return delegatePtrFun->equal(f);

    return false;
  }

  /**
   * Return true if function encapsulated by this delegate is the given function.
   * An empty delegate is equal to NULL function
   */
  template<typename ClassType>
  bool equal(ClassType& a, ReturnType (ClassType::*f)(ArgType)) const
  {
    if ( m_delegateImpl.ptr() == NULL )
      return (f == NULL);

    inventor::impl::SbDelegateMemberFunImpl<ReturnType, ArgType, ClassType>* delegateMemberFun = NULL;
    delegateMemberFun = dynamic_cast< inventor::impl::SbDelegateMemberFunImpl<ReturnType, ArgType, ClassType>* >(m_delegateImpl.ptr());
    if ( delegateMemberFun != NULL )
      return delegateMemberFun->equal(a, f);
    else
      return false;
  }

private:

  /** DelegateImpl associated to this delegate */
  SoRef< inventor::impl::SbDelegateImpl<ReturnType, ArgType> > m_delegateImpl;
};

} // helper
} // inventor
/*! \endcond */
#endif // _SB_DELEGATE_H_

