/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef _SB_LINE_
#define _SB_LINE_

#include <Inventor/SbBase.h>
#include <Inventor/SbVec.h>
#include <Inventor/SbBox.h>


//////////////////////////////////////////////////////////////////////////////
//
//  Class: SbLine
//
//  Represents a directed line in 3D space.
//
//////////////////////////////////////////////////////////////////////////////

class SbLined;

/**
* Directed line in 3D.
*
* @ingroup Basics
*
* @DESCRIPTION
*   Represents a directed line in 3D. This is a basic Open Inventor datatype that is
*   used for representing a 3D line. It is used as input and output by a variety of
*   Open Inventor classes.
*
* @SEE_ALSO
*    SbVec3f,
*    SbPlane
*
* [OIVJAVA-WRAPPER-CLASS BASIC_TYPE{true},LAYOUTED_ARRAY_TYPE{float,6}]
*/
class INVENTORBASE_API SbLine {
public:
  /**
  * Default constructor.  The line is not initialized.
  */
  SbLine()  {}

  /**
  * Constructor. The line is directed from @B p0 @b to @B p1@b. @BR
  * To construct a line from a position and direction you can use: SbLine(p0,
  * p0 + dir), but using the default constructor followed by the setPosDir()
  * method is recommended to avoid floating point precision problems.
  */
  SbLine(const SbVec3f &p0, const SbVec3f &p1);

  /**
  * Sets line to pass through points @B p0 @b and @B p1@b.
  */
  void    setValue(const SbVec3f &p0, const SbVec3f &p1);

  /**
  * Sets line using a position and a direction vector.
  * The direction vector will be normalized automatically.
  * This method is preferred over setValue when the point has very
  * large magnitude, for example 1e6, because adding a normalized
  * direction vector to such a point may not be significant.
  */
  void    setPosDir( const SbVec3f &position, const SbVec3f &direction );

  /**
  * Sets line from a double precision line.
  */
  void    setValue( const SbLined& line );

  /**
  * Finds the two closest points between this line and @B line2@b, and loads them
  * into @B ptOnThis @b and @B ptOnLine2@b. Returns FALSE if the lines are
  * parallel (results undefined), and returns TRUE otherwise.
  */
  SbBool    getClosestPoints(const SbLine  &line2,
    SbVec3f &ptOnThis,
    SbVec3f &ptOnLine2 ) const;

  /**
  * Returns the closest point on the line to the given point.
  */
  SbVec3f    getClosestPoint(const SbVec3f &point) const;

  /**
  * Returns position of line origin point.
  */
  inline const SbVec3f &getPosition() const  { return pos; }

  /**
  * Returns direction vector of line.
  */
  inline const SbVec3f &getDirection() const  { return dir; }

  /**
  * Setup epsilon to detect intersection of very small geometry.
  */
  static void setIntersectEpsilon(const float epsilon);

  SoINTERNAL public:

  /**
   * Intersect the line with a box.
   */
  SbBool intersect(const SbBox3f &box, SbVec3f &enter, SbVec3f &exit) const;

  /**
   * Intersect the line with a box.
   */
  SbBool intersect(const SbXfBox3f &xbox, SbVec3f &enter, SbVec3f &exit) const;

  /**
   * Intersect the line with a box.
   */
  SbBool intersect(float angle, const SbBox3f &box) const;

  /**
   * Intersect the line with a point.
   */
  SbBool intersect(float angle, const SbVec3f &point) const;

  /**
   * Intersect the line with a triangle.
   */
  SbBool intersect(float angle, const SbVec3f &v0, const SbVec3f &v1, SbVec3f &pt) const;

  /**
   * Intersect the line with a triangle.
   */
  SbBool intersect(
    const SbVec3f &v0, const SbVec3f &v1, const SbVec3f &v2, 
    SbVec3f &pt, SbVec3f &barycentric, SbBool &front
    ) const;

private:
  // Parametric description:
  //  l(t) = pos + t * dir
  SbVec3f  pos;
  SbVec3f  dir;
};

//======================================================================

/**
* Directed line in 3D (double precision).
*
* @ingroup Basics
*
* @DESCRIPTION
*   Represents a (double precision) directed line in 3D. 
*   This is a basic Open Inventor datatype that is
*   used for representing a 3D line. It is used as input and output by a variety of
*   Open Inventor classes.
*
* @SEE_ALSO
*    SbVec3d,
*    SbLine,
*    SbPlane
*
* [OIVJAVA-WRAPPER-CLASS BASIC_TYPE{true},LAYOUTED_ARRAY_TYPE{double,6}]
*/
class INVENTORBASE_API SbLined {
public:
  /**
  * Default constructor.  The line is not initialized.
  */
  SbLined()  {}

  /**
  * Constructor. The line is directed from @B p0 @b to @B p1@b. @BR
  * To construct a line from a position and direction you can use: SbLine(p0,
  * p0 + dir), but using the default constructor followed by the setPosDir()
  * method is recommended to avoid floating point precision problems.
  */
  SbLined(const SbVec3d &p0, const SbVec3d &p1);

  /**
  * Sets line to pass through points @B p0 @b and @B p1@b.
  */
  void    setValue(const SbVec3d &p0, const SbVec3d &p1);

  /**
  * Sets line using a position and a direction vector.
  * The direction vector will be normalized automatically.
  * This method is preferred over setValue when the point has very
  * large magnitude, for example 1e6, because adding a normalized
  * direction vector to such a point may not be significant.
  */
  void    setPosDir( const SbVec3d &position, const SbVec3d &direction );

  /**
  * Set value from single precision line
  */
  void    setValue( const SbLine& line );

  /**
  * Finds the two closest points between this line and @B line2@b, and loads them
  * into @B ptOnThis @b and @B ptOnLine2@b. Returns FALSE if the lines are
  * parallel (results undefined), and returns TRUE otherwise.
  */
  SbBool    getClosestPoints(const SbLined  &line2,
    SbVec3d &ptOnThis,
    SbVec3d &ptOnLine2 ) const;

  /**
  * Returns the closest point on the line to the given point.
  */
  SbVec3d    getClosestPoint(const SbVec3d &point) const;


  /**
  * Returns position of line origin point.
  */
  inline const SbVec3d &getPosition() const  { return pos; }

  /**
  * Returns direction vector of line.
  */
  inline const SbVec3d &getDirection() const  { return dir; }

  /**
  * Setup epsilon to detect intersection of very small geometry.
  */
  static void setIntersectEpsilon(const double epsilon);

  SoINTERNAL public:
  // Intersect the line with a box, point, line, and triangle.
  SbBool intersect(const SbBox3d &box,
    SbVec3d &enter, SbVec3d &exit) const;
  SbBool intersect(double angle, const SbBox3d &box) const;
  SbBool intersect(double angle, const SbVec3d &point) const;
  SbBool intersect(double angle, const SbVec3d &v0,
    const SbVec3d &v1, SbVec3d &pt) const;
  SbBool intersect(const SbVec3d &v0,
    const SbVec3d &v1,
    const SbVec3d &v2,
    SbVec3d &pt, SbVec3d &barycentric,
    SbBool &front) const;
    
  SbVec3d getClosestPoint_( const SbVec3d &point, 
    double &projlen ) const;

  bool getClosestPoints_( const SbLined &line2,
    SbVec3d &ptOnThis,
    SbVec3d &ptOnLine2,
    double &s,
    double &t) const;

private:
  // Parametric description:
  //  l(t) = pos + t * dir
  SbVec3d  pos;
  SbVec3d  dir;
};

// Inline methods

inline void SbLine::setValue( const SbLined& line )
{
  pos.setValue( line.getPosition() );
  dir.setValue( line.getDirection() );
}

inline void SbLined::setValue( const SbLine& line )
{
  pos.setValue( line.getPosition() );
  dir.setValue( line.getDirection() );
}

#endif /* _SB_LINE_ */


