/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef _SB_MATRIX_
#define _SB_MATRIX_

#include <Inventor/SbBase.h>
#include <Inventor/STL/iostream>
#include <Inventor/SbVec.h>
#include <string.h>

class SbLine;
class SbLined;
class SbRotation;
class SbRotationd;
class SbMatrixd;

typedef float SbMat[4][4];
typedef float SbMat3[3][3];
typedef double SbMatd[4][4];

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SbMatrix
//
//  4x4 matrix of floating-point elements.
//
//////////////////////////////////////////////////////////////////////////////



/**
* 4x4 matrix class.
*
* @ingroup Basics
*
* @DESCRIPTION
*   4x4 matrix class/datatype used by many Open Inventor node and action classes.
*
* @B Matrices@b
*
* The Open Inventor API uses the convention that positions and directions in 3D
* space are represented by @I row vectors@i.  Therefore, to apply a transform
* matrix, the vector is post-multiplied by the matrix as shown in the following
* figure.  Many mathematics and computer graphics books use column vector notation,
* however there is no functional difference between these two approaches.
*
* Note that the commonly used terms "row major" and "column major" refer to the
* storage order of the matrix components in memory. This has nothing to do with
* how you use matrices and vectors with the Open Inventor API.  Internally Open
* Inventor uses the same storage order as OpenGL to allow matrices to be passed
* efficiently to/from the GPU. When using the Open Inventor API just remember
* that positions are row vectors, as shown here.
* \verbatim

 [X' Y' Z' 1] = [X Y Z 1] * | m11 m12 m13 m14 |
                            | m21 m22 m23 m24 |
                            | m31 m32 m33 m34 |
                            | m41 m42 m43 m44 |

\endverbatim
*
* Some common 4x4 transform matrices look like this:
* \verbatim

 Identity  | 1 0 0 0 |  Translate  | 1  0  0  0 |  Scale  | Sx  0  0  0 |  RotateX  | 1    0     0 0 |
           | 0 1 0 0 |             | 0  1  0  0 |         |  0 Sy  0  0 |           | 0 cosT -sinT 0 |
           | 0 0 1 0 |             | 0  0  1  0 |         |  0  0 Sz  0 |           | 0 sinT  cosT 0 |
           | 0 0 0 1 |             | Tx Ty Tz 1 |         |  0  0  0  1 |           | 0    0     0 1 |

\endverbatim
*
* Therefore, to create a translation matrix you could initialize the SbMatrix object
* like this (or you could simply use the setTranslate() convenience method):
* \verbatim
 SbMatrix( 1,0,0,0, 0,1,0,0, 0,0,1,0, Tx,Ty,Tz,1 ) \endverbatim
*
*   For convenience SbMatrix allows its values to be accessed using 2D array
*   syntax, like this:
* \if_cpp
*   \verbatim
  value = matrix[row][column]; \endverbatim
* \endif
* \if_dotnet
*   \verbatim
  value = matrix[row,column]; \endverbatim
* \endif
* \if_java
*   \verbatim
  value = matrix.getElement( row, column ); \endverbatim
* \endif
*
* For example, the translation X, Y, Z values in the above example can be retrieved using: \verbatim
  Tx = matrix[3][0]  // Row 3, Column 0
  Ty = matrix[3][1]  // Row 3, Column 1
  Tz = matrix[3][2]  // Row 3, Column 2 \endverbatim
*
* @B Multiplying points@b
*
* Points (positions in 3D space) are transformed by post-multiplying
* the row vector with the transform matrix like this:
*   \verbatim
  P' = P * M \endverbatim
*   If you need to transform a point by a matrix use the multVecMatrix() method as
*   shown here:
* \if_cpp
*   \code
*   SbMatrix M;
*   SbVec3f  src, dst;
*   M.multVecMatrix( src, dst );
*   \endcode
* \endif
* \if_dotnet
*   \code
*   SbMatrix M;
*   SbVec3f  src, dst;
*   M.multVecMatrix( src, dst );
*   \endcode
* \endif
* \if_java
*   \code
*   SbMatrix M;
*   SbVec3f  src;
*   SbVec3f dst = M.multVecMatrix( src );
*   \endcode
* \endif
*   Note that it is safe to use the same SbVec3f object as both src and dst.
*
*   In SbViewVolume, for example, the projectToScreen() method first calls the
*   getMatrix() method to get the combined model/view/projection matrix, then
*   calls that object's multVecMatrix() method to transform the 3D point into
*   normalized clipping space (-1 to 1).  (It then does one more step to convert
*   that position to 0..1 normalized screen space but that's not important here.)
*
* @B Multiplying directions@b
*
*   Vectors that represent a direction in 3D space rather than a position, for
*   example surface normal vectors for geometry, can also be transformed.  But in
*   this case the translation portion of the matrix (if any) must not be used.
*   For example, if a matrix contains the translation [10, 20, 30], then transforming
*   the normal vector [0, 0, 1] using multVecMatrix() would produce the result [10, 20, 31].
*   However the correct result is still [0, 0, 1] because translation has no meaning for
*   a direction vector.  The method multDirMatrix() is provided to
*   transform direction vectors ignoring the translation portion of the matrix.
*
*   Generally normals should be transformed by the inverse transpose of the matrix.
*   See standard computer graphic references for the explanation.
*   \if_cpp
*   \code
*   SbMatrix M;
*   SbVec3f  src, dst;
*   M.transpose().inverse().multDirMatrix( src, dst );
*   \endcode
*   \endif
*   \if_dotnet
*   \code
*   SbMatrix M;
*   SbVec3f  src, dst;
*   M.Transpose().Inverse().MultDirMatrix( src, out dst );
*   \endcode
*   \endif
*   \if_java
*   \code
*   SbMatrix M;
*   SbVec3f  src;
*   SbVec3f dst = M.transpose().inverse().multDirMatrix( src );
*   \endcode
*   \endif
*   However note that if the matrix is orthonormal, i.e. purely rotational with
*   no scaling or shearing, then the inverse transpose is the same as the original
*   matrix and it is not necessary to compute the inverse transpose.
*
* @B Multiplying matrices@b
*
* A series of transforms, for example scale, rotate and translate can be
* combined into a single transform matrix by multiplying the matrices together.
* The result of such a multiplication is @I order dependent@i. Using the row vector
* convention, we can say that transforms are applied from "left to right".
* We normally want scaling applied first, then rotation, then translation,
* as shown here:
* \verbatim
  P' = P * S * R * T \endverbatim
*
* So we would build the combined transform matrix M from scale, rotate and
* translate matrices S, R and T like this:
* \verbatim
  M = S * R * T \endverbatim
*
* Note that convenience nodes like SoTransform do this (combine the scale, rotate
* and translate) for you automatically.
* So you don't necessarily need to remember all the details.
*
* If you need to combine matrices yourself, you can use the multLeft() or multRight()
* method to multiple each matrix with the combined matrix.
* The name "multLeft" means to pre-multiply the SbMatrix object with the specified
* SbMatrix parameter, so we would combine the matrices like this:
* \if_cpp
*   \code
*   SbMatrix M, S, R, T;
*   M = T;
*   M.multLeft( R );
*   M.multLeft( S );
*   \endcode
* \endif
* \if_dotnet
*   \code
*   SbMatrix M, S, R, T;
*   M = T;
*   M.MultLeft( R );
*   M.MultLeft( S );
*   \endcode
* \endif
* \if_java
*   \code
*   SbMatrix M, S, R, T;
*   M = T;
*   M.multLeft( R );
*   M.multLeft( S );
*   \endcode
* \endif
* Note that multLeft() @a overwrites the matrix currently in the SbMatrix object.
* So usually (as shown) you will start by making a copy
* of the first matrix as the starting point for accumulation.
*
* The name "multRight" means to post-multiply the SbMatrix object with the specified
* SbMatrix parameter.  So we would combine the matrices like this:
* \if_cpp
*   \code
*   SbMatrix M, S, R, T;
*   M = S;
*   M.multRight( R );
*   M.multRight( T );
*   \endcode
* \endif
* \if_dotnet
*   \code
*   SbMatrix M, S, R, T;
*   M = S;
*   M.MultRight( R );
*   M.MultRight( T );
*   \endcode
* \endif
* \if_java
*   \code
*   SbMatrix M, S, R, T;
*   M = S;
*   M.multRight( R );
*   M.multRight( T );
*   \endcode
* \endif
* Note that multRight() also overwrites the matrix currently in the SbMatrix object.
* So usually (as shown) you will start by making a copy
* of the first matrix as the starting point for accumulation.
*
* \if_cpp In C++, you can also use the "*=" operator to do a multRight operation that replaces
* the target matrix and you can also use the "*" operator to multiply two matrices without modifying
* either matrix. For example, to concatenate scale, rotate and translate matrices as
* above you can simply write:
* \code
* SbMatrix M, S, R, T;
* M = S * R * T;
* \endcode
* \endif
*
*
* @SEE_ALSO
*    SbMatrix3,
*    SbMatrixd,
*    SbRotation,
*    SbRotationd,
*    SbVec2d,
*    SbVec2f,
*    SbVec2i32,
*    SbVec2s,
*    SbVec3d,
*    SbVec3f,
*    SbVec3i32,
*    SbVec3s,
*    SbVec4b,
*    SbVec4d,
*    SbVec4f,
*    SbVec4i32,
*    SbVec4s,
*    SbVec4ub,
*    SbVec4ui32,
*    SbVec4us
*
* [OIVJAVA-WRAPPER-CLASS BASIC_TYPE{true},LAYOUTED_ARRAY_TYPE{float,16}]
*/
class INVENTORBASE_API SbMatrix {
public:

  /**
  * Default constructor.  The matrix is initialized with zeros.
  */
  SbMatrix()    { memset(matrix, 0, sizeof(SbMat)); }

  /**
  * Constructor.
  */
  SbMatrix(float a11, float a12, float a13, float a14,
    float a21, float a22, float a23, float a24,
    float a31, float a32, float a33, float a34,
    float a41, float a42, float a43, float a44);

  /**
  * Constructor.
  */
  SbMatrix(const SbMat &m);

  /**
  * Sets matrix from a 4x4 array of elements.
  */
  void  setValue(const SbMat &m);

  /**
   * Sets matrix from a 16 value float array.
   * [OIVJAVA-WRAPPER-NO-WRAP]
   */
  void    setValue(const float *pMat)
  {
    matrix[0][0] = *pMat++ ;
    matrix[0][1] = *pMat++ ;
    matrix[0][2] = *pMat++ ;
    matrix[0][3] = *pMat++ ;

    matrix[1][0] = *pMat++ ;
    matrix[1][1] = *pMat++ ;
    matrix[1][2] = *pMat++ ;
    matrix[1][3] = *pMat++ ;

    matrix[2][0] = *pMat++ ;
    matrix[2][1] = *pMat++ ;
    matrix[2][2] = *pMat++ ;
    matrix[2][3] = *pMat++ ;

    matrix[3][0] = *pMat++ ;
    matrix[3][1] = *pMat++ ;
    matrix[3][2] = *pMat++ ;
    matrix[3][3] = *pMat++ ;
  } ;

  /**
  * Sets value from a double precision matrix.
  */
  void  setValue(const SbMatrixd &md) ;

public:

  /**
  * Sets matrix to be identity.
  */
  void  makeIdentity();

  /**
  * Returns an identity matrix.
  */
  static SbMatrix  identity();

  /**
  * Sets matrix to rotate by given rotation.
  */
  void  setRotate(const SbRotation &q);

  /**
  * Sets matrix to scale by given uniform factor.
  */
  void  setScale(float s);

  /**
  * Sets matrix to scale by given vector.
  */
  void  setScale(const SbVec3f &s);

  /**
  * Sets matrix to translate by given vector.
  */
  void  setTranslate(const SbVec3f &t);

  /**
  * Composes the matrix based on a translation, rotation, scale, orientation for
  * scale, and center. The @B scaleOrientation @b chooses the primary axes for the scale.
  * The @B center @b is the center point for scaling and rotation.
  */
  void  setTransform(const SbVec3f &translation,
    const SbRotation &rotation,
    const SbVec3f &scaleFactor,
    const SbRotation &scaleOrientation,
    const SbVec3f &center);

  /**
  * Composes the matrix based on a translation, rotation, and scale.  A scale orientation
  * value of (0,0,0,1) is used. The center point for scaling and rotation is (0,0,0).
  */
  void setTransform(const SbVec3f &t, const SbRotation &r, const SbVec3f &s);

  /**
  * Composes the matrix based on a translation, rotation, scale, and orientation for
  * scale. The @B scaleOrientation @b chooses the primary axes for the scale. The
  * center point for scaling and rotation is (0,0,0).
  */
  void setTransform(const SbVec3f &t, const SbRotation &r,
    const SbVec3f &s, const SbRotation &so);

  /**
  * Decomposes the matrix into a translation, rotation, scale, and scale
  * orientation. Any projection information is discarded. The decomposition depends
  * upon choice of center point for rotation and scaling, which is optional as the
  * last parameter. Note that if the center is 0, decompose() is the same as
  * factor() where t is @B translation, @b u is @B rotation, @b s is
  * @B scaleFactor, @b and r is @B scaleOrientation. @b
  * [OIVJAVA-WRAPPER NAME{decompose},PACK{Decomposition}]
  */
  void  getTransform(SbVec3f &translation,
    SbRotation &rotation,
    SbVec3f &scaleFactor,
    SbRotation &scaleOrientation,
    const SbVec3f &center) const;

  /**
  * Returns the translation, rotation, scale, and scale orientation components of the
  * matrix.
  * [OIVJAVA-WRAPPER NAME{decompose},PACK{Decomposition}]
  */
  void  getTransform(SbVec3f &t, SbRotation &r,
    SbVec3f &s, SbRotation &so) const;

  // The following methods return matrix values and other info:

  /**
  * Returns matrix as a 4x4 array of elements.
  */
  void  getValue(SbMat &m) const;

#ifdef _WIN32
  /**
  * Returns matrix as a 4x4 array of elements.
  */
  SbMat &getValue() { SbMat &rMat = matrix; return rMat; }
#endif

  /**
  * Returns matrix as a 4x4 array of elements.
  */
  const SbMat &getValue() const { return matrix; }

  /**
  * Returns determinant of 3x3 submatrix composed of given row and column indices
  * (0-3 for each).
  */
  float  det3(int r1, int r2, int r3, int c1, int c2, int c3) const;

  /**
  * Returns determinant of upper-left 3x3 submatrix.
  */
  float  det3() const { return det3(0, 1, 2, 0, 1, 2); }

  /**
  * Returns determinant of entire matrix.
  */
  float  det4() const;

  /**
  * Factors a matrix m into 5 pieces: m = r s r^ u t, where r^ means transpose of r,
  * and r and u are rotations, s is a scale, and t is a translation. Any projection
  * information is returned in @B proj@b.
  * [OIVJAVA-WRAPPER PACK{Factorization}]
  */
  SbBool factor(SbMatrix &r, SbVec3f &s, SbMatrix &u,
    SbVec3f &t, SbMatrix &proj) const;

  /**
  * Returns inverse of matrix. Results are undefined for singular matrices. Uses LU
  * decomposition. @BR
  * Matrix is not modified.
  */
  SbMatrix  inverse() const;

  /** Translates this matrice by the given vector */
  void translate(const SbVec3f& translation);

  /** Scales this matrice by the given vector */
  void scale(const SbVec3f& scaleFactor);

  /**
  * Perform in-place LU decomposition of matrix. @B index @b is index of rows in
  * matrix. @B d @b is the parity of row swaps. Returns FALSE if singular.
  */
  SbBool  LUDecomposition(int index[4], float &d);

  /**
  * Perform back-substitution on LU-decomposed matrix. Index is permutation of rows
  * from original matrix.
  */
  void  LUBackSubstitution(int index[4], float b[4]) const;

  /**
  * Returns transpose of matrix.
  * Matrix is not modified.
  */
  SbMatrix  transpose() const;


  // The following methods provide Mx/mx and mx/vec arithmetic:

  /**
  * Post-multiplies the matrix by the given matrix.
  * Matrix is replaced by the result.
  */
  SbMatrix &multRight(const SbMatrix &m);

  /**
  * Pre-multiplies matrix by the given matrix.
  * Matrix is replaced by the result.
  */
  SbMatrix &multLeft(const SbMatrix &m);

  /**
  * Post-multiplies matrix by the given column vector, giving a 3D vector result.
  * The intermediate homogeneous (vec4) value is converted to 3D by
  * dividing the X, Y and Z components by W.
  *
  * \if_java \else It is safe to let src and dst be the same instance of SbVec3f. \endif
  */
  void  multMatrixVec(const SbVec3f &src, SbVec3f &dst) const;

  /**
  * Pre-multiplies matrix by the given row vector, giving a 3D vector result.
  * The intermediate homogeneous (vec4) value is converted to 3D by
  * dividing the X, Y and Z components by W.
  *
  * Use this method to transform a point (position vector). @BR
  * Use multDirMatrix() to transform a normal (direction vector).
  *
  *  \if_java \else It is safe to let src and dst be the same instance of SbVec3f. \endif
  */
  inline void  multVecMatrix(const SbVec3f &src, SbVec3f &dst) const
  {
    float    x,y,z,w;

    x = src[0]*matrix[0][0] + src[1]*matrix[1][0] +
        src[2]*matrix[2][0] + matrix[3][0];
    y = src[0]*matrix[0][1] + src[1]*matrix[1][1] +
        src[2]*matrix[2][1] + matrix[3][1];
    z = src[0]*matrix[0][2] + src[1]*matrix[1][2] +
        src[2]*matrix[2][2] + matrix[3][2];
    w = src[0]*matrix[0][3] + src[1]*matrix[1][3] +
        src[2]*matrix[2][3] + matrix[3][3];

    w = 1.f/w;
    dst.setValue(x*w, y*w, z*w);
  }

  /**
  * Posts-multiplies matrix by the given column vector, giving vector result in homogeneous coordinates.
  * \if_java \else It is safe to let src and dst be the same instance of SbVec3f. \endif
  * [OIVJAVA-WRAPPER NAME{multMatrixVec4}]
  */
  void  multMatrixVec(const SbVec3f &src, SbVec4f &dst) const;

  /**
  * Pre-multiplies matrix by the given row vector, giving vector result in homogeneous coordinates.
  * Use this method to transform a point (position vector). @BR
  * Use multDirMatrix() to transform a normal (direction vector).
  * [OIVJAVA-WRAPPER NAME{multVec4Matrix}]
  */
  void  multVecMatrix(const SbVec3f &src, SbVec4f &dst) const;

  /**
  * Pre-multiplies the matrix by the given row vector, giving vector result.
  * src is assumed to be a direction vector, so translation part of matrix is ignored.
  *
  * Note: If you need to transform surface points and normal vectors by a matrix, call
  * multVecMatrix() for the points and call multDirMatrix() for the normals.
  * Generally normals should be transformed by the inverse transpose of the matrix.
  * However note that the inverse transpose is equal to the original matrix if the matrix
  * is orthonormal, i.e. purely rotational with no scaling or shearing.
  *
  *  \if_java \else It is safe to let src and dst be the same instance of SbVec3f. \endif
  */
  void  multDirMatrix(const SbVec3f &src, SbVec3f &dst) const;

  /**
  * Multiplies the given line's origin by the matrix, and the line's direction by
  * the rotation portion of the matrix.
  *  \if_java \else It is safe to let src and dst be the same instance of SbLine. \endif
  */
  void  multLineMatrix(const SbLine &src, SbLine &dst) const;


  // The following methods are miscellaneous Mx functions:

  /**
  * Prints a formatted version of the matrix to the given file pointer.
  */
  void  print(FILE *fp) const;


  /**
  * Cast: Returns pointer to storage of first element.
  */
  operator float *() { return &matrix[0][0]; }

  /**
  * Cast: returns reference to a 4x4 array.
  */
  operator SbMat &() { return matrix; }

  /**
  * Make it look like a usual matrix (so you can do m[3][2]).
  */
  float * operator [](int i)   { return &matrix[i][0]; }

  /**
  * Make it look like a usual matrix (so you can do m[3][2]).
  */
  const float * operator [](int i) const  { return &matrix[i][0]; }

  /**
  * Sets value from 4x4 array of elements.
  */
  SbMatrix & operator =(const SbMat &m);

  /**
  * Set the matrix from another SbMatrix.
  */
  SbMatrix & operator =(const SbMatrix &m);

  /**
  * Set the matrix from an SbRotation.
  */
  SbMatrix & operator =(const SbRotation &q)  { setRotate(q); return *this; }

  /**
  * Post-multiplies the matrix by the given matrix (equivalent to multRight() method).
  * Matrix is replaced by the resulting matrix.
  */
  SbMatrix & operator *=(const SbMatrix &m)  { return multRight(m); }

  /**
   * Multiply matrix by given vector.
   * Return m * v
   */
  SbVec4f operator *(const SbVec4f& v) const;

 /**
  * Multiplies matrices by vector, returning a vector result.
  * Return v*m.
  */
  INVENTORBASE_API friend SbVec4f operator *(const SbVec4f &v, const SbMatrix &m);

  /**
  * Multiplies two matrices, returning a matrix result.
  */
  INVENTORBASE_API friend SbMatrix operator *(const SbMatrix &m1, const SbMatrix &m2);

  /**
  * Equality comparison operator.
  */
  INVENTORBASE_API friend int operator ==(const SbMatrix &m1, const SbMatrix &m2);
  /**
  * Inequality comparison operator.
  */
  INVENTORBASE_API friend int operator !=(const SbMatrix &m1, const SbMatrix &m2)
  { return !(m1 == m2); }

  /**
  * Equality comparison within given tolerance, for each component.
  */
  SbBool    equals(const SbMatrix &m, float tolerance) const;

  /**
  * Returns true if the matrix is invertible
  */
  bool isInvertible() const;

SoINTERNAL public:
  /**
   * Diagonalizes 3x3 matrix
   */
  void jacobi3(float evalues[3], SbVec3f evectors[3], int &rots) const;

private:
  // Storage for 4x4 matrix
  SbMat matrix;

  SbBool affine_inverse(const SbMatrix &in, SbMatrix &out) const;


};

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SbMatrixd
//
//  4x4 matrix of double precision floating-point elements.
//
//////////////////////////////////////////////////////////////////////////////

/**
* @VSGEXT 4x4 matrix class (double precision).
*
* @ingroup Basics
*
* @DESCRIPTION
*   4x4 double precision matrix class/datatype. Although Open Inventor fields
*   store only single precision values, for certain applications it is useful and
*   convenient to be able to store and manipulate double precision values, for
*   example, double precision coordinate data or values that will be used for
*   further computation.
*
*   See SbMatrix for discussion of storage layout and usage of matrices.
*
* @SEE_ALSO
*    SbMatrix,
*    SbMatrix3,
*    SbRotation,
*    SbRotationd,
*    SbVec2d,
*    SbVec2f,
*    SbVec2i32,
*    SbVec2s,
*    SbVec3d,
*    SbVec3f,
*    SbVec3i32,
*    SbVec3s,
*    SbVec4b,
*    SbVec4d,
*    SbVec4f,
*    SbVec4i32,
*    SbVec4s,
*    SbVec4ub,
*    SbVec4ui32,
*    SbVec4us
* [OIVJAVA-WRAPPER-CLASS BASIC_TYPE{true},LAYOUTED_ARRAY_TYPE{double,16}]
*/
class INVENTORBASE_API SbMatrixd {
public:

  /**
  * Default constructor.  The matrix is initialized with zeros.
  */
  SbMatrixd()    { memset(matrix, 0, sizeof(SbMatd)); }

  /**
  * Constructor.
  */
  SbMatrixd(double a11, double a12, double a13, double a14,
    double a21, double a22, double a23, double a24,
    double a31, double a32, double a33, double a34,
    double a41, double a42, double a43, double a44);

  /**
  * Constructor.
  */
  SbMatrixd(const SbMatd &m);

  /**
  * Sets value from 4x4 array of elements.
  */
  void  setValue(const SbMatd &m);

  /**
   * Sets value from a 16 value double array.
   * [OIVJAVA-WRAPPER-NO-WRAP]
   */
  void    setValue(const double *pMat)
  {
    matrix[0][0] = *pMat++ ;
    matrix[0][1] = *pMat++ ;
    matrix[0][2] = *pMat++ ;
    matrix[0][3] = *pMat++ ;

    matrix[1][0] = *pMat++ ;
    matrix[1][1] = *pMat++ ;
    matrix[1][2] = *pMat++ ;
    matrix[1][3] = *pMat++ ;

    matrix[2][0] = *pMat++ ;
    matrix[2][1] = *pMat++ ;
    matrix[2][2] = *pMat++ ;
    matrix[2][3] = *pMat++ ;

    matrix[3][0] = *pMat++ ;
    matrix[3][1] = *pMat++ ;
    matrix[3][2] = *pMat++ ;
    matrix[3][3] = *pMat++ ;
  } ;

  /**
  * Sets values from a single precision matrix.
  */
  void  setValue(const SbMatrix &m) ;

public:

  /**
  * Sets matrix to be identity.
  */
  void  makeIdentity();

  /**
  * Returns an identity matrix.
  */
  static SbMatrixd  identity();

  /**
  * Sets matrix to rotate by given rotation.
  */
  void  setRotate(const SbRotationd &q);

  /**
  * Sets matrix to scale by given uniform factor.
  */
  void  setScale(double s);

  /**
  * Sets matrix to scale by given vector.
  */
  void  setScale(const SbVec3d &s);

  /**
  * Sets matrix to translate by given vector.
  */
  void  setTranslate(const SbVec3d &t);

  /**
  * Composes the matrix based on a translation, rotation, scale, orientation for
  * scale, and center. The @B scaleOrientation @b chooses the primary axes for the scale.
  * The @B center @b is the center point for scaling and rotation.
  */
  void  setTransform(const SbVec3d &translation,
    const SbRotationd &rotation,
    const SbVec3d &scaleFactor,
    const SbRotationd &scaleOrientation,
    const SbVec3d &center);

  /**
  * Composes the matrix based on a translation, rotation, and scale.  A scale orientation
  * value of (0,0,0,1) is used. The center point for scaling and rotation is (0,0,0).
  */
  void setTransform(const SbVec3d &t, const SbRotationd &r, const SbVec3d &s);

  /**
  * Composes the matrix based on a translation, rotation, scale, and orientation for
  * scale. The @B scaleOrientation @b chooses the primary axes for the scale. The
  * center point for scaling and rotation is (0,0,0).
  */
  void setTransform(const SbVec3d &t, const SbRotationd &r,
    const SbVec3d &s, const SbRotationd &so);

  /**
  * Decomposes the matrix into a translation, rotation, scale, and scale
  * orientation. Any projection information is discarded. The decomposition depends
  * upon choice of center point for rotation and scaling, which is optional as the
  * last parameter. Note that if the center is 0, decompose() is the same as
  * factor() where t is @B translation, @b u is @B rotation, @b s is
  * @B scaleFactor, @b and r is @B scaleOrientation. @b
  * [OIVJAVA-WRAPPER NAME{decompose},PACK{Decomposition}]
  */
  void  getTransform(SbVec3d &translation,
    SbRotationd &rotation,
    SbVec3d &scaleFactor,
    SbRotationd &scaleOrientation,
    const SbVec3d &center) const;

  /**
  * Return translation, rotation, scale, and scale orientation components of the
  * matrix.
  * [OIVJAVA-WRAPPER NAME{decompose},PACK{Decomposition}]
  */
  void getTransform(SbVec3d &t, SbRotationd &r,
    SbVec3d &s, SbRotationd &so) const;


  // The following methods return matrix values and other info:

  /**
  * Returns matrix as a 4x4 array of elements.
  */
  void  getValue(SbMatd &m) const;

#ifdef _WIN32

  /**
  * Returns matrix as a 4x4 array of elements.
  */
  SbMatd &getValue() { SbMatd &rMat = matrix; return rMat; }
#endif

  /**
  * Returns matrix as a 4x4 array of elements.
  */
  const SbMatd &getValue() const { return matrix; }

  /**
  * Returns determinant of 3x3 submatrix composed of given row and column indices
  * (0-3 for each).
  */
  double  det3(int r1, int r2, int r3, int c1, int c2, int c3) const;

  /**
  * Returns determinant of upper-left 3x3 submatrix.
  */
  double  det3() const { return det3(0, 1, 2, 0, 1, 2); }

  /**
  * Returns determinant of entire matrix.
  */
  double  det4() const;

  /**
  * Factors a matrix m into 5 pieces: m = r s r^ u t, where r^ means transpose of r,
  * and r and u are rotations, s is a scale, and t is a translation. Any projection
  * information is returned in @B proj@b.
  * [OIVJAVA-WRAPPER PACK{Factorization}]
  */
  SbBool  factor(SbMatrixd &r, SbVec3d &s, SbMatrixd &u,
    SbVec3d &t, SbMatrixd &proj) const;

  /**
  * Returns inverse of matrix. Results are undefined for singular matrices. Uses LU
  * decomposition. @BR
  * Matrix is not modified.
  */
  SbMatrixd  inverse() const;

  /** Translates this matrice by the given vector */
  void translate(const SbVec3d& translation);

  /** Scales this matrice by the given vector */
  void scale(const SbVec3d& scaleFactor);

  /**
  * Perform in-place LU decomposition of matrix. @B index @b is index of rows in
  * matrix. @B d @b is the parity of row swaps. Returns FALSE if singular.
  */
  SbBool  LUDecomposition(int index[4], double &d);

  /**
  * Perform back-substitution on LU-decomposed matrix. Index is permutation of rows
  * from original matrix.
  */
  void  LUBackSubstitution(int index[4], double b[4]) const;

  /**
  * Returns transpose of matrix.
  * Matrix is not modified.
  */
  SbMatrixd  transpose() const;


  // The following methods provide Mx/mx and mx/vec arithmetic:

  /**
  * Post-multiplies the matrix by given matrix.
  * Matrix is replaced by the result.
  */
  SbMatrixd &multRight(const SbMatrixd &m);

  /**
  * Pre-multiplies the matrix by given matrix.
  * Matrix is replaced by the result.
  */
  SbMatrixd &multLeft(const SbMatrixd &m);

  /**
  * Post-multiplies matrix by given column vector, giving a 3D vector result.
  * The intermediate homogeneous (vec4) value is converted to 3D by
  * dividing the X, Y and Z components by W.
  *
  * \if_java \else It is safe to let src and dst be the same instance of SbVec3f. \endif
  */
  void  multMatrixVec(const SbVec3d &src, SbVec3d &dst) const;

  /**
  * Pre-multiplies matrix by the given row vector, giving a 3D vector result.
  * The intermediate homogeneous (vec4) value is converted to 3D by
  * dividing the X, Y and Z components by W.
  *
  * Use this method to transform a point (position vector). @BR
  * Use multDirMatrix() to transform a normal (direction vector).
  *
  *  \if_java \else It is safe to let src and dst be the same instance of SbVec3f. \endif
  */
  void  multVecMatrix(const SbVec3d &src, SbVec3d &dst) const;

  /**
   * @brief Pre-multiplies matrix by the given row vector, giving vector result in homogeneous coordinates.
   * @details Use this method to transform a point (position vector). @BR
   * Use multDirMatrix() to transform a normal (direction vector).
   * @param src the source vector to transform
   * @param dst the destination vector to retrieve the result
   * \if_java \else It is safe to let src and dst be the same instance of SbVec3d. \endif
   * [OIVJAVA-WRAPPER NAME{multVec4Matrix}]
   */
  void multVecMatrix( const SbVec3d& src, SbVec4d& dst ) const;

  /**
  * Pre-multiplies the matrix by the given row vector, giving vector result.
  * src is assumed to be a direction vector, so translation part of matrix is ignored.
  *
  * Note: If you need to transform surface points and normal vectors by a matrix, call
  * multVecMatrix() for the points and call multDirMatrix() for the normals.
  * Generally normals should be transformed by the inverse transpose of the matrix.
  * However note that the inverse transpose is equal to the original matrix if the matrix
  * is orthonormal, i.e. purely rotational with no scaling or shearing.
  *
  *  \if_java \else It is safe to let src and dst be the same instance of SbVec3f. \endif
  */
  void  multDirMatrix(const SbVec3d &src, SbVec3d &dst) const;

  /**
  * Multiplies the given line's origin by the matrix, and the line's direction by
  * the rotation portion of the matrix.
  *  \if_java \else It is safe to let src and dst be the same instance of SbLine. \endif
  */
  void  multLineMatrix(const SbLined &src, SbLined &dst) const;

  /**
   * Multiply matrix by given vector.
   * Return m * v
   */
  SbVec4d operator *(const SbVec4d& v);


  // The following methods are miscellaneous Mx functions:

  /**
  * Prints a formatted version of the matrix to the given file pointer.
  */
  void  print(FILE *fp) const;


  /**
  * Cast: returns pointer to storage of first element.
  */
  operator double *() { return &matrix[0][0]; }

  /**
  * Cast: returns reference to 4x4 array.
  */
  operator SbMatd &() { return matrix; }

  /**
  * Make it look like a usual matrix (so you can do m[3][2]).
  */
  double * operator [](int i)   { return &matrix[i][0]; }

  /**
  * Make it look like a usual matrix (so you can do m[3][2]).
  */
  const double * operator [](int i) const  { return &matrix[i][0]; }

  /**
  * Sets value from 4x4 array of elements.
  */
  SbMatrixd & operator =(const SbMatd &m);

  /**
  * Set the matrix from another SbMatrixd.
  */
  SbMatrixd & operator =(const SbMatrixd &m);

  /**
  * Set the matrix from an SbRotationd.
  */
  SbMatrixd & operator =(const SbRotationd &q)  { setRotate(q); return *this; }

  /**
  * Post-multiplies the matrix by the given matrix (equivalent to multRight() method).
  * Matrix is replaced by the resulting matrix.
  */
  SbMatrixd & operator *=(const SbMatrixd &m)  { return multRight(m); }

  /**
  * Multiplies matrices by vector, returning a vector result.
  * Return v*m.
  */
  INVENTORBASE_API friend SbVec4d operator *(const SbVec4d &v, const SbMatrixd &m);

  /**
  * Multiplies two matrices, returning a matrix result.
  */
  INVENTORBASE_API friend SbMatrixd operator *(const SbMatrixd &m1, const SbMatrixd &m2);

  /**
  * Equality comparison operator.
  */
  INVENTORBASE_API friend int operator ==(const SbMatrixd &m1, const SbMatrixd &m2);
  /**
  * Inequality comparison operator.
  */
  INVENTORBASE_API friend int operator !=(const SbMatrixd &m1, const SbMatrixd &m2)
  { return !(m1 == m2); }

  /**
  * Equality comparison within given tolerance, for each component.
  */
  SbBool    equals(const SbMatrixd &m, double tolerance) const;

  /**
  * Returns true if the matrix is invertible
  */
  bool isInvertible() const;

SoINTERNAL public:

  /**
   * Diagonalizes 3x3 matrix 
   */
  void  jacobi3(double evalues[3], SbVec3d evectors[3], int &rots) const;

private:
  // Storage for 4x4 matrix
  SbMatd  matrix;

  SbBool affine_inverse(const SbMatrixd &in, SbMatrixd &out) const;


};

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SbMatrix3
//
//  3x3 matrix of floating-point elements.
//
//////////////////////////////////////////////////////////////////////////////



/**
* 3x3 matrix class.
*
* @ingroup Basics
*
* @DESCRIPTION
*   3x3 matrix class/datatype.
*
*   See SbMatrix for discussion of storage layout and usage of matrices.
*
* @SEE_ALSO
*    SbMatrix,
*    SbMatrixd,
*    SbRotation,
*    SbRotationd,
*    SbVec2d,
*    SbVec2f,
*    SbVec2i32,
*    SbVec2s,
*    SbVec3d,
*    SbVec3f,
*    SbVec3i32,
*    SbVec3s,
*    SbVec4b,
*    SbVec4d,
*    SbVec4f,
*    SbVec4i32,
*    SbVec4s,
*    SbVec4ub,
*    SbVec4ui32,
*    SbVec4us
*
* [OIVJAVA-WRAPPER-CLASS BASIC_TYPE{true},LAYOUTED_ARRAY_TYPE{float,9}]
*/
class INVENTORBASE_API SbMatrix3 {
public:

  /**
  * Default constructor.  The matrix is initialized with zeros.
  */
  SbMatrix3() { memset(matrix, 0, sizeof(SbMat3)); }

  /**
  * Constructor.
  */
  SbMatrix3(float a11, float a12, float a13,
    float a21, float a22, float a23,
    float a31, float a32, float a33);

  /**
  * Constructor.
  */
  SbMatrix3(const SbMat3 &m);

  /**
  * Sets value from 3x3 array of elements.
  */
  void  setValue(const SbMat3 &m);

  /**
   * Sets value from a 9 value float array.
   * [OIVJAVA-WRAPPER-NO-WRAP]
   */
  void    setValue(const float *pMat)
  {
    matrix[0][0] = *pMat++ ;
    matrix[0][1] = *pMat++ ;
    matrix[0][2] = *pMat++ ;

    matrix[1][0] = *pMat++ ;
    matrix[1][1] = *pMat++ ;
    matrix[1][2] = *pMat++ ;

    matrix[2][0] = *pMat++ ;
    matrix[2][1] = *pMat++ ;
    matrix[2][2] = *pMat++ ;
  } ;

  /**
  * Sets matrix to be identity.
  */
  void  makeIdentity();

  /**
  * Returns an identity matrix.
  */
  static SbMatrix3  identity();

  /**
  * Sets matrix to rotate by given rotation.
  */
  void  setRotate(const SbRotation &q);

  /**
  * Sets matrix to scale by given uniform factor.
  */
  void  setScale(float s);

  /**
  * Sets matrix to scale by given vector.
  */
  void  setScale(const SbVec3f &s);

  /**
  * Returns 3x3 array of elements.
  */
  void  getValue(SbMat3 &m) const;

  /**
  * Returns 3x3 array of elements.
  */
  const SbMat3 &getValue() const { return matrix; }

  /**
  * Post-multiplies the matrix by the given matrix.
  * Matrix is replaced by the result.
  */
  SbMatrix3 &multRight(const SbMatrix3 &m);

  /**
  * Pre-multiplies matrix by the given matrix.
  * Matrix is replaced by the result.
  */
  SbMatrix3 &multLeft(const SbMatrix3 &m);

  /**
  * Post-multiplies matrix by the given column vector, giving a 3D vector result.
  * \if_java \else It is safe to let src and dst be the same instance of SbVec3f. \endif
  */
  void  multMatrixVec(const SbVec3f &src, SbVec3f &dst) const;

  /**
  * Pre-multiplies matrix by the given row vector, giving a 3D vector result.
  *  \if_java \else It is safe to let src and dst be the same instance of SbVec3f. \endif
  */
  void  multVecMatrix(const SbVec3f &src, SbVec3f &dst) const;

  /**
   * Returns the determinant of the matrix.
   */
  float det() const;

  /**
   * Returns the inverse of the matrix. Results are undefined for singular matrices. @BR
   * The matrix is not modified.
   */
  SbMatrix3 inverse() const;

  /**
  * Prints a formatted version of the matrix to the given file pointer.
  */
  void  print(FILE *fp) const;

  /**
  * Cast: returns pointer to storage of first element.
  */
  operator float *() { return &matrix[0][0]; }

  /**
  * Cast: returns reference to 3x3 array.
  */
  operator SbMat3 &() { return matrix; }

  /**
  * Make it look like a usual matrix (so you can do m[2][2]).
  */
  float * operator [](int i)   { return &matrix[i][0]; }

  /**
  * Make it look like a usual matrix (so you can do m[2][2]).
  */
  const float * operator [](int i) const  { return &matrix[i][0]; }

 /**
  * Multiplies matrices by vector, returning a matrix result.
  * Return M*v
  */
  SbVec3f operator *(const SbVec3f &v) const;

  /**
  * Sets value from 3x3 array of elements.
  */
  SbMatrix3 & operator =(const SbMat3 &m);

  /**
  * Set the matrix from another SbMatrix3.
  */
  SbMatrix3 & operator =(const SbMatrix3 &m);

  /**
  * Set the matrix from an SbRotation.
  */
  SbMatrix3 & operator =(const SbRotation &q)  { setRotate(q); return *this; }

  /**
  * Post-multiplies the matrix by the given matrix (equivalent to multRight() method).
  * Matrix is replaced by the resulting matrix.
  */
  SbMatrix3 & operator *=(const SbMatrix3 &m)  { return multRight(m); }

  /**
  * Multiplies matrices by vector, returning a vector result.
  * Return v*m.
  */
  INVENTORBASE_API friend SbVec3f operator *(const SbVec3f &v, const SbMatrix3 &m);

  /**
  * Multiplies two matrices, returning a matrix result.
  */
  INVENTORBASE_API friend SbMatrix3 operator *(const SbMatrix3 &m1, const SbMatrix3 &m2);

  /**
  * Equality comparison operator.
  */
  INVENTORBASE_API friend int operator ==(const SbMatrix3 &m1, const SbMatrix3 &m2);
  /**
  * Inequality comparison operator.
  */
  INVENTORBASE_API friend int operator !=(const SbMatrix3 &m1, const SbMatrix3 &m2)
  { return !(m1 == m2); }

private:
  SbMat3 matrix;    // Storage for 3x3 matrix
};

inline std::ostream& operator << (std::ostream& os, const SbMatrix& mat)
{
  for ( int j = 0; j < 4; j++ )
  {
    os << "[ ";
    for ( int i = 0; i < 4; i++ )
      os << mat[j][i] << " ";
    os << "]" << std::endl;
  }
  return os;
}

inline std::ostream& operator << (std::ostream& os, const SbMatrix3& mat)
{
  for ( int j = 0; j < 3; j++ )
  {
    os << "[ ";
    for ( int i = 0; i < 3; i++ )
      os << mat[j][i] << " ";
    os << "]" << std::endl;
  }
  return os;
}

#endif /* _SB_MATRIX_ */


