/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#ifndef HIDDEN_FROM_DOC

#ifndef _SB_PIMPL_MACRO_H_
#define _SB_PIMPL_MACRO_H_


// Define supported C++ level for PIMPL design, not that
// C++ 11 is fully supported if __cplusplus >= 201103L.
// This value defines the way we store the implementation class:
// "0", using SoRef
// "1", using std::auto_ptr
// "2", using std::unique_ptr (c++ 11 should be at least partially supported)
#define OIV_PIMPL_CXX11 0

#if !(OIV_PIMPL_CXX11)
#include <Inventor/misc/SoRefCounter.h>
#else
#include <memory>
#endif

/**
 * Private impl design pattern implementation.
 *   _____             _______
 *  |  A  | <>------- | AImpl |
 *  |_____| --------<>|_______|
 *     |                  |
 *     |                  |
 *   __|__             ___|___
 *  |  B  |           | BImpl |
 *  |_____|           |_______|
 *
 *
 * example:
 *
 * - File A.h:
 *
 *   #include <Inventor/SbPImpl.h>
 *   SO_PIMPL_BASE_PUBLIC_DECLARATION(A)
 *   class A
 *   {
 *     SO_PIMPL_BASE_PUBLIC_HEADER(A);
 *   }
 *
 * - File A.cxx:
 *
 *   #include <.../A.h>
 *   #include <.../AImpl.h>
 *   using inventor::impl::AImpl;
 *   A::A() : m_impl( new AImpl(this) ) { [...] }
 *   A::A( AImpl* impl ) : m_impl( impl ) { [...] }
 *
 * - File AImpl.h:
 *
 *   #include <Inventor/SbPImpl.h>
 *   #include <.../A.h>
 *   namespace inventor
 *   {
 *     namespace impl
 *     {
 *       class AImpl : public SbPImpl<A>
 *       {
 *        public:
 *         AImpl( A* pub );
 *       }
 *     }
 *   }
 *
 * - File AImpl.cxx:
 *
 *   #include <.../A.h>
 *   #include <.../AImpl.h>
 *   namespace inventor
 *   {
 *     namespace impl
 *     {
 *       AImpl::AImpl( A* pub ) : SbPImpl<A>( pub ) { [...] }
 *     }
 *   }
 *
 * - File B.h:
 *
 *   #include <Inventor/SbPImpl.h>
 *   #include <.../A.h>
 *   SO_PIMPL_PUBLIC_DECLARATION(B)
 *   class B : public A
 *   {
 *     SO_PIMPL_PUBLIC_HEADER(B);
 *   }
 *
 * - File B.cxx:
 *
 *   #include <.../B.h>
 *   #include <.../BImpl.h>
 *   using inventor::impl::BImpl;
 *   B::B() : A( new BImpl(this) ) { [...] }
 *   B::B( BImpl* impl ) : A( impl ) { [...] }
 *
 * - File BImpl.h:
 *
 *   #include <Inventor/SbPImpl.h>
 *   #include <.../AImpl.h>
 *   class B;
 *   namespace inventor
 *   {
 *     namespace impl
 *     {
 *       class BImpl : public AImpl
 *       {
 *        public:
 *         BImpl( B* pub );
 *       }
 *     }
 *   }
 *
 * - File BImpl.cxx:
 *
 *   #include <.../B.h>
 *   #include <.../BImpl.h>
 *   namespace inventor
 *   {
 *     namespace impl
 *     {
 *       BImpl::BImpl( B* pub ) : AImpl( pub ) { [...] }
 *     }
 *   }
 */

namespace inventor
{
namespace impl
{

/**
 * @brief Base class for private implementation class
 */
template<typename PublicType>
#if !(OIV_PIMPL_CXX11)
class SbPImpl : public SoRefCounter
#else
class SbPImpl
#endif
{
public:

  /**
   * Constructor
   */
  SbPImpl( PublicType* p ) : m_public(p) {}

  /**
   * Destructor
   */
  virtual ~SbPImpl() {}

  /**
   * Access to public node
   */
  template<typename SuperPublicType>
  SuperPublicType* getPublic () const { return static_cast<SuperPublicType*>(m_public); }

protected:

  // public node
  PublicType* m_public;

private:

  // force using constructor with parameter
  SbPImpl () {}
};

} // namespace impl
} // namespace inventor

/** Macro used by classe B. Define some helpers declaration.
 * Must be called before the class declaration. */
#define SO_PIMPL_PUBLIC_DECLARATION(Class) \
namespace inventor\
{\
namespace impl\
{\
class Class##Impl;\
}\
}

/** Macro used by classe A. Define some helpers declaration.
 * Must be called before the class declaration. */
#define SO_PIMPL_BASE_PUBLIC_DECLARATION(Class) SO_PIMPL_PUBLIC_DECLARATION(Class)

/** Macro used by classe B. Define the constructor that take an AImpl* in parameter.
 * Must be called at the beginning of class declaration. */
#define SO_PIMPL_PUBLIC_HEADER(Class) \
SoINTERNAL protected:\
  friend class inventor::impl::Class##Impl;\
  /** [OIV-WRAPPER NO_WRAP] */ \
  Class(inventor::impl::Class##Impl *impl);

/** Macro used by abstract class to not break derivation process. If a hierarchy is defined
 * such as A -> B -> C. A is the base class for Pimpl scheme. If B is abstract, this breaks
 * the derivation rules for Pimpl scheme. B must have a constructor for parsing the C pimpl
 * class to the A class.
 */
#define SO_PIMPL_ABSTRACT_PUBLIC_HEADER(Class, ParentClass) \
SoINTERNAL protected:\
  /** [OIV-WRAPPER NO_WRAP] */ \
  Class(inventor::impl::ParentClass##Impl *impl);

/** Macro used by classe B if class B is final, i.e. not derivable in the PImpl sense.
  * In this way, the constructor with impl type is not defined. It means that the class
  * is derivable but not its implementation. */
#define SO_PIMPL_FINAL_PUBLIC_HEADER(Class) \
SoINTERNAL protected:\
  friend class inventor::impl::Class##Impl;

/** Macro used by A.
 * Declare the private implementation pointer. 
 * Must be called at the beginning of class declaration. */
#if OIV_PIMPL_CXX11 == 0
#define SO_PIMPL_BASE_PUBLIC_HEADER(Class) \
SO_PIMPL_PUBLIC_HEADER(Class) \
  SoRef< inventor::impl::SbPImpl< Class > > m_impl; \
  /** [OIV-WRAPPER NO_WRAP] */ \
  template<typename ImplType> \
  ImplType* getImpl() const { \
    assert(m_impl->getPublic< Class >() == this); \
    return static_cast< ImplType *>(m_impl.ptr()); \
  } \
  /** [OIV-WRAPPER NO_WRAP] */ \
  void setImpl( inventor::impl::SbPImpl< Class >* impl ) { m_impl = impl; }
#elif OIV_PIMPL_CXX11 == 1
#define SO_PIMPL_BASE_PUBLIC_HEADER(Class) \
SO_PIMPL_PUBLIC_HEADER(Class) \
  std::auto_ptr< inventor::impl::SbPImpl< Class > > m_impl; \
  /** [OIV-WRAPPER NO_WRAP] */ \
  template<typename ImplType> \
  ImplType* getImpl() const { \
    assert(m_impl->getPublic< Class >() == this); \
    return static_cast< ImplType *>(m_impl.ptr()); \
  } \
  /** [OIV-WRAPPER NO_WRAP] */ \
  void setImpl ( inventor::impl::SbPImpl< Class >* impl ) { m_impl.reset(impl); }
#else
#define SO_PIMPL_BASE_PUBLIC_HEADER(Class) \
SO_PIMPL_PUBLIC_HEADER(Class);\
  std::unique_ptr< inventor::impl::SbPImpl< Class > > m_impl; \
  /** [OIV-WRAPPER NO_WRAP] */ \
  template<typename ImplType> \
  ImplType* getImpl() const { \
    assert(m_impl->getPublic< Class >() == this); \
    return static_cast< ImplType *>(m_impl.ptr()); \
  } \
  /** [OIV-WRAPPER NO_WRAP] */ \
  void setImpl( inventor::impl::SbPImpl< Class >* impl ) { m_impl.reset( impl ); }
#endif

#endif //_SB_PIMPL_MACRO_H_

#endif // HIDDEN_FROM_DOC
