/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2022 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef _SO_INPUT_
#define _SO_INPUT_

#include <Inventor/misc/SoBasic.h>
#include <Inventor/SbDict.h>
#include <Inventor/SbPList.h>
#include <Inventor/SbString.h>
#include <Inventor/SoDB.h>
#include <Inventor/threads/SbThreadMutex.h>
#include <Inventor/threads/SbThreadSpinlock.h>

#include <Inventor/STL/fstream>
#include <Inventor/STL/iostream>
#include <Inventor/STL/list>
#include <Inventor/STL/vector>
#include <Inventor/STL/map>

#if defined(_WIN32)
#pragma warning( push )
#pragma warning( disable: 4251 ) // 'identifier' : class 'type' needs to have dll-interface to be used by clients of class 'type2'
#endif


class SoNode;
class SoPath;
class SoBase;
class SoDB;
class SbStringList;
class SoInputImpl;
class SoStreamBuffer;
class SbThreadSignal;
class SoInputParameters;

#ifndef HIDDEN_FROM_DOC
//////////////////////////////////////////////////////////////////////////////
//
// Structure: SoInputFile (internal)
//
// This structure holds info about an opened file for use in the SoInput
// class.
//
// One of the items is a dictionary that correlates reference names
// in files to nodes and paths (SoBase instances).
//
//////////////////////////////////////////////////////////////////////////////

SoINTERNAL struct SoInputFile
{
  // Name of file
  SbString name;
  // Name of file with full path
  SbString fullName;
  // File pointer
  FILE *fp;
  // ZCompressed file pointer
  void* zFp;
  // Buffer to read from (or NULL)
  void *buffer;
  // TRUE if buffer is created internally
  SbBool bufferOwnership;
  // Current location in buffer
  char *curBuf;
  // Buffer size
  size_t bufSize;
  // Number of line currently reading
  int lineNum;
  // TRUE if opened by SoInput
  SbBool openedHere;
#ifdef _WIN32
  // TRUE if opened from handle
  SbBool openedFromHandle;
#endif
  // TRUE if file has binary data
  SbBool binary;
  // TRUE if file uses UTF8 incoding
  SbBool utf8;
  // True if file is compressed using zlib
  SbBool zCompressed;
  // TRUE if current platform endianness is big endian
  SbBool isBigEndian;
  // TRUE if the file is streamed
  SbBool isStreamed;
  // Number of buffers used for the streaming process
  int streamedBuffersNumber;
  // Size of a streamBuffer;
  int streamedBuffersSize;
  // Size of the file in bytes
  long long fileSize;
  // TRUE if header was checked for A/B
  SbBool readHeader;
  // TRUE if header was read ok
  bool headerOk;
  // Node/path reference dictionary
  SbDict *refDict;
  // TRUE if dict from another SoInput
  SbBool borrowedDict;
  // Version if standard Inventor file;
  float ivVersion;
  // The header string of the input file
  SbString headerString;
  // CB to be called after reading file
  SoDBHeaderCB *postReadCB;
  // User data to pass to the postReadCB
  void *CBData;
  // Too complex for inlining
  SoInputFile();
};

SoINTERNAL struct SoStreaming
{
};

#endif // HIDDEN_FROM_DOC

//////////////////////////////////////////////////////////////////////////////
//
// Class: SoInput
//
// This file contains the definition of the SoInput (input stream)
// class. This class is used for reading Inventor data files; it
// supports both ASCII (default) and binary formats. It skips over
// Inventor comments (from '#' to end of line) and can stack input
// files when file inclusion is encountered; when EOF is reached, the
// stack is popped. Both the ASCII and binary formats can be compressed
// using the gz format (e.g: Using gzip). The file extension for compressed files is 'ivz' but it's
// not mandatory, the gz header is checked when the file is opened in order
// to figure out if it is compressed or not. This feature uses the module IvDLZlib.
//
// Another feature is file opening. This will open a named file,
// looking for it in any of several user-specified directories.
// Alternatively, the caller can specify a buffer in memory (and its
// size) to read from. The specified filenames can contain variables
// which are replaced by SoInput using SoPreferences. The variables must
// follow the usual $name format. (e.g: $OIVHOME).
//
//
// SoInput also contains a dictionary that correlates node and path
// pointers to temporary names written to files. This is so
// references to previously defined nodes and paths are written
// correctly.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Used to read Open Inventor data files.
 *
 * @ingroup General
 *
 * @DESCRIPTION
 *   This class is used by the SoDB reading routines when reading Open Inventor
 *   (.iv) format data or various geometry data formats that can be converted into
 *   an Open Inventor scene graph.  The input source can be the standard input
 *   stream (default), a file (see openFile) or a buffer in memory (see setBuffer).  
 *   SoInput can search for a file in a
 *   list of directories (see addDirectoryFirst etc).  SoInput can report progress
 *   during the input operation (see updateReadPercent).
 *
 *   SoInput is typically used with SoDB::read or SoDB::readAll to load data and
 *   create a scene graph. There is an example code fragment on the SoDB page.
 *
 *   @B NOTE: @b Applications should always check the result of calling the #openFile()
 *   method.  If the call fails, the SoInput object itself is still valid and
 *   remains in the default state.  Calling SoDB::readAll(), or methods like
 *   #getHeader() and #isValidFile(), with a default SoInput object will make the 
 *   application appear to "hang" because it is waiting for data on standard input.
 *   To check if a file exists, get file properties, etc see SbFileHelper.
 *
 * @B File formats @b
 *
 *   SoInput supports ASCII (default) and binary Open Inventor formats.
 *   Since Open Inventor 8.1, both the ASCII and binary formats can be compressed
 *   using the gz format (e.g. using gzip). The file extension 'ivz' is recognized
 *   for compressed Open Inventor files, but is not mandatory.  In fact the initial
 *   bytes of the file are checked when the file is opened to automatically determine
 *   if it is compressed or not. This feature uses the module IvDLZlib.
 *
 *   SoInput also has built-in support for:
 *   - VRML97 format (.wrl)
 *   - X3D Classic VRML format (.x3dv)
 *   See @ref VRML_LIMITATIONS for details.
 *
 *   Additional input readers are provided as plugins for:
 *   - DXF format (.dxf)
 *   - STL format (.stl)
 *
 *   These formats require redistributing the corresponding plugin library with the
 *   application.
 *   - DXF : fei_inventor_io_dxf (e.g. fei_inventor_io_dxf.dll on Windows).
 *   - STL : fei_inventor_io_stl (e.g. fei_inventor_io_stl.dll on Windows).
 *
 *   Open Inventor also supports many standard CAD file formats.
 *   See SoCADInputReader for the current list.  These formats also require
 *   redistributing the corresponding plugin library (fei_inventor_io_cad)
 *   with the application.
 *
 *   SoInput can also be extended with custom file format readers.
 *   Input readers are defined as new classes inheriting from the SoInputReader class..
 *   See SoInputReader class for more information about defining and loading new readers
 *
 *   For Open Inventor format files, users
 *   can also register additional valid file headers. When reading, SoInput skips
 *   over Open Inventor comments (from '#' to end of line) and can stack input files.
 *   When EOF is reached, the stack is popped.
 *
 * @B Directory List @b
 *
 * SoInput maintains a list of directories (initially empty). Generally Open Inventor
 * classes that need to open a named file will search for the file in each of these
 * directories in order. This includes SoInput (the openFile() method), SoTexture2,
 * SoFragmentShader and other nodes that have a "filename" field.
 *
 * Note: SoInput is @I not @i currently used for loading volume data files
 * in VolumeViz.  SoDataSet and its subclasses (SoVolumeData etc) do not search the
 * list of directories maintained by SoInput when attempting to open a file.
 * However the directory list is searched to find volume shader files (SoVolumeShader,
 * SoVolumeRenderingQuality, etc).
 *
 * The SoInput directory list is initialized from the environnment variable 
 * OIV_FILE_SEARCH_PATHS.  This variable may be set to a semi-colon separated list of 
 * directory paths. The specified paths may use variables in $name format, e.g. 
 * "$OIVHOME", which will be replaced by the value returned by SoPreferences for
 * that name.  Additional directories may be added to the directory list using the 
 * static methods addDirectoryLast etc.
 *
 * Static utility method #findAbsolutePath() returns the full path of a file
 * if it can be found in one of the directories in the list.
 *
 * @anchor DXF_LIMITATIONS @B DXF Input Limitations@b:
 *
 * The supported DXF file format release is 14.
 *
 * The limitations are:
 *  - CLASSES, OBJECTS and THUMBNAILIMAGE sections are ignored.
 *  - TABLES section: APPID, BLOCK_RECORD,DIMSTYLE, LTYPE, LAYER, STYLE, UCS, VIEW, VIEWPORT are ignored.
 *  - ENTITIES section: ARCALIGNEDTEXT, DIMENSION, IMAGE, REGION, WIPEOUT are ignored.
 *
 *  @anchor VRML_LIMITATIONS @B VRML Input Limitations@b:
 *
 *  Starting from OIV10, it is no more possible to create VRML nodes. However, import of .vrml file
 *  is till supported but node are automatically converted to OIV nodes.
 *  Not all nodes are supported when reading VRML files. Here is list of supported nodes and their
 *  corresponding OIV nodes. Some VRML nodes will be converter to more or less complexe scenegraph.
 *  In this case, VRML node may be converted to an SoSeparator or an SoGroup containing scene graph.
 *
 *  @B GeoVRML nodes: @b
 *  - SoVRMLGeoLOD -> SoGeoLOD
 *  - SoVRMLGeoElevationGrid -> SoSeparator
 *  - SoVRMLGeoOrigin -> SoGeoOrigin
 *  - SoVRMLGeoLocation -> SoSeparator
 *  - SoVRMLGeoCoordinate -> SoGeoCoordinate
 *
 *  @B Special Group nodes: @b
 *  - SoVRMLInline -> SoSeparator
 *  - SoVRMLLOD -> SoLevelOfDetail
 *  - SoVRMLSwitch -> SoSwitch
 *
 *  @B Common nodes: @b
 *  - SoVRMLDirectionalLight -> SoDirectionalLight
 *  - SoVRMLPointLight -> SoPointLight
 *  - SoVRMLSpotLight -> SoSpotLight
 *  - SoVRMLWorldInfo -> SoInfo
 *
 *  @B Geometry nodes: @b
 *  - SoVRMLElevationGrid -> SoSeparator
 *  - SoVRMLExtrusion -> SoGroup
 *  - SoVRMLIndexedFaceSet -> SoGroup
 *  - SoVRMLIndexedLineSet -> SoGroup
 *  - SoVRMLPointSet -> SoGroup
 *  - SoVRMLBox -> SoCube
 *  - SoVRMLCone -> SoCone
 *  - SoVRMLCylinder -> SoCylinder
 *  - SoVRMLSphere -> SoSphere
 *  - SoVRMLText -> SoSeparator
 *  - SoVRMLShape -> SoSeparator
 *
 *  @B Geometry properties: @b
 *  - SoVRMLCoordinate -> SoCoordinate3
 *  - SoVRMLColor -> SoMaterial
 *  - SoVRMLTextureCoordinate -> SoTextureCoordinate2
 *  - SoVRMLNormal -> SoNormal
 *
 *  @B Geometry appearance: @b
 *  - SoVRMLMaterial -> SoMaterial
 *  - SoVRMLImageTexture -> SoTexture2
 *  - SoVRMLPixelTexture -> SoTexture2
 *  - SoVRMLTextureTransform -> SoTexture2Transform
 *  - SoVRMLFontStyle -> SoFontStyle
 *  - SoVRMLAppearance -> SoGroup
 *
 *  @B Group nodes: @b
 *  - SoVRMLTransform -> SoSeparator
 *  - SoVRMLAnchor -> SoSeparator
 *  - SoVRMLBillboard -> SoBillboard
 *  - SoVRMLCollision -> SoSeparator
 *  - SoVRMLParent -> SoGroup
 *  - SoVRMLGroup -> SoSeparator
 *  - SoVRMLStaticGroup -> SoSeparator
 *
 *  @B Special nodes: @b
 *  - SoVRMLPROTOInstance -> SoSeparator
 *
 *
 * @SEE_ALSO
 *    SoDB,
 *    SoOutput,
 *    SoTranReceiver,
 *    SoInputParameters,
 *    SoInputReader,
 *    SoCADInputReaderParameters
 *
 *  [OIVNET-WRAPPER-CLASS DERIVABLE{Default},CUSTOM_DISPOSE,EXTRA_CONSTRUCTOR]
 */
class INVENTOR_API SoInput
{

 public:

  /**
   * Constructor. The default SoInput reads from stdin.
   */
  SoInput();

  /**
   * The destructor closes any files opened by the SoInput.
   */
  virtual ~SoInput();

  /**
   * The SoInput class maintains a global list of directories that is searched to
   * find files when opening them. Directories are searched in order. This
   * routine adds a directory to the beginning of the list.
   *
   * @UNICODE_WARNING
   */
  SoNONUNICODE static void addDirectoryFirst(const char *dirName);


  /**
   * The SoInput class maintains a global list of directories that is searched to
   * find files when opening them. Directories are searched in order. This
   * routine adds a directory to the beginning of the list.
   */
  static void addDirectoryFirst( const SbString& dirName );

  /**
   * The SoInput class maintains a global list of directories that is searched to
   * find files when opening them. Directories are searched in order. This
   * routine adds a directory to the end of the list.
   *
   * @UNICODE_WARNING
   */
  SoNONUNICODE static void addDirectoryLast(const char *dirName);

  /**
   * The SoInput class maintains a global list of directories that is searched to
   * find files when opening them. Directories are searched in order. This
   * routine adds a directory to the end of the list.
   */
  static void addDirectoryLast( const SbString& dirName );

#define DIRECTORIES_SEPARATOR ":;"
  /**
   * The SoInput class maintains a global list of directories that is searched to
   * find files when opening them. Directories are searched in order.
   * This routine adds directories named in the value of the
   * given environment variable to the beginning of the list.
   *
   * @param envVarName The name of the environment variable.
   * @param dirSep A string containing zero or more directory separator characters.
   *
   * The default value of dirSep is ":;", in other words @I either@i a colon or an
   * a semicolon character will be interpreted as separating directory paths.
   * Directory paths in an environment variable are normally separated by semicolon
   * characters on Microsoft Windows platforms and by colon (or whitespace) on
   * UNIX/Linux systems.
   *
   * NOTE: On Microsoft Windows systems the default dirSep value causes a problem.
   * For example, the string "C:/myDir" will be interpreted as two @I different@i
   * directories "C/" and "/myDir"..  To avoid this problem explicitly set dirSep to ";" (semicolon).
   *
   * @UNICODE_WARNING
   */
  SoNONUNICODE static void addEnvDirectoriesFirst(const char *envVarName,
                                     const char *dirSep = DIRECTORIES_SEPARATOR);

  /**
   * The SoInput class maintains a global list of directories that is searched to
   * find files when opening them. Directories are searched in order.
   * This routine adds directories named in the value of the
   * given environment variable to the beginning of the list.
   *
   * NOTE: On Microsoft Windows systems the default dirSep value causes a problem.
   * For example, the string "C:/myDir" will be interpreted as two @I different@i
   * directories "C/" and "/myDir"..  To avoid this problem explicitly set dirSep to ";" (semicolon).
   *
   * @param envVarName The name of the environment variable.
   * @param dirSep A string containing zero or more directory separator characters.
   */
  static void addEnvDirectoriesFirst(const SbString& envVarName,
                                     const SbString& dirSep = DIRECTORIES_SEPARATOR);

  /**
   * The SoInput class maintains a global list of directories that is searched to
   * find files when opening them. Directories are searched in order.
   * This routine adds directories named in the value of the
   * given environment variable to the end of the list.
   *
   * NOTE: On Microsoft Windows systems the default dirSep value causes a problem.
   * For example, the string "C:/myDir" will be interpreted as two @I different@i
   * directories "C/" and "/myDir"..  To avoid this problem explicitly set dirSep to ";" (semicolon).
   *
   * @param envVarName The name of the environment variable.
   * @param dirSep A string containing zero or more directory separator characters.
   *
   * @UNICODE_WARNING
   */
  SoNONUNICODE static void addEnvDirectoriesLast(const char *envVarName,
                                    const char *dirSep = DIRECTORIES_SEPARATOR);

  /**
   * The SoInput class maintains a global list of directories that is searched to
   * find files when opening them. Directories are searched in order.
   * This routine adds directories named in the value of the
   * given environment variable to the end of the list.
   *
   * NOTE: On Microsoft Windows systems the default dirSep value causes a problem.
   * For example, the string "C:/myDir" will be interpreted as two @I different@i
   * directories "C/" and "/myDir"..  To avoid this problem explicitly set dirSep to ";" (semicolon).
   *
   * @param envVarName The name of the environment variable.
   * @param dirSep A string containing zero or more directory separator characters.
   */
  static void addEnvDirectoriesLast(const SbString& envVarName,
                                    const SbString& dirSep = DIRECTORIES_SEPARATOR);

  /**
   * Removes named directory from the list.
   *
   * @UNICODE_WARNING
   */
  SoNONUNICODE static void removeDirectory(const char *dirName);

  /**
   * Removes named directory from the list.
   */
  static void removeDirectory( const SbString& dirName );

  /**
   * Clears the list of directories (including the current directory).
   */
  static void clearDirectories();

  /**
   * Returns the list of directories as an SbStringList.
   * [OIV-WRAPPER-CUSTOM-CODE]
   */
  static const SbStringList &getDirectories();


  /**
   * Sets file pointer to read from. Clears the stack of input files if necessary.
   * [OIV-WRAPPER-NOT-WRAP]
   */
  virtual void setFilePointer(FILE *newFP);

  /**
   * Gets the file pointer read.
   * [OIV-WRAPPER-NOT-WRAP]
   */
  virtual FILE* getFilePointer();

  /**
  *\if_cpp
  * Opens named file, sets file pointer to result. Clears the stack of input files
  * if necessary.
  *\else
  * Opens named file.
  *\endif
  * Returns FALSE on error.
  * If @B okIfNotFound @b is FALSE (the
  * default), prints an error message if the file could not be found.
  * If @B aSync @b is true (FALSE by default) the file is read asynchronously (the reading
  * starts immediately although it isn't fully buffered).
  *
  * The file name may contain variables in $name format, e.g. "$OIVHOME", which will be 
  * replaced by the value returned by SoPreferences for that name.
  *
  * @UNICODE_WARNING
  */
  SoNONUNICODE virtual SbBool openFile(const char *fileName, SbBool okIfNotFound = FALSE, SbBool aSync = FALSE);

  /**
  *\if_cpp
  * Opens named file, sets file pointer to result. Clears the stack of input files
  * if necessary.
  *\else
  * Opens named file.
  *\endif
  * If @B okIfNotFound @b is FALSE (the
  * default), prints an error message if the file could not be found.
  * If @B aSync @b is true (FALSE by default) the file is read asynchronously (the reading
  * starts immediately although it isn't fully buffered).
  *
  * The file name may contain variables in $name format, e.g. "$OIVHOME", which will be 
  * replaced by the value returned by SoPreferences for that name.
  */
  virtual SbBool openFile( const SbString& fileName, SbBool okIfNotFound = FALSE, SbBool aSync = FALSE);

  /**
  *\if_cpp
  * Opens named file, pushing the resulting file pointer onto the stack.
  *\else
  * Opens named file.
  *\endif
  * Returns FALSE on error. When EOF is reached, the stack is popped.
   *
  * The file name may contain variables in $name format, e.g. "$OIVHOME", which will be 
  * replaced by the value returned by SoPreferences for that name.
 *
  * @UNICODE_WARNING
  */
  SoNONUNICODE virtual SbBool pushFile(const char *fileName);

  /**
  *\if_cpp
  * Opens named file, pushing the resulting file pointer onto the stack.
  *\else
  * Opens named file.
  *\endif
  * Returns FALSE on error. When EOF is reached, the stack is popped.
  *
  * The file name may contain variables in $name format, e.g. "$OIVHOME", which will be 
  * replaced by the value returned by SoPreferences for that name.
  */
  virtual SbBool pushFile( const SbString& fileName );

  /**
   * Closes all files on stack opened with openFile() or pushFile().
   */
  virtual void closeFile();

  /**
   * Returns TRUE if the currently open file is a valid Open Inventor file.
   * File must begin with a standard Open Inventor header or one that has 
   * been registered with SoDB::registerHeader().
   * Note that by default (when no file or buffer is open), SoInput reads 
   * from standard input, which may make the application appear to "hang".
   */
  virtual SbBool isValidFile();

  /**
   * Returns TRUE if the current buffer is valid.
   * Buffer must begin with a standard Open Inventor header or one that has 
   * been registered with SoDB::registerHeader().
   * Note that by default (when no file or buffer is open), SoInput reads 
   * from standard input, which may make the application appear to "hang".
   */
  virtual SbBool isValidBuffer();

  /**
  * Returns the current file.
  * Returns standard input (stdin) if no file or buffer is open.
  * Returns NULL if reading from a buffer.
  */
  virtual FILE *getCurFile() const;

  /**
   * Returns full name (including directory path) of current file.
   * Returns NULL if no file is open or if reading from a buffer.
   *
   * @UNICODE_WARNING
   */
  SoNONUNICODE virtual const char *getCurFileName() const;

  /**
   * Returns full name (including directory path) of current file.
   * Returns an empty string (isEmpty() will be true) if no file is open
   * or if reading from a buffer.
   */
  virtual SbString getCurStringFileName() const;

  /**
   * Sets an in-memory buffer to read from, along with its size.
   * [OIV-WRAPPER-CUSTOM-CODE]
   */
  virtual void setBuffer(void *buffer, size_t bufSize);

  /**
   * Returns the number of bytes read.
   */
  virtual int getNumBytesRead() const;

  /**
   * Returns the header of the file being read.
   * Note that by default (when no file or buffer is open), SoInput reads 
   * from standard input, which may make the application appear to "hang".
   */
  virtual SbString getHeader();

  /**
   * Returns the Open Inventor file version of the file being read (2.1).
   * If the file has a header registered through SoDB::registerHeader(), the returned
   * version is the Open Inventor version registered with the header.
   * Returns 0.0 if no file or buffer is open.
   */
  virtual float getIVVersion() { return curFile->ivVersion; }

  /**
   * Reports the percentage of bytes read from the file. @BR
   * Derive a new class and override this method if you want to implement a progress bar.
   */
  virtual void updateReadPercent( double readPercentage );

  /**
   * Returns absolute path of given file by looking in SoInput standard directories.
   * Returns true on success.
   *
   * The file name may contain variables in $name format, e.g. "$OIVHOME", which will be 
   * replaced by the value returned by SoPreferences for that name.
   */
  static SbBool findAbsolutePath( const SbString& fileName, SbString &fullName );


  /**
  * Specify parameters to modify/control actions during the read of a file. 
  * For example, use an instance of SoCADInputReaderParameters for file formats handled
  * by the SoCADFileFormat module (IGES, STEP, CATIA, ...).
  */
  void setInputParameters( SoInputParameters* parameters);

  /**
  * Return the current SoInputParameters.
  */
  SoInputParameters* getInputParameters() const;

SoEXTENDER public:

  // Returns whether current file/buffer being read is binary
  virtual SbBool isBinary();

  // Returns whether current file/buffer being read uses utf8 encoding
  virtual SbBool isUtf8();

  // Returns whether current file/buffer is compressed using ZLIB
  virtual SbBool isZCompressed();

  // Reads next character from current file/buffer. Returns FALSE on
  // EOF or error.
  virtual SbBool get(char &c);

  // Reads next ASCII character from current buffer. Returns FALSE on
  // EOF or error.
  virtual SbBool getASCIIBuffer(char &c);

  // Reads next ASCII character from current file. Returns FALSE on
  // EOF or error.
  virtual SbBool getASCIIFile(char &c);

  // Reads item of particular type from current file pointer/buffer. All
  // skip white space before reading and return FALSE on EOF or if
  // item could not be read.
  virtual SbBool read(char &c);
  virtual SbBool readByte(char &c);
  virtual SbBool readByte(unsigned char &c);
  virtual SbBool read(SbString &s);
  virtual SbBool read(SbName &n, SbBool validIdent = FALSE);
  virtual SbBool readBitMask(SbName &n, SbBool validIdent = FALSE);
  virtual SbBool read(bool &i);
  virtual SbBool read(int32_t &i);
  virtual SbBool read(int64_t &i);
  virtual SbBool read(uint32_t &i);
  virtual SbBool read(uint64_t &i);
  virtual SbBool read(short &s);
  virtual SbBool read(unsigned short &s);
  virtual SbBool read(float &f);
  virtual SbBool read(double &d);
  virtual SbBool readBinaryArray(unsigned char *c, int length);
  virtual SbBool readBinaryArray(int32_t *l, int length);
  virtual SbBool readBinaryArray(int64_t *l, int length);
  virtual SbBool readBinaryArray(float *f, int length);
  virtual SbBool readBinaryArray(short *s, int length);
  virtual SbBool readBinaryArray(double *d, int length);

  // Returns TRUE if current file/buffer is at EOF
  virtual SbBool eof() const;

  // This read does not skip whiteSpace. It's used to read a PROTO into
  // the buffer. The second argument is just to differentiate the two
  // character reads.
  virtual SbBool read(char &c, SbBool skip);

  // Returns TRUE if current header is VRML V2.0
  SbBool isFileVRML2();

  // Sets the vrmlFile flag. This is necessary for
  // SoFieldData::readFieldDescriptions for UnKnownNodes to temporarily i
  // turn off the flag so Inventor extension nodes field descriptions will
  // read in correctly. They assume that the comma is between the fields
  // but in VRML2, commas are whitespace.
  void setFileVRML2(SbBool flag);

  // Returns 0 for Inventor, 2 for VRML2.0 and 3 for VRML 3.0
  int  getVRMLVersion() { return vrmlFile; }

  // Sets the vrmlFile flag. This is necessary for
  // SoFieldData::readFieldDescriptions for UnKnownNodes to temporarily i
  // turn off the flag so Inventor extension nodes field descriptions will
  // read in correctly. They assume that the comma is between the fields
  // but in VRML2, commas are whitespace.
  void setVRMLVersion(int flag) { vrmlFile = flag; }

#ifdef _WIN32
  // Returns the file name from its handle.
  SbString getFileNameFromHandle( HANDLE hFile ) const;

  virtual SbBool openFromHandle( UINT hFile, SbBool okIfNotFound = FALSE, SbBool aSync = FALSE );
#endif

 SoINTERNAL public:

  // Init function sets up global directory list
  static void init();
  static void finish();

  // Constructor that gets reference dictionary from another SoInput
  SoInput(SoInput *dictIn);

  // Fills in passed string to contain description of current
  // location in all open input files
  virtual void getLocationString(SbString &string) const;

  // Puts a just-read character or string back in input stream/buffer
  virtual void putBack(char c);
  virtual void putBack(const char *string);

  // Adds a reference to dictionary in current file. This may also
  // add a reference to the global dictionary if addToGlobalDict is
  // TRUE (the default).
  virtual void addReference(const SbName &name, SoBase *base,
                            SbBool addToGlobalDict = TRUE);

  // Removes a reference to dictionary in current file. This may
  // also remove a reference from the global dictionary.
  virtual void removeReference(const SbName &name);

  // Looks up a reference, returning the base pointer or NULL
  virtual SoBase * findReference(const SbName &name) const;

  virtual void resetFilePointer(FILE *fptr) { curFile->fp = fptr;}

  virtual SbBool readLargeBinaryArray(unsigned char *c, size_t length);

  SoInputImpl* m_soInputImpl;

  // find absolute path of a file return TRUE if found
  // and the absolute found path in fullName
  //
  // @UNICODE_WARNING
  //
  SoNONUNICODE static SbBool findAbsolutePath( const char* fileName, const SbStringList *some_dir, SbString &fullName );


  // find absolute path of a file return TRUE if found
  // and the absolute found path in fullName
  //
  static SbBool findAbsolutePath( const SbString& fileName, const SbStringList *some_dir, SbString &fullName );

  int32_t isReadingBinArrayAsMemObj() const;

  // Adds/subtracts the new value to/from the number of read bytes. Useful when an fseek is done on the
  // currently opened file in order to keep m_numReadBytes up to date (i.e. reading PROTO of vrml files).
  void updateNumReadBytes( int value);

  // Set the Inventor version number of the current file
  void setIVVersion(float version) { curFile->ivVersion = version; }
  bool isHeaderOk() { return curFile->headerOk; }

  void seekCurBuf( size_t size ) { curFile->curBuf += size; updateNumReadBytes(static_cast<int>(size) ); }

  /**
   * Read a text file and put the contents into the string variable @param dest.
   * Returns true on success.
   *
   * If the file has been cached in memory using #addInMemoryTextFile the
   * cached file contents are returned, else the file is opened and read.
   * This mechanism is enabled by default but can be disabled using the SoPreferences
   * environment variable OIV_ENABLE_INMEMORY_TEXT_FILE. If this variable is set to
   * false then the file is always read from disk.
   *
   * End-of-line characters are returned "as is" and may be platform dependent.
   */
  static bool readTextFile(const SbString& filename, SbString& dest);

  /**
   * Cache the contents of a text file in memory.
   * Returns true if the file has been correctly added in the memory cache.
   *
   * The #readTextFile() method will first check if the requested file is in
   * the memory cache created by addInMemoryTextFile, 
   * if so then the cached contents are returned, avoiding any disk access.
   */
  static bool addInMemoryTextFile(const SbString& filename, const SbString& fileContent);

  /** 
   * Remove a file from the in-memory text file cache.
   * Returns true on success.
   *
   * See #addInMemoryTextFile().
   */
  static bool removeInMemoryTextFile(const SbString& filename);

  /** 
   * Returns true if the given filename has been previously added to
   * the in-memory cache by an addInMemoryTextFile() call.
   */
  static bool isInMemoryTextFile(const SbString& filename);

  /**
   * Returns the position in the current file.
   */
  long fileTell();

  /**
   * Sets the position in the file.
   * In compressed streams it may take a long to perform this operation.
   */
  int fileSeek( long offset, int whence );

  /** Returns TRUE if current file can be read by a plugin. */
  bool isReadByPlugin();

  SbBool isReadingHeader();

 protected:

  // Directory search path.
  static void setDirectories(SbStringList *dir);
  // Stack of SoInputFiles (depth >=1)
  SbPList files;
  // Top of stack
  struct SoInputFile *curFile;
  // For strings that are put back
  SbString backBuf;
  // Index into backBuf (-1 if no buf)
  int backBufIndex;

  // Buffer for binary read from file
  void *tmpBuffer;
  // Current location in temporary buffer
  char *curTmpBuf;
  // Size of temporary buffer
  size_t tmpBufSize;

  // Buffer for storing data that
  // has been read but can't be put back.
  char backupBuf[8];
  // True if backupBuf contains data
  SbBool backupBufUsed;
  // Contains char that were put back in the stream in ascii mode.
  std::vector<char> putBackCharBuffer;

  // Looks for named file and opens it. Returns NULL if not found.
  FILE *findFile(const SbString& fileName, SbString &fullName);

  // Initializes reading from file
  void initFile(FILE *newFP, const SbString& fileName,
                SbString *fullName, SbBool openedHere,
                SbDict *refDict = NULL);

  // Initializes streamed reading from file
  void initStreamedFile( FILE* newFP, const SbString& fileName, SbString* fullName, SbBool openedHere,
                         SbDict* refDict = NULL );

#ifdef _WIN32
  FILE *findFromHandle(int nHandle) const;

  void initFromHandle(FILE *newFP,
                      int nHandle, SbBool openedHere,
                      SbDict *refDict = NULL);
#endif

  // Checks current Inventor/VRML file for ASCII/binary header comment.
  // Returns FALSE if no header was found.
  SbBool checkInventorHeader(SbBool bValidateBufferHeader=FALSE);

  // Checks current file for ASCII/binary header comment. Returns
  // FALSE if no header was found.
  SbBool checkHeader(SbBool bValidateBufferHeader=FALSE);

  // Returns TRUE if reading from memory buffer rather than file
  SbBool fromBuffer() const { return (curFile->buffer != NULL); }

  // Skips over white space in input. Pops file if EOF is hit.
  // Returns FALSE on error.
  SbBool skipWhiteSpace();

  // Pops current file from stack, if possible.
  SbBool popFile();

  // Returns number of bytes left in current buffer
  size_t freeBytesInBuf() const
  { return (curFile->bufSize - (curFile->curBuf - static_cast<char *>(curFile->buffer) )); }

  // Reads integer, unsigned integer, or floating-point number.
  // Returns FALSE on EOF or error
  SbBool readInteger(int32_t &l);
  SbBool readUnsignedInteger(uint32_t &l);
  SbBool readInteger(int64_t &l);
  SbBool readUnsignedInteger(uint64_t &l);
  SbBool readReal(double &d);

  // Reads next ASCII format hex value from current file/buffer.
  // Returns FALSE on EOF or error.
  virtual SbBool readHex(uint32_t &l);

  // Reads unsigned integer string into str. Accepts decimal, octal,
  // and hex integers. Returns FALSE on EOF or error
  SbBool readUnsignedIntegerString(char *str);

  // Reads string of decimal or hexadecimal digits into string.
  // Returns number of bytes read.
  int readDigits(char *string);
  int readHexDigits(char *string);

  // Reads given character from buffer into string. Returns 0 or 1
  int readChar(char *string, char charToRead);

  // Make room in the temporary buffer for reading from a binary file
  SbBool makeRoomInBuf(size_t nBytes);

  // Convert datatypes to network format during writing
  void convertShort(char *from, short *s);
  void convertInt32(char *from, int32_t *l);
  void convertInt64(char *from, int64_t *l);
  void convertFloat(char *from, float *f);
  void convertDouble(char *from, double *d);
  void convertShortArray(char *from, short *to, size_t len);
  void convertInt32Array(char *from, int32_t *to, size_t len);
  void convertInt64Array(char *from, int64_t *to, size_t len);
  void convertFloatArray(char *from, float *to, size_t len);
  void convertDoubleArray(char *from, double *to, size_t len);

  SbString bufferToString( const char* buffer );

  /**
   * Reads a buffer from the current file as fread does.
   * The main difference with fread is that this function uses the zlib to read
   * the buffer if the file is compressed.
   */
  size_t fileRead( void* ptr, size_t size, size_t count );

  /**
   * Read a byte from the current file. It uses the zlib module if the file is compressed.
   */
  int fileReadByte();

  static SbBool checkFileCompression( FILE* file, int64_t& fileSize, SbBool& isCompressed );
  static SbBool checkFileCompression( unsigned char byte1, unsigned char byte2 );

  // deprecated
  void convertShortArray(char *from, short *to, int len);
  void convertInt32Array(char *from, int32_t *to, int len);
  void convertFloatArray(char *from, float *to, int len);
  void convertDoubleArray(char *from, double *to, int len);

  int vrmlFile;

//  friend class SoBase;
//  friend class SoDB;

 private:

   SoInputParameters*  m_inputParameters;

  // Directory search path.
  static SbStringList *directories;
  static SbString *defaultString;

  // Callback put in a thread to fill-in the stack of streamed buffers.
  static void* streamingStackCB( void* data );

  void commonInit();

  // Method called when the current streamed buffer is read and needed to be swapped
  void swapStreamBuffer();
  // Methods called for binary files when there's not enough available space in the buffer to read the type
  char* getType( char* from, size_t sizeOfType, size_t freeBytes );
  char* getLongType( char* from, size_t sizeOfType, size_t freeBytes );

  static std::list<SoStreamBuffer*>* m_streamBufStack; // Streamed buffers stack
  // Synchronization between app thread and streaming thread
  static SbThreadSignal* m_availableSig; // Used to signal the app thread that a buffer has been pushed in the stack
  static SbThreadMutex* m_availableSigMutex;
  static SbThreadSignal* m_stackFillSig; // Used to signal the callback thread that a buffer has been popped out of the stack
  static SbThreadMutex* m_stackFillSigMutex;
  static SbThreadMutex* m_stackMutex; // Used when accessing in the stack.
  static SbThreadMutex m_directoriesMutex; // Used when accessing in the directories list.
  SbThread* m_stackThread;

  int64_t m_numReadBytes;

  // Progress bar for debug purposes only!
  int m_percent, m_progress, m_prevProgressVal, m_prevPercentVal;
  double m_result;
  std::string m_progressBar;

  // manage in memory text file for readTextFile method (filename,fileContent)
  typedef std::map<SbString, SbString> InMemoryTextFileMap;
  static InMemoryTextFileMap s_InMemoryTextFileMap;
  static SbThreadSpinlock s_InMemoryTextFileMapMutex;
  static SbBool s_InMemoryTextFileEnabled;

  // manage zlib buffer uncompressing
  SbBool uncompressZData(const void *src, size_t srcLen, void*& dst, size_t& dstLen);
  int inflateZData(const void *src, size_t srcLen, void *dst, size_t dstLen);

};

#if defined(_WIN32)
#pragma warning( pop )
#endif

#endif /* _SO_INPUT_ */

