/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Nick Thompson (MMM yyyy)
** Modified by : Alan Norton (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_PATH_
#define  _SO_PATH_

#include <Inventor/misc/SoBase.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/SoLists.h>
#include <Inventor/lists/NodeIndex.h>
#include <Inventor/STL/iostream>
#include <Inventor/SbEventHandler.h>

class SoTempPath;

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoPath
//
//  A SoPath represents a scene graph or subgraph. It contains
//  pointers to a chain of nodes, each of which is a child of the
//  previous. The child index of each node is also stored, to
//  disambiguate cases where a node appears more than once as a child
//  of the same group.
//
//  The graph defined by the path consists of all of the nodes in the
//  path, all nodes below the last node in the chain, and any other
//  nodes that have an effect on these nodes.
//
//////////////////////////////////////////////////////////////////////////////

class SoPathList;
class SoWriteAction;

/**
* Path that points to a list of hierarchical nodes.
*
* @ingroup General
*
* @DESCRIPTION
*   A path represents a scene graph or subgraph. It contains a list of 
*   nodes forming a chain from some root to some descendant. Each node in the chain
*   is a child of the previous node. Paths are used to refer to some object in a
*   scene graph precisely and unambiguously, even if there are many instances of the
*   object. Therefore, paths are returned by both the SoRayPickAction and
*   SoSearchAction.
*
*   When an action is applied to a path, only the nodes in the subgraph defined by
*   the path are traversed. These include: the nodes in the path chain, all nodes
*   (if any) below the last node in the path, and all nodes whose effects are
*   inherited by any of these nodes.
*
*   SoPath attempts to maintain consistency of paths even when the structure of the
*   scene graph changes. For example, removing a child from its parent when both are
*   in a path chain cuts the path chain at that point, leaving the top part intact.
*   Removing the node to the left of a node in a path adjusts the index for that
*   node. Replacing a child of a node when both the parent and the child are in the
*   chain replaces the child in the chain with the new child, truncating the path
*   below the new child.
*
*   @B Hidden children@b
*
*   Note that only public children of nodes are accessible from an SoPath. Nodes like
*   node kits limit access to their children and these "hidden children" will not be
*   visible using the \if_java "regular.* \endif SoPath methods. 
*
*   For example, if your application does picking
*   and gets a path from SoRayPickAction, the actual last node in the path (tail) will
*   normally be a shape node.  But if that shape is part of a node kit, then the
*   \if_cpp  getTail() \endif \if_dotnet GetTail() \endif \if_java regular.getTail() \endif
*   method will return the @I node kit@i, not the shape node. To access all hidden children
*   in the path, \if_cpp simply cast the path object to SoFullPath before calling get
*   methods. \endif \if_dotnet get the full path with the SoPath.FullPath property before
*   calling get methods. \endif \if_java use the "full.*" methods, for example
*   full.getTail(). \endif To access hidden node kit children (but not all children) in the
*   path, \if_java use the "nodekit.*" methods, for example nodekit.getTail(). \else cast
*   the path object to SoNodeKitPath before calling get methods. \endif @BR
*
*  \if_cpp 
*  @B [C++] Reference counting:@b
*
*  - This is a reference counted object, similar to a node.
*    It can only be destroyed by incrementing and decrementing the reference count.
*    The initial reference count is 0. You may use the ref() and unref() methods or
*    the SoRef "smart pointer" template.
*
*  - SoPath automatically calls ref() when a node is added to the path and unref()
*    when the node is removed from the path or the path object is destroyed. This
*    additional ref() will prevent nodes from being destroyed as long as the path
*    object exists.
*
*  - Actions that return an SoPath, for example SoSearchAction and SoRayPickAction,
*    automatically call ref() when the path is created and unref() when the action
*    is reset or destroyed. The application does not need to explicitly destroy these
*    paths. But be aware that as long as the action is not reset or destroyed, the
*    path is not destroyed and so the nodes referenced by the path cannot be destroyed.
*    See the SoAction method clearApplyResult().
*  \else
*  @B Node References:@b
*
*  An SoPath object maintains a reference to every node in the path. These nodes will
*  not be garbage collected as long as the SoPath exists. Actions that return an
*  SoPath, for example SoSearchAction and SoRayPickAction, maintain a reference to
*  the returned path (or paths). As long as the action exists and is not reset, the
*  nodes referenced by the path cannot be garbage collected.
*  See the SoAction method clearApplyResult().
*  \endif
*
* @FILE_FORMAT_DEFAULT
*    SoPath {
*    @TABLE_FILE_FORMAT
*       @TR [head node]                     @TD
*       @TR [number of remaining indices]   @TD
*       @TR [index]                         @TD
*       @TR &...                            @TD
*       @TR [index]                         @TD
*    @TABLE_END
*    }
*
*   Note that the indices in a written path are adjusted based on the nodes that
*   are actually written to a file. Since nodes in the graph that have no effect on
*   the path (such as some separator nodes) are not written, the siblings of such
*   nodes must undergo index adjustment when written. The actual nodes in the graph
*   remain unchanged.
*
*   SoPath is not able to write "per instance" information to a file
*   for instancing nodes like SoMultipleInstance and SoMultipleCopy.
*   The saved path will reference the whole set of instances under SoMultipleInstance or SoMulitpleCopy node.
*
* @SEE_ALSO
*    SoFullPath,
*    SoNode,
*    SoRayPickAction,
*    SoSearchAction,
*    SoNodeKitPath
*
* [OIVNET-WRAPPER-CLASS DERIVABLE]
*/
class INVENTOR_API SoPath : public SoBase {

 public:

  /**
   * Constructs an empty path.
   */
  SoPath();

  /**
   * Constructs a path with a hint to length (number of nodes in chain).
   */
  SoPath(int approxLength);

  /**
   * Constructs a path and sets the head node to the given node.
   */
  SoPath(SoNode *node);

  /**
   * Sets head node (first node in chain). The head node must be set before the
   * append() or push() methods may be called.
   */
  void setHead(SoNode *node);

  /**
   * Adds node to end of chain; the node is the childIndex'th child of the current
   * tail node.
   */
  void append(int childIndex);

  /**
   * Adds node to end of chain; uses the first occurrence of childNode as child of
   * current tail node. If the path is empty, this is equivalent to
   * @B setHead(childNode) @b.
   */
  void append(SoNode *childNode);

  /**
   * Adds all nodes in fromPath's chain to end of chain; the head node of fromPath
   * must be the same as or a child of the current tail node.
   */
  void append(const SoPath *fromPath);

  /**
   * The push() and pop() methods allow a path to be treated as a stack;
   * they push a node at the end of the chain and pop the last node off.
   */
  void push(int childIndex);

  /**
   * The push() and pop() methods allow a path to be treated as a stack;
   * they push a node at the end of the chain and pop the last node off.
   */
  void pop();

  /**
   * Returns the first node in a path chain.
   */
  SoNode* getHead() const { return (SoNode*)(nodes.get(0)); }

  /**
   * Returns the last node in a path chain.
   */
  SoNode* getTail() const;

  /**
  * Returns the i'th node (within its parent) in the chain.
  * Calling @B getNode(0) @b is equivalent to calling
  * getHead().
  * [OIVNET-WRAPPER-ARG INDEX{0,(FullLength-1)}]
  */
  SoNode* getNode(int i) const;

  /**
  * Returns the first node and its index, from the head of
  * the given type in the chain.
  * NULL is returned and the index is set to -1 if no node of
  * the given type is found.
  */
  SoNode* getNode(const SoType type, int &i) const;


  /**
  * Returns the i'th node (within its parent)
  * in the chain, counting backward from the tail node. Passing 0 for i
  * returns the tail node.
  * [OIVNET-WRAPPER-ARG INDEX{0,(FullLength-1)}]
  */
  SoNode* getNodeFromTail(int i) const;

  /**
   * Returns the index of the i'th node (within its parent) in the chain.
   * [OIVNET-WRAPPER-ARG INDEX{0,(FullLength-1)}]
   */
  int getIndex(int i) const;

   /**
   * Returns the index of the instance inside the parent SoMultipleInstance, SoMultipleCopy or SoArray group.
   * The returned value is -1 for other parent nodes.
   */
  int getInstanceIndex(int i) const;

  /**
   * Returns the index of the i'th node (within
   * its parent) in the chain, counting backward from the tail node. Passing 0 for i
   * returns the index of the tail node.
   * [OIVNET-WRAPPER-ARG INDEX{0,(FullLength-1)}]
   */
  int getIndexFromTail(int i) const;

    /**
   * Returns the index of the i'th node instance (within its parent, 
   * if it is a SoMultipleInstance, SoMultipleCopy or SoArray  group
   * in the chain, counting backward from the tail node. Passing 0 for i
   * returns the instance index of the tail node.
   * [OIVNET-WRAPPER-ARG INDEX{0,(FullLength-1)}]
   */
  int getInstanceIndexFromTail(int i) const;

  /**
   * Returns length of path chain (number of nodes).
   * [OIVNET-WRAPPER PROPERTY{Length},GETTER,SINCE{9.2}]
   */
  int getLength() const;

  /**
   * Truncates the path chain, removing all nodes from index start on. Calling
   * @B truncate(0) @b empties the path entirely.
   * [OIVNET-WRAPPER-ARG INDEX{0,(FullLength-1)}]
   */
  void truncate(int start);

  /**
   * Returns TRUE if the node is found anywhere in the path chain.
   */
  SbBool containsNode(const SoNode *node) const;

  /**
   *
   * Returns TRUE if the node type is found anywhere in the path chain.
   */
  SbBool containsNode(const SoType type) const;

  /**
   * Returns TRUE if the nodes in the chain in the passed path are contained (in
   * consecutive order) in this path chain.
   */
  SbBool containsPath(const SoPath *path) const;

  /**
   * If the two paths have different head nodes, this returns -1. Otherwise, it
   * returns the path chain index of the last node (starting at the head) that is the
   * same for both paths.
   */
  int findFork(const SoPath *path) const;

  /**
   * Creates and returns a new path that is a copy of some or all of this path.
   * Copying starts at the given index (default is 0, which is the head node). A
   * numNodes of 0 (the default) means copy all nodes from the starting index to the
   * end. Returns NULL on error.
   * [OIVNET-WRAPPER-ARG INDEX{0,(FullLength-1)},INDEX{0,(FullLength-startFromNodeIndex)}]
   */
  SoPath *copy(int startFromNodeIndex = 0, int numNodes = 0) const;

  /**
  * Returns TRUE if all nodes in the two path chains are identical.
  */
  INVENTOR_API friend int operator ==(const SoPath &p1, const SoPath &p2);

  /**
   * @brief Not equal operator.
   * @param p1 First path to compare.
   * @param p2 Second path to compare.
   * @return TRUE if the two paths are not equal.
   */
  INVENTOR_API friend int operator !=(const SoPath &p1, const SoPath &p2);

  /**
  * Writes the path to the specified output stream.
  */
  INVENTOR_API friend std::ostream& operator << (std::ostream& os, const SoPath& path);

  /**
   * Returns type identifier for path instance.
   */
  virtual SoType getTypeId() const;

  /**
   * Returns type identifier for SoPath class.
   */
  static SoType getClassTypeId();

  /**
   * Method to return a path with a given name. Paths are named by
   * calling their setName() method (defined by the SoBase class). This method
   * returns the last path that was given the specified name, either by setName()
   * or by reading in a named path from a file. If there is no path with the given
   * name, NULL will be returned.
   */

  static SoPath *getByName(const SbName &name);
  /**
   * Method to return paths with a given name. Paths are named by
   * calling their setName() method (defined by the SoBase class).
   * This method appends all paths with the given
   * name to the given path list and returns the number of paths that were added. If
   * there are no paths with the given name, zero will be returned and nothing will
   * be added to the list.
   */
  static int getByName(const SbName &name, SoPathList &list);


 SoINTERNAL public:
  // Initializes path class
  static void initClass();
  static void exitClass();

  /** Event handler called each time the path change (add/replace/remove index)) */
  SbEventHandler<SoPath*> m_onPathChange;

  /**
  * Returns TRUE is p1 strictly less than p2: to provide a total order operation for SoPath
  */
  friend bool operator <(const SoPath &p1, const SoPath &p2);

  // This is called when a node in the path chain has a child added.
  // The passed index is the index of the new child
  void insertIndex(SoNode *parent, int newIndex);

  // This is called when a node in the path chain has a child removed.
  // The passed index is the index of the child to be removed
  void removeIndex(SoNode *parent, int oldIndex);

  // This is called when a node in the path chain replaces a child.
  // The passed index is the index of the child to be removed
  void replaceIndex(SoNode *parent, int index, SoNode *newChild);

  // Writes path using given write action
  virtual void write(SoWriteAction *writeAction) const;

  // Returns TRUE if the given notification list involves a change
  // to a node that affects the path. It is assumed that the last
  // (most recent) node in the list is the head node of the path.
  SbBool isRelevantNotification(SoNotList *list) const;

  void setNumPublic(int num);

  // Appends the given node and index to the lists, updating
  // numPublic as appropriate
  void append(SoNode *node, int index);

  // Appends the given node and index to the lists, updating
  // numPublic as appropriate
  void appendMI(SoNode *node, int index, int instanceIndex);

  // like copy but let user copy path in a temp path (if temp == true)
  SoPath *copy_(int startFromNodeIndex = 0, int numNodes = 0, bool createTempPath = false) const;

  // Returns this path hash value
  inline const size_t& getHash() const
  {
    if ( m_pathHash != static_cast< size_t >(-1) )
      return m_pathHash; // No need to update... just return value.
    return computeHash();
  }

  // Really does a truncate. Flag indicates whether to notify.
  void truncate(int start, SbBool doNotify);

  // Get nodes in a unsecured way: Only returns the found value without
  // checking out-of-array access.
  SoNode* u_getHead() const              { return (SoNode*)(const_cast<SoNodeList*>(&nodes)->getArray()[0]); }
  SoNode* u_getTail() const              { return (SoNode*)(const_cast<SoNodeList*>(&nodes)->getArray()[getLength() - 1]); }
  SoNode* u_getNode(int i) const         { return (SoNode*)(const_cast<SoNodeList*>(&nodes)->getArray()[i]); }
  SoNode* u_getNodeFromTail(int i) const { return (SoNode*)(const_cast<SoNodeList*>(&nodes)->getArray()[getLength() - 1 - i]); }
  int u_getIndex(int i) const            { return (int)indices[i].index; }
  int u_getIndexFromTail(int i) const    { return (int)indices[getLength() - 1 - i].index; }
  int u_getInstanceIndex(int i) const    { return (int)indices[i].instanceIndex; }

  // Multiple instance API see SoMultipleInstance, SoMultipleCopy, SoArray

  /**
  * Sets head node (first node in chain) and its instance index. The head node must be set before the
  * appendMI() or pushMI() methods may be called. 
  */
  void setHeadMI(SoNode *node, int instanceIndex);

  /**
  * Adds node to end of chain; the node is the childIndex'th child of the current
  * tail node having the instanceIndex if the tail node is a SoMultipleInstance, SoMultipleCopy 
  * or SoArray group and -1 otherwise
  */
  void appendMI(int childIndex, int instanceIndex);

  /**
  * Adds node to end of chain; uses the first occurrence of childNode as child of
  * current tail node and if it is  is a SoMultipleInstance or a SoMulipleCopy its instanceIndex.
  * If the path is empty, this is equivalent to
  * @B setHead(childNode) @b.
  */
  void appendMI(SoNode *childNode, int instanceIndex);

  /**
  * The pushMI() and popMI() methods allow a path to be treated as a stack;
  * they push a node with its instanceIndex (if sensible) at the end of the chain and pop the last node off.
  */
  void pushMI(int childIndex, int instanceIndex);

  bool isForwardTraversing() const;

 protected:
  // Allows internal SoTempPath subclass to forego auditor overhead
  void auditPath(SbBool flag) { doAuditors = flag; }

  // Reads stuff into instance of SoPath. Returns FALSE on error.
  virtual SbBool readInstance(SoInput *in, unsigned short flags);

  virtual ~SoPath();

 private:
   const size_t& computeHash() const;

  SoNodeList nodes;          // Pointers to nodes
  std::vector<inventor::NodeIndex> indices;      // Child indices
  mutable int numPublic;             // How many children are public
  mutable int minNumPublic;          // Minimum we KNOW are public
  SbBool doAuditors;         // TRUE if auditors to be maintained
  static SoType classTypeId; // TypeId of paths

  // Returns real length of path, including hidden children
  int getFullLength() const
    { return nodes.getLength(); }

  // Returns new SoPath; called by SoType::createInstance()
  static void* createInstance(SoType* dynamicType = NULL);

  // This path hash value
  mutable size_t m_pathHash;

  friend class SoFullPath;
  friend class SoTempPath;
  friend class SoAction;
  friend class SoPathNoRef;
  friend class SoTraversalPassImpl;

};

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoFullPath
//
//  This SoEXTENDER class allows expert users to examine "hidden"
//  children in paths. SoPath allows access from the head node down to
//  the first node with hidden children, but no further. Casting an
//  SoPath to an SoFullPath (which is always guaranteed to be safe)
//  allows you to get at hidden children. SoFullPath overrides some of
//  the methods defined on paths to make this possible.
//
//////////////////////////////////////////////////////////////////////////////

/**
* Path that allows access to hidden children.
*
* @ingroup General
*
* @DESCRIPTION
*   This class allows expert users to examine "hidden" children in paths.
*   For example children inside a nodeKit.  SoPath allows access from the head node
*   down to the first node with hidden children, but no further. \if_cpp Casting an SoPath
*   to an SoFullPath (which is always guaranteed to be safe) \else Using SoPath.FullPath
*   property \endif allows you to access hidden children.
*   SoFullPath overrides some SoPath methods to make this possible.
*
*   You need SoFullPath, for example, to get the actual tail of the pick
*  path (the geometry node) when picking geometry in node kits.
*  For example:
*     \par
*     \if_dotnet
*     \code
*       SoPath path = ...
*       SoNode tail = path.FullPath.GetTail();
*     \endcode
*     \else
*     \code
*       SoPath* pPath = ...
*       SoNode* pTail = ((SoFullPath*)pPath)->getTail();
*     \endcode
*     \endif
*
*   @B NOTE:@b
*   Applying an action on an SoFullPath may not give the expected result. @BR
*   When applying an action on an SoFullPath, the "hidden" children will @I not@i be traversed.
*   If you want to apply an action on all the children in the path, use the following technique:
*   \if_dotnet
*     \code
*     SoPath path = ...
*     SoPath tempPath = path.FullPath.Copy();
*     action.Apply( tempPath );
*     \endcode
*   \else
*     \code
*       SoPath* path = ...
*     SoRef<SoPath> tempPath = ((SoFullPath*)path)->copy();
*       action.apply( tempPath );
*     tempPath = NULL;
*     \endcode
*   \endif
*
* @FILE_FORMAT_DEFAULT
*    SoPath {
*    @TABLE_FILE_FORMAT
*       @TR [head node]                     @TD
*       @TR [number of remaining indices]   @TD
*       @TR [index]                         @TD
*       @TR &...                            @TD
*       @TR [index]                         @TD
*    @TABLE_END
*    }
*
* @SEE_ALSO
*    SoPath
* [OIVJAVA-WRAPPER-CLASS NO_WRAP]
*/
class INVENTOR_API SoFullPath : public SoPath {

 public:
  //
  // Override methods that normally stop at last public child:
  //
  /** \copydoc SoPath::pop() */
  void pop()
    { truncate(getFullLength() - 1); }

  /** \copydoc SoPath::getTail() */
  SoNode *getTail() const
    { return (nodes[getFullLength() - 1]); }

  /** \copydoc SoPath::getNodeFromTail() */
  SoNode *getNodeFromTail(int i) const
    { return (nodes[getFullLength() - 1 - i]); }

  /** \copydoc SoPath::getIndexFromTail() */
  int getIndexFromTail(int i) const
  {
    return indices[getFullLength() - 1 - i].index;
  }

  /** get instance Index from tail */
  int getInstanceIndexFromTail(int i) const
  {
    return indices[getFullLength() - 1 - i].instanceIndex;
  }

  /** \copydoc SoPath::getLength() */
  int getLength() const
    { return getFullLength(); }

 private:
  SoFullPath(int approxLength) : SoPath(approxLength) {}
  virtual ~SoFullPath();

  friend class SoTempPath;
};

#ifndef HIDDEN_FROM_DOC

//////////////////////////////////////////////////////////////////////////////
//
//  SoINTERNAL Class: SoLightPath
//
//  A SoLightPath is a light-weight version of an SoTempPath, intended to
//  be used just to keep track of the current path during traversal.
//  Eventually it is intended to replace SoTempPath (when appropriate
//  modifications are made to SoDraggers).
//  Unlike SoPath, it keeps only a chain of childIndices and a headnode.
//  The methods implemented are only those needed for traversal.
//
//////////////////////////////////////////////////////////////////////////////

/** [OIV-WRAPPER-CLASS NO_WRAP] */
class INVENTOR_API SoLightPath {

 SoINTERNAL public:

  // Constructor given approximate number of nodes in chain
  SoLightPath(int approxLength=0);

  SoLightPath(const SoLightPath& other);

  // Sets head node (first node in chain)
  // Resulting path has only one node.
  void                setHead(SoNode *node);

  // Adds node specified by child index to end of chain.
  void                append(int childIndex)
  { indices.push_back(inventor::NodeIndex(childIndex, inventor::NodeIndex::NODE));}

  // Adds node specified by child index along with its instanceIndex to end of chain.
  void                appendMI(int childIndex, int instanceIndex)
  { indices.push_back(inventor::NodeIndex(childIndex, instanceIndex)); }


  // Allows path to be treated as a stack: push a node at the end of
  // the chain and pop the last node off
  void                push(int childIndex)    { append(childIndex); }
  void                pushMI(int childIndex, int instanceIndex)    { appendMI(childIndex, instanceIndex); }
  void                push()                  { append(-1);}
  void                pop()           { truncate(getFullLength() - 1); }

  void setTail(int childIndex)
  {
    setTail(childIndex, inventor::NodeIndex::NODE);
  }

  void setTail(int childIndex, int instanceIndex)
  {
    setTail(inventor::NodeIndex(childIndex, instanceIndex));
  }

  void setTail(const inventor::NodeIndex& index)
  {
    size_t tailPos = (size_t)(getFullLength()-1);
    if ( tailPos >= indices.size() )
      indices.resize(tailPos+1);
    indices[tailPos] = index;
  }

  SoNode *            getTail(){
    return getNode(getFullLength()-1);
  }

  // Returns the first node in a path chain.
  SoNode *            getHead() const { return headNode; }

  // Returns pointer to ith node in chain
  SoNode *            getNode(int i) const;

  // BA - added getIndex
  // Returns index of ith node in chain
  int                 getIndex(int i) const
  {    
    return indices[i].index;
  }

  // Returns index of ith node in chain
  int                 getInstanceIndex(int i) const
  { return indices[i].instanceIndex; }


  // Returns full length of path chain (number of nodes)
  // note that public/private distinction is ignored.
  int                 getFullLength() const {return (int) indices.size();}

  // Removes all nodes from indexed node on
  void                truncate(int start)
  {
    indices.resize(start > 0 ? start : 0);
  }


  // fills in nodes for a TempPath that is represented by this
  // SoLightPath.  Called by SoAction::getCurPath();
  void                makeTempPath(SoTempPath *) const;

  SoLightPath& operator=(const SoLightPath& other);

 private:

  ~SoLightPath();
  SoNode *            headNode;       // Pointer to headnode
  mutable std::vector<inventor::NodeIndex> indices;        // Child indices


  friend class SoAction;
};

#endif // HIDDEN_FROM_DOC

/**
* Writes the path to the specified output stream.
*/
INVENTOR_API std::ostream& operator << (std::ostream& os, const SoPath& path);

#endif /* _SO_PATH_ */

