/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef _SO_PRIMITIVE_VERTEX_
#define _SO_PRIMITIVE_VERTEX_

#include <Inventor/SbLinear.h>

class SoDetail;
class SoPointDetail;
class SoTextureCoordinateBundle;

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoPrimitiveVertex
//
//  An SoPrimitiveVertex represents a vertex of a primitive (triangle,
//  line segment, point) that is being generated via the
//  SoCallbackAction.
//
//  An SoPrimitiveVertex contains an object-space point, normal,
//  texture coordinates, material index, and a pointer to an instance
//  of some SoDetail subclass. This detail may contain more
//  information about the vertex, or may be a NULL pointer if there is
//  no such info.
//
//  Instances of SoPrimitiveVertex are typically created on the stack
//  by shape classes while they are generating primitives. Anyone who
//  wants to save them as return values from SoCallbackAction should
//  probably make copies of them.
//
//////////////////////////////////////////////////////////////////////////////

/**
* Represents a vertex of a generated primitive.
* 
* @ingroup General
* 
* @DESCRIPTION
*   An SoPrimitiveVertex represents a vertex of a primitive (triangle, line segment,
*   or point) that is being generated by an SoCallbackAction. It contains an
*   object-space point, normal, texture coordinates, material index, and refers
*   an instance of an SoDetail subclass. This detail may contain more information
*   about the vertex, or may be NULL if there is no such info.
*   
*   Instances of SoPrimitiveVertex are typically created on the stack by shape
*   classes while they are generating primitives. Anyone who wants to save them as
*   return values from SoCallbackAction should probably make copies of them.
* 
* 
*/
class INVENTOR_API SoPrimitiveVertex {

 public:
  /**
   * Default constructor.
   */
  SoPrimitiveVertex();

  /**
  *\if_cpp
  * Constructor. Note that copying a primitive vertex copies the
  * detail pointer, and not the detail itself.
  *\else
  * Constructor. Note that copying a primitive vertex copies the
  * detail reference, and not the detail itself.
  *\endif
  */
  SoPrimitiveVertex(const SoPrimitiveVertex &pv);

  /**
   * Destructor.
   */
  ~SoPrimitiveVertex();

  /**
   * Returns the surface point in object space.
   */
  const SbVec3f &     getPoint() const                { return point; }
  /**
   * Returns the normal in object space.
   */
  const SbVec3f &     getNormal() const               { return normal; }
  /**
   * Returns the texture coordinates in object space.
   */
  const SbVec4f &     getTextureCoords() const;

  /**
   * Returns the index into the current set of materials of the material active at
   * the vertex.
   */
  int                 getMaterialIndex() const    { return materialIndex; }

  /**
  * Returns the point detail giving more information about the vertex. Note that
  * the return value is NULL if there is no more info.
  */
  const SoPointDetail *    getPointDetail() const               { return m_pointDetail; }

  /**
  * Returns the detail giving more information about the vertex. Note that the
  * return value is NULL if there is no more info.
  */
  const SoDetail *    getDetail() const               { return detail; }

  /**
  *\if_cpp
  * Copies the given vertex. Note that just the pointer to the detail is copied, and
  * not the detail itself.
  *\else
  * Copies the given vertex. Note that just the reference to the detail is copied, and
  * not the detail itself.
  *\endif
  */
  SoPrimitiveVertex & operator =(const SoPrimitiveVertex &pv);

 SoEXTENDER public:

  // These methods are typically called by shape classes during
  // primtiive generation

  // These set the object space point, normal, and texture coordinates:
  void                setPoint(const SbVec3f &pt)       { point     = pt; }
  void                setNormal(const SbVec3f &norm)    { normal    = norm; }
  void                setTextureCoords( const SbVec4f &t ){ m_texCoords = t; }
  void                setTextureCoords( const SoTextureCoordinateBundle  *tcb, int curCoord );

  // Sets the material index. The index is set to 0 during construction.
  void                setMaterialIndex(int index)  { materialIndex = index; }

  // Sets the detail corresponding to the vertex. The pointer may be
  // NULL, although it is set to NULL during construction.
  void                setDetail(SoDetail *d)            { detail = d; }

  // Sets the point detail corresponding to the vertex. The pointer may be NULL.
  void                setPointDetail(SoPointDetail *p)            { m_pointDetail= p; }

 private:
  SbVec3f                           point;            // Object-space point
  SbVec3f                           normal;           // Object-space normal
  int                               materialIndex;    // Material index
  SoDetail*                         detail;           // Extra detail info
  SoPointDetail*                    m_pointDetail;    //Point detail info 

  // Please see note in implementation 
  mutable const SoTextureCoordinateBundle*  m_tcb;
  mutable int                               m_currCoord;
  mutable SbVec4f                           m_texCoords;  // Object-space texture coordinates

};

#endif /* _SO_PRIMITIVE_VERTEX_ */

