/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Author(s) (MMM yyyy)
** Modified by : David Mott (MMM yyyy)
** Modified by : Alain Dumesny (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef _SO_SCENE_MANAGER_
#define _SO_SCENE_MANAGER_

#include <Inventor/SbColor.h>
#include <Inventor/SbColorRGBA.h>
#include <Inventor/SbViewportRegion.h>
#include <Inventor/SoType.h>
#include <Inventor/antialiasing/SoAntialiasingParameters.h>
#include <Inventor/helpers/SbGlContextHelper.h>
#include <Inventor/helpers/SbGlContextHelper.h>
#include <Inventor/STL/vector>
#include <Inventor/STL/set>
#include <Inventor/threads/SbThreadSpinlock.h>
#include <SoDeprecationRules.h>

class SiAntialiasingEventListener;
class SoNodeSensor;
class SoEvent;
class SoAction;
class SoState;
class SoHandleEventAction;
class SoGLRenderAction;
class SoNode;
class SoSceneManager;
class SoSceneManagerImpl;
class SoSensor;
class SoSFTime;
class SoOneShotSensor;
class SoGLRenderFromPBuffer;
class SoGLContext;
class SoSceneManagerSync;

/*! \cond PRIVATE */
namespace inventor
{
  namespace impl
  {
    class SoRenderAreaCoreImpl;
  }
}
/*! \endcond */

using inventor::impl::SoRenderAreaCoreImpl;

// callback function prototypes
/**
 * This typedef defines the calling sequence for setRenderCallback() callbacks.
 *
 * @memberof SoSceneManager
 *
 * [OIV-WRAPPER NAME{RenderCB}]
 */
typedef void SoSceneManagerRenderCB(void *userData, SoSceneManager *mgr);

/** 
 * This typedef defines the calling sequence for setAbortRenderCallback() callbacks.
 *
 * @memberof SoSceneManager
 *
 * [OIV-WRAPPER NAME{AbortRenderCallbacks}]
 */
typedef SbBool SoAbortRenderCB(SoAction*,void *userData);

#if defined(_WIN32)
#pragma warning( push )
#pragma warning( disable: 4251 ) // 'identifier' : class 'type' needs to have dll-interface to be used by clients of class 'type2'
#endif

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoSceneManager
//
//  Class which performs Inventor rendering and event handling
//  on a scene graph.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Manages scene graph rendering and event handling.
 *
 * @ingroup General
 *
 * @DESCRIPTION
 *   SoSceneManager provides Open Inventor rendering and event handling inside a
 *   window provided by the caller. The scene manager is able to render in only a
 *   portion of a window if desired. The SoWinRenderArea class employs an
 *   SoSceneManager, and handles most all the details for setting up a window,
 *   converting Windows messages to Open Inventor events, automatically redrawing the
 *   scene when necessary, and so on. It is simplest to use a render area when
 *   rendering in an entire window. The SoSceneManager class is available for
 *   programmers not using the SoXt / SoWin or SoQt libraries.
 *
 * @SEE_ALSO
 *    SoWinRenderArea,
 *    SoGLRenderAction,
 *    SoHandleEventAction
 *
 *
 */
class INVENTOR_API SoSceneManager
{
 public:

#if SoDEPRECATED_BEGIN(2024200)
  /** Constructor. */
  SoDEPRECATED_METHOD(2024.2, "This method should no longer be used. Use constructor SoSceneManager() (without argument) instead.")
  SoSceneManager(int nb);
#endif /** @DEPRECATED_END */

  /**
   * Constructor.
   */
  SoSceneManager();

  /** Destructor. */
  virtual ~SoSceneManager();

  /**
   * Applies an SoGLRenderAction to the scene graph managed here.
   * Note that this method just applies an SoGLRenderAction that traverses the
   * scene graph and makes the necessary rendering calls for each node.
   * It is not the same as calling the render method on an Open Inventor render
   * area or viewer object.
   * The viewer's render method will typically do pre-rendering operations like
   * adjusting the near/far clip planes, as well as post-rendering operations like
   * calling "swap buffers" to make the rendered content visible.
   * Also it is possible to call a viewer's render method without explicitly
   * making an OpenGL render context current (the viewer takes care of that).
   *
   * When calling this method, the application is responsible for providing a current
   * OpenGL render context that is known to Open Inventor through an SoGLContext object.
   * If there is no current OpenGL render context, Open Inventor may throw an exception
   * or crash, depending on the API language and system environment.
   *   - If the application is using an Open Inventor render area or viewer and wants to
   *     render in that window, there may not be an OpenGL render context current. However the
   *     viewer's render context can be made current by calling the viewer's bindNormalContext()
   *     method.  Call the viewer's unbindNormalContext() method after calling render().
   *     \if_cpp
   *     \par
   *     \code
   *     viewer->bindNormalContext();
   *     viewer->getSceneManager()->render();
   *     viewer->unbindNormalContext();
   *     \endcode
   *     \endif
   *
   *   - This method is more likely to be used in cases where there is no Open Inventor render
   *     area or viewer. For example when integrating Open Inventor rendering in an OpenGL application.
   *     If the application has its own OpenGL render context that is already current,
   *     you can call the static SoGLContext method getCurrent(true) to get an SoGLContext
   *     object that wraps the current context. Then call bind and unbind on the returned object.
   *     (Note that this method will return null if there is no current context.) 
   *     \if_cpp
   *     \par
   *     \code
   *     SoGLContext* ctx = SoGLContext::getCurrent( true );
   *     ctx->bind();
   *     viewer->getSceneManager()->render();
   *     ctx->unbind();
   *     \endcode
   *     \endif
   *
   * \if_dotnet
   * @dotnet_exception{System.InvalidOperationException,No OpenGL context active found.}
   * [OIVNET-WRAPPER HELPER_BEGIN{CheckGLContext();}]
   * \endif
   */
  virtual void render();
  
#if SoDEPRECATED_BEGIN(10100)
  
  /**
   * @see render().
   */
  SoDEPRECATED_METHOD(10100,"No longer used and the parameters are ignored. Use render() instead.")
  virtual void render(SbBool clearWindow, SbBool clearZbuffer = TRUE
#ifndef HIDDEN_FROM_DOC // internal undocumented parameters.
    , int id = 0, bool force = false
#endif
    );

#endif /** @DEPRECATED_END */


  /**
   * Processes the passed event by applying an SoHandleEventAction to the scene graph
   * managed here. Returns TRUE if the event was handled by a node.
   */
  virtual SbBool processEvent(const SoEvent *event);

  /**
   * Reinitializes graphics. This should be called, for instance, when there is a new
   * window.
   */
  void reinitialize();

  /**
   * Schedules a redraw for some time in the near future. If there is no render
   * \if_dotnet delegate \else callback \endif set, or this is not active, no redraw will be scheduled.
   */
  void scheduleRedraw();

  /**
   * Defines the scene graph which is managed here. This is the Open Inventor
   * scene which will be traversed for rendering and event processing.
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetSceneGraph(newScene)}]
   */
  virtual void setSceneGraph(SoNode *newScene);

  /**
   * @see setSceneGraph().
   */
  virtual SoNode* getSceneGraph() const;

  /**
   * Defines the size of the window in which the scene manager should render.
   * This size must be set before render() and processEvent() are
   * called.
   */
  void setWindowSize(const SbVec2s &newSize, const float &newScale = 1.0);

  /**
   * @see setWindowSize().
   */
  const SbVec2s& getWindowSize() const;

  /**
   * Defines the size of the viewport within the window. Default is to
   * render the entire window region.
   */
  void setSize(const SbVec2s &newSize, const float &newScale = 1.0);

  /**
   * @see setSize().
   */
  const SbVec2s& getSize() const;
  
  /**
   * Defines the origin of the viewport within the window.
   * The origin (0,0) is the lower left corner of
   * the window.
   */
  void setOrigin(const SbVec2s &newOrigin);

  /**
   * @see setOrigin().
   */
  const SbVec2s& getOrigin() const;

  /**
   * Defines current viewport region to use for rendering.
   * This can be used instead of setting the size and origin separately.
   */
  void setViewportRegion(const SbViewportRegion &newRegion);

  /**
   * @see setViewportRegion().
   */
  const SbViewportRegion &getViewportRegion() const;

  /**
   * Defines the window background color when in RGB mode. This is the color the
   * scene manager viewport is cleared to when render() is called.
   * Default is black (0,0,0).
   * See also #setBackgroundColorRGBA().
   *
   * Setting the background color will automatically call the scheduleRedraw()
   * method.
   *
   * The default value can be set using the environment variable
   * OIV_BACKGROUND_COLOR. Specify three floats (R, G, B) in the range 0. to 1.,
   * separated by spaces.
   */
  void setBackgroundColor(const SbColor &c);

  /**
   * @see setBackgroundColor().
   */
  SbColor getBackgroundColor() const;

  /**
   * Defines the window background color when in RGBA mode. This is the color the
   * scene manager viewport is cleared to when render() is called.
   * Default is transparent black (0,0,0,0).
   *
   * The default RGB color (but NOT alpha) can be set using the environment variable
   * OIV_BACKGROUND_COLOR or by calling setBackgroundColor().
   *
   * @param color RGBA background color
   */
  void setBackgroundColorRGBA(const SbColorRGBA &color);

  /**
   * @see setBackgroundColorRGBA().
   */
  SbColorRGBA getBackgroundColorRGBA() const;

  /**
   * Defines the window background color when in color index mode. This is the
   * color the scene manager viewport is cleared to when render() is called.
   * Default is black (index 0).
   */
  void setBackgroundIndex(int index);

  /**
   * @see setBackgroundIndex().
   */
  int getBackgroundIndex() const;

  /**
   * Defines the color mode (TRUE - RGB mode, FALSE - color map mode). Default is
   * RGB mode. Only a subset of Open Inventor nodes will render correctly in color
   * map mode. Basically, when in color index mode, lighting should be turned off
   * (the @B model @b field of SoLightModel should be set to @B BASE_COLOR @b), and
   * the SoColorIndex node should be used to specify colors.
   */
  void setRGBMode(SbBool onOrOff);

  /**
   * @see setRGBMode().
   */
  SbBool isRGBMode() const;

  /**
   * Defines the depth value used when the depth buffer is cleared. The default value is 1.
   * This is the value used to clear the depth buffer when render() is called.
   *
   * @param depth value used to clear the depth buffer. Value is clamped to the range [0,1].
   */
  void setClearDepth(float depth);

  /**
   * @see setClearDepth().
   */
  float getClearDepth() const;

  /**
   * Activates the scene manager. The scene manager will only employ
   * sensors for automatic redraw while it is active. Typically, the scene manager
   * should be activated whenever its window is visible on the screen, and
   * deactivated when its window is closed or iconified.
   */
  virtual void activate();

  /**
   * Deactivates the scene manager. The scene manager will only employ
   * sensors for automatic redraw while it is active. Typically, the scene manager
   * should be activated whenever its window is visible on the screen, and
   * deactivated when its window is closed or iconified.
   */
  virtual void deactivate();

  /** 
   * Defines the auto interactive mode. Default is FALSE.
   *
   * When this mode is activated, the sceneManager will decide depending on
   * scenegraph changes to switch to interactive mode or not.
   *
   * Default value can be changed through OIV_AUTO_INTERACTIVE_MODE envvar.
   *
   * @M_SINCE 9.2
   */
  void setAutoInteractiveMode(SbBool flag);

  /** 
   * @see setAutoInteractiveMode().
   *
   * @M_SINCE 9.2
   */
  SbBool isAutoInteractiveMode() const;
  
  /**
   * Indicates that the scene manager is in interactive mode or not.
   * This is usually called by Inventor viewer classes, but it is also usefull
   * for custom application viewer.
   *
   * It mainly setup SoInteractionElement for all used actions (preRenderAction and renderAction).
   *
   * @M_SINCE 9.2
   */
  void setInteractive(SbBool flag);  
  
  /**
   * The render callback provides a mechanism for automatically redrawing the scene
   * in response to changes in the scene graph. The scene manager employs a sensor to
   * detect scene graph changes. When the sensor is triggered, the render callback
   * registered here is invoked.  If the callback is set to NULL
   * (the default), auto-redraw is turned off.
   *
   * If the application is not using an Open Inventor render area or viewer, the
   * callback should make sure that an OpenGL render context is current, then call the
   * scene manager render() method.  See the render() method for more information.
   *
   * If the application is using an Open Inventor render area or viewer, then the
   * callback should call render() method on the render area or viewer object.
   *
   * The callback should not modify any nodes in the scene graph before or after
   * calling the render() method.  The modification will be detected by the scene
   * manager, which will schedule another call to the render callback, resulting
   * in continuous calls to the render callback.  If necessary the callback can
   * temporarily disable notification by calling enableNotify() on the node that
   * will be modified.
   */
  void setRenderCallback( SoSceneManagerRenderCB *f, void *userData = NULL );

  /**
   * Returns @B TRUE @b if there is currently a render \if_dotnet delegate \else callback \endif registered.
   */
  SbBool isAutoRedraw() const;

  /**
   * Sets the priority of the redraw sensor. Sensors are processed based on
   * priority, with priority values of 0 processed immediately. The default priority
   * for the scene manager redraw sensor is 10000.
   */
  void setRedrawPriority(uint32_t priority);

  /**
   * @see setRedrawPriority().
   */
  uint32_t getRedrawPriority() const;

  /**
   * Gets the default priority of the redraw sensor.
   */
  static uint32_t getDefaultRedrawPriority() { return 10000; }

  /**
   * Enables the realTime global field update which normally happen right
   * after a redraw.
   */
  static void enableRealTimeUpdate(SbBool flag);

  /**
   * @see enableRealTimeUpdate().
   */
  static SbBool isRealTimeUpdateEnabled();

  /**
   * Sets the OpenGL context to be shared by the scene manager.
   * This avoids the necessity to re-generate textures and display lists if
   * they are already available in another OpenGL context
   * (another viewer context, for instance).
   */
  void setShareContext(SbGLShareContext sc, SbBool issc = TRUE);

  /** 
   * Enum which indicates the desired antialiasing algorithm.
   * This is used by the antialiasing API.
   * @see setAntialiasing()
   */
  enum AntialiasingMode
  {
    /** This is the default mode for antialiasing.
     *  The different algorithms are tested in the following order and the first available is used:
     *  SMAA, FSAA, SUPERSAMPLING.
     */
    AUTO,

    /**
     * FXAA is a screen space antialiasing algorithm which uses shaders and is applied after the rendering.
     * The main advantages of this technique are reduced memory footprint and cost.
     */
    FXAA,

    /**
     * SMAA is a screen space antialiasing algorithm which uses shaders and is applied after the rendering.
     * The main advantages of this technique are reduced memory footprint and cost with better quality result
     * than FXAA algorithm.
     */
    SMAA,

    /**
     * FSAA is a full-screen antialiasing algorithm based on super-sampling.
     * The main advantage is the quality of the antialiasing but the memory and speed cost can be huge.
     * @BR Note: Use the SoFullSceneAntialiasing node to control FSAA during render traversal.
     * @BR @B Limitations:@b @BR
     *   - Transparency: @BR
     *     This mode is not compatible with SORTED_PIXEL transparency mode (see SoGLRenderAction::TransparencyType).
     *     Use SUPERSAMPLING, FXAA or SMAA instead if you need high quality transparency.
     **/
    FSAA,

    /**
     * The advantage of this algorithm is it gives high quality images even on lines.
     * However multiple rendering passes are required, which reduces performance.
     * (Old name was ACCUM_BUFFERS.)
     */
    SUPERSAMPLING,

     /**
     * Deprecated name of SUPERSAMPLING.
     */
    SoDEPRECATED_ENUM_VALUE(9500,"It corresponds to the SUPERSAMPLING antialiasing type.")
    ACCUM_BUFFERS = SUPERSAMPLING,

#ifndef HIDDEN_FROM_DOC
    NUM_ANTIALIASING_MODE,
#endif

    /**
     * Turn off antialiasing.
     */
    NO_ANTIALIASING
  };
  
  /**
   * Set options for supersampling when "still" (not interacting).
   * When quality is greater than 0, still images will be automatically supersampled.
   *
   * @param quality The quality is a factor in the range [0.0,1.0]. @BR
   *        Use the value 0.0 to turn off still supersampling. 0.5 is a typical value.
   * @param delay The delay is in seconds. @BR
   *        If greater than 0, images will be supersampled after the specified delay
   */
  void setStillSuperSampling(float quality, float delay);

  /** 
   * @see setStillSuperSampling().
   */
  float getStillSuperSamplingQuality();

  /**
   * @see setStillSuperSampling().
   */
  float getStillSuperSamplingDelay();

  /** 
   * Enable (or disable) antialiasing with specified quality and mode.
   *
   * Specific antialiasing parameters will be set automatically based on
   * the quality value. Note that the quality and mode settings are overridden if 
   * specific antialiasing parameters are subsequently set using the 
   * setAntialiasing(SoAntialiasingParameters*) method.
   *
   * The default mode is AUTO but this may be overridden by setting the environment
   * variable OIV_ANTIALIASING_DEFAULT_MODE (see SoPreferences).
   *
   * @param quality The quality is a factor in the range [0.0,1.0]. @BR
   *        Use the value 0.0 to turn off antialiasing. 0.5 is a typical value.
   * @param mode The antialiasing algorithm. Default is AUTO, which means use the best for the current hardware. @BR
   *        Use the value NO_ANTIALIASING to turn off antialiasing.
   */
  void setAntialiasing(const float quality, const AntialiasingMode mode = AUTO);

  /**
   * Enable (or disable) antialiasing with specific parameters.
   *
   * Use one of the subclasses of SoAntialiasingParameters.
   * The antialiasing mode is determined by which subclass is used to set the parameters. 
   * For example, passing an SoFXAAParameters object automatically sets FXAA mode. Note
   * that the parameters are overridden if a quality and mode are subsequently set using the
   * setAntialiasing(float,AntialiasingMode) method.
   *
   * @NOTES
   *  - When the antialiasing parameters are modified a listener is called if it has been defined.
   *    See also setAntialiasingEventListener().
   *  @if_cpp
   *  - The caller is responsible for allocation and destruction of the advancedParameters object.
   *    SoSceneManager makes a copy of the parameter values, so the object can be destroyed after
   *    calling this method.
   *  @endif
   *
   * @param advancedParameters Provides specific parameters for an antialiasing mode. @BR
   *        Use a null parameter to turn off antialiasing or use one of the subclasses
   *        of SoAntialiasingParameters. 
   */
  void setAntialiasing(SoAntialiasingParameters* advancedParameters);

  /** 
   * Returns the antialiasing quality set using the setAntialiasing(float,AntialiasingMode) method.
   * Returns 0.0 by default.  Parameters set using the setAntialiasing(SoAntialiasingParameters*) method
   * override internal parameters, but do not affect the value returned by this method. 
   * Therefore this method does not necessarily return the current actual antialiasing quality.
   */
  float getAntialiasingQuality() const;

  /** 
   * Returns the antialiasing parameters set using the setAntialiasing(SoAntialiasingParameters*) method.
   * Returns null by default.  A quality value set using the setAntialiasing(float,AntialiasingMode) 
   * method may modify internal parameters, but does not affect the value returned by this 
   * method. Therefore this method does not necessarily
   * return the current actual antialiasing parameters.
   */
  SoAntialiasingParameters* getAntialiasingParameters() const;

  /** 
   * Returns the antialiasing mode set using the setAntialiasing(float,AntialiasingMode) method.
   * Returns AUTO by default. Parameters set using the setAntialiasing(SoAntialiasingParameters*) method
   * may change the antialiasing mode, but do not affect the value returned by this method. 
   * Therefore this method does not necessarily
   * return the current actual antialiasing mode.
   */
  AntialiasingMode getAntialiasingMode() const;

  /** 
   * Sets an event listener which is called when the antialiasing configuration is modified.
   *
   * It is useful to define this listener because the scene manager is not responsible for the pixelformat
   * changes required for the FSAA and accumulation algorithms. The Open Inventor viewer classes use the 
   * listener to automatically switch the pixel format (not necessary for application to handle this when
   * using a viewer class).
   * 
   * @param listener The listener object.
   */
  void setAntialiasingEventListener(SiAntialiasingEventListener* listener);

  /** 
   * @see setAntialiasingEventListener().
   *
   * @return A pointer on the current antialiasing event listener, NULL if there is no listener.
   */
  SiAntialiasingEventListener* getAntialiasingEventListener() const;

  /**
   * User supplied handle event action. This should not be done in
   * the middle of event handling. Passing NULL turns event handling off.
   * SceneManager will never delete a handle event action passed to
   * this method.
   *
   * @M_SINCE 9.2
   */
  void setHandleEventAction(SoHandleEventAction *hea);

  /**
   * @see setHandleEventAction().
   *
   * @M_SINCE 9.2
   */
  SoHandleEventAction *getHandleEventAction() const;

  /**
   * User supplied render action. Highlights fall into this category.
   * SceneManager will never delete a render action passed to this method.
   * return the renderAction 0.
   *
   * @M_SINCE 9.2
   */
  void setGLRenderAction(SoGLRenderAction *ra);

  /**
   * @see setGLRenderAction().
   *
   * @M_SINCE 9.2
   */
  SoGLRenderAction* getGLRenderAction() const;

  /** 
   * Setup a callback that returns TRUE if rendering should be aborted.
   * It allows some Open Inventor nodes to stop their work, if requested,
   * in order to keep reasonable interactivity.
   *
   * When using Open Inventor standard GUI classes, the callback is setup
   * by default to return TRUE if MousePress events are pending during a STILL frame.
   *
   * @M_SINCE 9.5
   */
  void setAbortRenderCallback( SoAbortRenderCB *callback, void *userData = NULL );
  
#if SoDEPRECATED_BEGIN(9100)

  /**
   * Enables smoothing and/or multi-pass antialiasing for rendering. @BR
   *
   * There are two kinds of antialiasing
   * available: smoothing and multipass antialiasing. If smoothing is set to TRUE,
   * smoothing is enabled. Smoothing uses OpenGL's line- and point-smoothing features
   * to provide cheap antialiasing of lines and points. The value of numPasses
   * controls multipass antialiasing. Each time a render action is applied, Open
   * Inventor renders the scene numPasses times from slightly different camera
   * positions, averaging the results. numPasses can be from one to 255, inclusive.
   * Setting numPasses to one disables multipass antialiasing. You can use either,
   * both, or neither of these antialiasing techniques. By default, both smoothing
   * and multipass antialiasing are disabled.
   *
   */
  SoDEPRECATED_METHOD(9100,"To enable smoothing and/or multi-pass antialiasing for rendering use the setAntialiasing(float,AntialiasingMode) method with mode SUPERSAMPLING or use the setAntialiasing(SoAntialiasingParameters*) method with an SoAccumulationAntialiasingParameters object.")
  void setAntialiasing(SbBool smoothing, int numPasses);

  /**
   * @see setAntialiasing().
   */
  SoDEPRECATED_METHOD(9100,"Use method getAntialiasingParameters() to get this information.")
  void getAntialiasing(SbBool &smoothing, int &numPasses) const;
  
#endif /** @DEPRECATED_END */

SoEXTENDER public:

  void updateRealTimeSensor();

  //Multi GLWidget case
  void setGLRenderAction(SoGLRenderAction *ra, int i);
  SoGLRenderAction* getGLRenderAction(int i) const;

SoINTERNAL public:

  /**
   * Returns TRUE rendering should be aborted.
   *
   * If application use a custom viewer implementation ( not based on SoXt,SoWin,SoQt)
   * the application has to implement this features and setup the correct callback through
   * setAbortRenderCallback.
   *
   * @M_SINCE 9.2
   */
  SbBool shouldAbortRender(SoAction* action) const;

  enum FloatColorBufferSize
  {
    /**
     * 16-bit rendering per component.
     */
    FLOAT_16_COLOR_BUFFER = 0,

    /**
     * 32-bit rendering per component.
     */
    FLOAT_32_COLOR_BUFFER = 1
  };

  /**
   * Enables/disables floating point rendering using 16- or 32-bit components.
   * If TRUE, Open Inventor will automatically render to a floating point color
   * buffer.
   *
   * To determine if floating point rendering was successfully enabled,
   * use #getFloatingColorBuffer.
   *
   * Using floating point rendering can improve image quality, particularly when
   * many objects are being blended.
   */
  void setFloatingColorBuffer(SbBool enable, FloatColorBufferSize size = FLOAT_16_COLOR_BUFFER);

  /**
   * Returns TRUE if floating point rendering is used and its precision.
   */
  void getFloatingColorBuffer(SbBool& enable, FloatColorBufferSize& size);

  bool needToSetViewport( const int id = 0 );

  void setNeedToSetViewport( const int id = 0 );


  /** Ask all SceneManagers containing given node to redraw. */
  static void scheduleRedraw(SoNode* node);

  /** Schedule a forced redraw, which is guaranteed to be done */
  void scheduleForcedRedraw();
  
  SoSceneManagerImpl* getImpl() const
  {
    return m_impl;
  }

  /**
   * @copydoc SoSceneManagerImpl::canGPUPick(SoState* state)
   */
  bool canGPUPick(SoState* state);

  enum Components
  {
    RGB,
    RGBA,
  };

  bool saveSnapshot(const SbString& filename, Components components = RGBA) const;

private:

  SoSceneManagerImpl* m_impl;

  friend class SoGuiRenderArea;
  friend class SoRenderAreaCoreImpl;

};

#if defined(_WIN32)
#pragma warning( pop )
#endif

#endif // _SO_SCENE_MANAGER_

