/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Nick Thompson (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef _SO_TYPE_
#define _SO_TYPE_

#include <Inventor/SbString.h>
#include <Inventor/SbDict.h>
#include <Inventor/STL/ostream>

class SoTypeList;
struct SoTypeData;
class ScDataInputStream;

typedef void *SoTypeFromNameCB(SbName name);

#ifndef HIDDEN_FROM_DOC
//Int32 value of a badtype
#define SO_BAD_TYPE_INT_INTERNAL 0x80000000
#endif

/**
 * Stores runtime type information.
 *
 * @ingroup General
 *
 * @DESCRIPTION
 * The SoType class keeps track of runtime type information in Open Inventor. Each
 * type is associated with a given name, so lookup is possible in either
 * direction.
 *
 * Many Open Inventor classes request a unique SoType when they are initialized.
 * This type can then be used to find out the actual class of an instance when only
 * its base class is known, or to obtain an instance of a particular class given
 * its type or name.
 *
 * Note that the names associated with types of Open Inventor classes do not contain
 * the "So" prefix.
 *
 * @SEE_ALSO
 * SoAction,
 * SoBase,
 * SoDetail,
 * SoError,
 * SoEvent,
 * SoField
 *
 *
 * [OIV-WRAPPER-NO-WRAP]
 */
class INVENTORBASE_API SoType {

#ifdef __DELTA
#  pragma __nondynamic_class
#endif

 public:

  /**
   * Returns the type associated with the given name.
   */
  static SoType fromName(const SbName &name);

  /**
   * Returns the name associated with a type.
   */
  SbName getName() const;

  /**
   * Returns the type of the parent class.
   */
  SoType getParent() const;

  /**
   * Returns an always-illegal type. Useful for returning errors.
   */
  inline static SoType badType()
  {
    return (SoType)SO_BAD_TYPE_INT_INTERNAL;
  }

  /**
   * Returns TRUE if the type is a bad type.
   */
  SbBool isBad() const;

  /**
   * Returns TRUE if the type is derived from type @B t@b.
   */
  SbBool isDerivedFrom(const SoType &t) const;

  /**
   * Returns TRUE if the type is derived from the type of class @B TypedObjectClass@b.
   *
   * Note: TypedObjectClass must be derived from SoTypedObject
   */
  template<typename TypedObjectClass>
  SbBool isDerivedFrom() const
  {
    return isDerivedFrom( TypedObjectClass::getClassTypeId() );
  }

  /**
   * Adds all types derived from the given type to the given type list. Returns the
   * number of types added.
   */
  static int getAllDerivedFrom(const SoType &type, SoTypeList &list);

  /**
   * Some types are able to create instances; for example, most nodes and engines
   * (those which are not abstract classes) can be created this way. This method
   * returns TRUE if the type supports such creation.
   */
  SbBool canCreateInstance() const;

  // Internal note: Only "if_cpp" is needed here.
  //                .NET and Java API do not use this class.

  /**
  * Creates and returns a pointer to an instance of the type. Returns NULL if an
  * instance could not be created for some reason. The pointer is returned as a
  * generic pointer, but can be cast to the appropriate type.
  * \if_cpp
  * For example:
  *  \code
  *   SoCube *c = (SoCube *) SoCube::getClassTypeId().createInstance();
  *  \endcode
  *  is a convoluted way of creating a new instance of an SoCube.
  * \endif
  */
  void *createInstance(SoType *overrideType=NULL) const;

  /**
   * Returns TRUE if this type is the same as the given type.
   */
  bool operator ==(const SoType t) const;

  /**
   * Returns TRUE if this type is not the same as the given type.
   */
  bool operator !=(const SoType t) const;

  /**
   * Less-than comparison operator that can be used to sort types.
   * This is pretty useless otherwise.
   */
  bool operator <(const SoType t) const;

  /**
   * Writes the SoType to the specified output stream.
   */
  INVENTORBASE_API friend std::ostream& operator << (std::ostream& os, const SoType& t)
  {
    return os << "<" << t.getName() << "," << t.getKey() << ">";
  }

 SoEXTENDER public:

  // set a callback that will be called if unknown node are found
  static SoTypeFromNameCB *setUnknownNameCB(SoTypeFromNameCB *func);

  // Create a new type
  static SoType createType(const SoType &parent, const SbName &name,
                           void * (*createMethod)(SoType *) = NULL,
                           short data = 0);

  // Make an new type act like an existing type. The new type MUST
  // be a C++ subclass of the original (e.g. MyCubeClass must be
  // derived from SoCube), but there is no way for us to check that.
  // This can be used to get the database to create a different
  // subclass whenever it reads in an SoNode class from a file.
  static SoType overrideType(const SoType &existingType,
                             void * (*createMethod)(SoType *) = NULL);

  static SbBool removeType( const SbName &name );

 SoINTERNAL public:

  // Constructor.
  SoType() { storage.intValue = SO_BAD_TYPE_INT_INTERNAL; }

  SoType(unsigned int storageValue) { storage.intValue = storageValue; }

  // Initialize the type system
  static void init();
  static void finish();

  // get max key value
  static int maxKey() { return nextIndex; }

  // Get data
  short getData() const;

  // Returns the type key as a short
  short getKey() const;

  // Mark this type as internal; if internal, getAllDerivedFrom and
  // fromName will not return the type.
  void makeInternal();
  bool isInternal() const;

  // Get the number of types currently registed in the types dictionary.
  // This is used by SoAction when setting up the action method list.
  static int getNumTypes();

  // Get the name of the types currently registered with the key 'key'.
  static SbName getTypeName(const short key);

  inline unsigned int getStorageValue() const { return storage.intValue; }

  /**
   * Returns true if the type is derived from the type whose name is given.
   * This method is meant to avoid calling #fromName which can be inefficient
   * in some cases.
   */
  bool isDerivedFrom( const SbName& name ) const;

  /**
   * Returns the type associated with the given name.
   *
   * This method does not try to load new types from ext sources.
   * It only looks in the currently loaded types.
   *
   * Reserved for internal use.
   * Prefer using #fromName() in your application.
   */
  static SoType getType(const SbName& name);

 private:

  // SoTypes are passed around on the stack a lot, and are cast to
  // void *'s; they MUST fit into a single word.
  union
  {
    struct {
      unsigned int data : 16;
      unsigned int index : 15; // Assumes we have fewer than 32,768 types
      unsigned int isPublic :1; // 0 if is an internal class
    } storageStruct;
    unsigned int intValue;
  } storage;

  // name->sotype dictionary
  static SbDict *nameDict;

  // array of SoTypeData
  static int nextIndex;
  static int arraySize;
  static SoTypeData *typeData;

  static void expandTypeData();
  static SoType fromNameExt(const SbName &name, bool extent);

  static SoTypeFromNameCB *unknownNameFunc;
  static void *unknownNameData;
};

#include <Inventor/lists/SoTypeList.h>

#endif /* _SO_TYPE_ */
