/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Jerome Lague (Jan 2010)
**=======================================================================*/


#ifndef _SO_TYPED_OBJECT_
#define _SO_TYPED_OBJECT_

#include <Inventor/SoType.h>

/**
 * Base class for object storing runtime type information.
 *
 * @ingroup General
 *
 * @DESCRIPTION
 * The SoTypedObject class keeps track of runtime type information in Open Inventor
 * using SoType.
 *
 * @SEE_ALSO
 * SoType,
 * SoAction,
 * SoBase,
 * SoDetail,
 * SoError,
 * SoEvent,
 * SoField
 * [OIV-WRAPPER-NO-WRAP]
 */
class INVENTORBASE_API SoTypedObject
{
public:
  /**
   * Returns the type identifier for a specific instance.
   */
  virtual SoType getTypeId() const = 0; 

  // Internal note: Only "if_cpp" is needed here.
  //                .NET and Java API do not use this class.

  /**
   * Returns TRUE if this object is of the type specified in type or is derived from
   * that type. Otherwise, it returns FALSE.
   * \if_cpp
   * For example,
   * \code 
   *  objPtr->isOfType(SoGroup::getClassTypeId())
   * \endcode
   *  returns TRUE if objPtr is an instance of SoGroup or one of its subclasses.
   * \endif
   * [OIV-WRAPPER-NOT-WRAP]
   */
  SbBool isOfType(const SoType &type) const
  { return getTypeId().isDerivedFrom(type); }

  /**
   * Returns TRUE if this object is of the type of class TypedObjectClass
   * or is derived from that class. Otherwise, it returns FALSE.
   *
   * Note: TypedObjectClass must be derived from SoTypedObject
   *
   * \if_cpp
   * For example,
   * \code
   *  objPtr->isOfType<SoGroup>()
   * \endcode
   *  returns TRUE if objPtr is an instance of SoGroup or one of its subclasses.
   * \endif
   * [OIV-WRAPPER-NOT-WRAP]
   */
  template<typename TypedObjectClass>
  SbBool isOfType() const
  {
    return isOfType( TypedObjectClass::getClassTypeId() );
  }

  /** Returns the type identifier for this class. */
  static SoType getClassTypeId()
  { return s_classTypeId; }

SoINTERNAL public:

  static void initClass();
  static void exitClass();

  static void checkDatabase( const char* className, void* thisPtr, SoType classType, SoType& classTypeId );

  /**
   * Returns TRUE if this object is of the type specified by name or is derived from
   * that type. Otherwise, it returns FALSE.
   * \if_cpp
   * For example,
   * \code
   *  objPtr->isOfType("Group")
   * \endcode
   *  returns TRUE if objPtr is an instance of SoGroup or one of its subclasses.
   * \endif
   * [OIV-WRAPPER-NOT-WRAP]
   */
  SbBool isOfType(const SbName& name) const
  {
    return getTypeId().isDerivedFrom(name);
  }

protected:
  SoTypedObject();
  virtual ~SoTypedObject();

private:
  static SoType s_classTypeId;
  static bool s_isFirstTime;
  static bool s_checkDatabase;
};

#endif /* _SO_TYPED_OBJECT_ */


