/*=======================================================================
** VSG_COPYRIGHT_TAG
**=======================================================================*/
#ifndef RenderAreaInteractive_H
#define RenderAreaInteractive_H

#include <Inventor/ViewerComponents/MFC/RenderArea.h>
#include <Inventor/ViewerComponents/MFC/MFCEventToSoEvent.h>
#include <Inventor/ViewerComponents/SiRenderAreaInteractive.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>

class SceneInteractor;
class SoGroup;

/**
 * Class to render an OpenInventor scene in an MFC OpenGL window.
 * This class extend RenderArea to add mouse and keyboard interactions.
 *
 * @ingroup ViewerComponentsMFC
 */
class MFCVIEWERCOMPONENTS_API RenderAreaInteractive : public RenderArea, public SiRenderAreaInteractive
{

public:

  /**
   * Camera clipping planes adjustment mode.
   * When adjustment mode is set to AUTO, the camera near and far planes
   * are dynamically adjusted to be as tight as possible (least amount of stuff
   * is clipped) before each render traversal.
   * When adjustment mode is set to MANUAL, the user is expected to
   * manually set the camera near and far planes.
   */
  enum ClippingPlanesAdjustMode
  {
    AUTO,
    MANUAL
  };

  /**
   * Constructor.
   */
  RenderAreaInteractive();

  /**
   * Destructor.
   */
  virtual ~RenderAreaInteractive();

  /**
   * This method is called in order to create OpenGL context.
   */
  virtual void initializeGL( HDC deviceContext, bool enableStereo = false );

  /**
   * Function to set the attached scene graph
   */
  virtual void setSceneGraph( SoNode* sceneGraph );

  /**
   * Return the root of the scene graph.
   */
  SoDEPRECATED_METHOD( 10500, "Use getSceneInteractor() method instead." )
  SceneInteractor* getRootSceneGraph();

  /**
   * Return the root of the scene graph.
   */
  virtual SceneInteractor* getSceneInteractor() const;

  /**
   * Set the camera clipping planes adjustment mode.
   * When adjustment mode is set to AUTO, the camera near and far planes
   * are dynamically adjusted to be as tight as possible (least amount of stuff
   * is clipped) before each render traversal.
   * When adjustment mode is set to MANUAL, the user is expected to
   * manually set those planes. Updating clipping planes after a camera move is
   * not enough, if a dragger or a rendered shape is moved, they can disappear
   * or become partially clipped.
   * Default is AUTO.
   */
  void setClippingPlanesAdjustMode(ClippingPlanesAdjustMode mode);

  /**
   * Get the camera clipping planes adjustment mode.
   */
  ClippingPlanesAdjustMode getClippingPlanesAdjustMode();

  /**
   * Set the camera in order to see the whole scene
   */
  void viewAll( const SbViewportRegion &viewport );

  /**
   * Moves the camera to be aligned with the given direction vector while
   * keeping the "up" direction of the camera parallel to the specified up
   * vector.
   */
  virtual void viewAxis( const SbVec3f& direction, const SbVec3f& up );

  /**
   * Activate/Deactivate stereo.
   */
  virtual void activateStereo( bool activated );

  /**
   * Returns true if stereo can be activated.
   */
  bool isStereoSupported();

  /**
   * Set the stereo offset.
   */
  void setStereoCameraOffset( float offset );

  /**
   * Set the stereo balance.
   */
  void setStereoCameraBalance( float balance );

#if SoDEPRECATED_BEGIN(101000)
  /**
   * Sets the current interactive mode.
   */
  SoDEPRECATED_METHOD( 101000, "Use SoInteractiveComplexity node instead." )
  virtual void setInteractiveMode(SoInteractiveComplexity::InteractiveMode mode);

  /**
   * Gets the current interactive mode.
   */
  SoDEPRECATED_METHOD( 101000, "Use SoInteractiveComplexity node instead." )
  virtual SoInteractiveComplexity::InteractiveMode getInteractiveMode() const;
#endif /** @DEPRECATED_END */

  /**
   * Processes the passed event to the scene graph managed here.
   * Returns TRUE if the event was handled by a node.
   *
   * @NOTES
   *   Events can only be processed once rendering area is initialized.
   */
  virtual SbBool processEvent(const SoEvent *event);

  /**
   * Processes the passed event to the scene graph managed here.
   * Returns TRUE if at least one event was handled by a node.
   *
   * @NOTES
   *   Events can only be processed once rendering area is initialized.
   */
  SbBool processEvents(const std::vector<const SoEvent*>& eventList);

  // Events treatment
  afx_msg void OnLButtonDown(UINT flags, CPoint loc);
  afx_msg void OnMButtonDown(UINT flags, CPoint loc);
  afx_msg void OnRButtonDown(UINT flags, CPoint loc);
  afx_msg void OnLButtonUp(UINT flags, CPoint loc);
  afx_msg void OnMButtonUp(UINT flags, CPoint loc);
  afx_msg void OnRButtonUp(UINT flags, CPoint loc);
  afx_msg void OnLButtonDblClk(UINT flags, CPoint loc);
  afx_msg void OnMButtonDblClk(UINT flags, CPoint loc);
  afx_msg void OnRButtonDblClk(UINT flags, CPoint loc);

  afx_msg void OnMouseMove(UINT flags, CPoint loc);
  afx_msg BOOL OnMouseWheel(UINT flags, short zDelta, CPoint pt);
  afx_msg void OnMouseHover(UINT nFlags, CPoint point);
  afx_msg void OnMouseLeave();
  afx_msg BOOL PreTranslateMessage(MSG* pMsg);

protected :

  RenderAreaInteractive( bool buildRootSceneGraph );

  /**
   * Build internal scene graph.
   */
  void buildSceneGraph();

  /**
   * Render the scene graph.
   */
  virtual SoRenderAreaCore::RenderStatus render();

  /**
   * This method is called by MFC when a key is pressed.
   */
  virtual void keyPressEvent( WPARAM param );

  /**
   * This method is called by MFC when a key is released.
   */
  virtual void keyReleaseEvent( WPARAM param );

  /**
   * This method is called by MFC when a mouse button is pressed.
   *
   * This function also manage the interactive mode.
   * We switch on interactive (FORCE_INTERACTION) mode when we press a mouse button,
   * and switch back to AUTO mode when we release the button.
   */
  virtual void mousePressEvent( CPoint loc, SoMouseButtonEvent::Button button );

  /**
   * This method is called by MFC when a mouse button is released.
   */
  virtual void mouseReleaseEvent( CPoint loc, SoMouseButtonEvent::Button button );

  /**
   * This method is called by MFC when a mouse button is double clicked.
   */
  virtual void mouseDoubleClickEvent( CPoint loc, SoMouseButtonEvent::Button button );

  /**
   * Specify is the mouse tracking is installed
   */
  bool m_mouseTracking;

  SceneInteractor* m_rootSceneGraph;
  // Auto interactive mode
  bool m_isAutoInteractive;

private:

  void init( bool buildRootSceneGraph );

  void startRender( SiRenderArea::RenderEventArg& arg );

  SoGroup* m_appSceneGraph;
  ClippingPlanesAdjustMode m_clippingMode;
  SoInteractiveComplexity::InteractiveMode m_interactiveMode;

  /**
  * This method is called to begin capture mouse events outside the current window.
  */
  void setCaptureOnButtonDown();

  /**
  * This method is called to end capture mouse events outside the current window.
  */
  void releaseCaptureOnButtonUp();
  int m_countDownButtons;

  DECLARE_MESSAGE_MAP()
};

#endif // RenderAreaInteractive_H
