/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#ifndef SI_RENDERAREA
#define SI_RENDERAREA

#include <Inventor/nodes/SoNode.h>
#include <Inventor/SbColorRGBA.h>
#include <Inventor/SbVec.h>
#include <Inventor/SbEventArg.h>
#include <Inventor/SbEventHandler.h>

/**
* @VSGEXT RenderArea interface.
*
* @ingroup ViewerComponents
*
* @DESCRIPTION
*
* This interface provides the list of methods that a renderArea classs must define.
*
* @SEE_ALSO
*  RenderArea
*
*/
class INVENTOR_API SiRenderArea
{

public:

  /**
   * Defines the scene graph which is managed here. This is the Open Inventor
   * scene which will be traversed for rendering and event processing.
   *
   * @param newScene scene graph.
   */
  virtual void setSceneGraph(SoNode *newScene) = 0;

  /**
   * @see setSceneGraph().
   */
  virtual SoNode* getSceneGraph() const = 0;

  /**
   * Policy values to clear color and depth buffers beforing rendering.
   */
  enum ClearPolicy
  {
    /** Do not clear both color buffer and depth buffer beforing rendering. */
    NONE,
    /** Clear only color buffer beforing rendering. */
    COLORBUFFER,
    /** Clear only depth buffer beforing rendering. */
    DEPTHBUFFER,
    /** Clear both color buffer and depth buffer beforing rendering. */
    COLORBUFFER_AND_DEPTHBUFFER
  };

  /**
   * Defines the color buffer and depth buffer clear policy.
   * See also setClearColor and setClearDepth.
   *
   * @param policy color buffer and depth buffer clear policy.
   */
  virtual void setClearPolicy(ClearPolicy policy) = 0;

  /**
   * @see setClearPolicy().
   */
  virtual ClearPolicy getClearPolicy() const = 0;

  /**
   * Defines the RGBA value used when the color buffer is cleared.
   * See also setClearPolicy.
   *
   * @param color RGBA value used to clear the color buffer.
   */
  virtual void setClearColor(const SbColorRGBA& color) = 0;

  /**
   * @see setClearColor().
   */
  virtual SbColorRGBA getClearColor() const = 0;

  /**
   * Defines the depth value used when the depth buffer is cleared.
   * See also setClearPolicy.
   *
   * @param depth value used to clear the depth buffer.
   */
  virtual void setClearDepth(float depth) = 0;

  /**
   * @see setClearDepth().
   */
  virtual float getClearDepth() const = 0;

  /**
   * Defines the size to use for rendering.
   */
  virtual void setSize(const SbVec2i32& size) = 0;

  /**
   * @see setSize().
   */
  virtual SbVec2i32 getSize() const = 0;

  /**
  * Returned by the render method
  */
  enum RenderStatus
  {
    /** The frame rendering has been ended before its completion */
    ABORTED,
    /** The frame has been rendered in interactive mode 
     * (some shapes may have been rendered with low quality settings) 
     */
    INTERACTIVE,
    /**
     * The frame has been rendered in still mode
     */
    STILL
  };

  /**
   * Render the managed scene graph (see setSceneGraph).
   * See RenderStatus documentation for details on the returned value.
   */ 
  virtual RenderStatus render() = 0;

  /**
   * Structure given when a start render event is raised.
   */
  struct RenderEventArg : public SbEventArg
  {
    public:
      RenderEventArg(SiRenderArea* source) :
        m_source(source)
      {}

      SiRenderArea* getSource() const
      {
        return m_source;
      }
    private:
      SiRenderArea* m_source;
  };

  /**
   * Returns the event handler that raises when a new render starts.
   */
  virtual SbEventHandler<RenderEventArg&>& onStartRender() = 0;
};

#endif // SI_RENDERAREA
