/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#ifndef SO_CAMERA_INTERACTOR
#define SO_CAMERA_INTERACTOR

#include <Inventor/SbBox.h>
#include <Inventor/projectors/SbSphereSheetProjector.h>
#include <Inventor/misc/SoRefCounter.h>
#include <Inventor/STL/stack>
#include <Inventor/SbVec.h>
#include <Inventor/SbPImpl.h>

class SoCamera;
class SbViewportRegion;
class SbViewVolume;
class SoCameraInteractorImpl;

/**
* @VSGEXT Utility class to manipulate a camera.
*
* @ingroup ViewerComponents
*
* @DESCRIPTION
*   This class provides some useful algorithms to manipulate a camera
*   (translation, rotation, zoom, etc).
*
*  This class is not, strictly speaking, a true interactor. It does not respond
*  to input events. However it provides the camera manipulation algorithms you
*  need to build an interactor that provides behavior similar to the Open Inventor
*  viewer classes, for example SoWinExaminerViewer.  An interactor could
*  handle system-dependent events or Open Inventor events (using SoEventCallback).
*  It could also be triggered from the user interface (buttons, sliders, etc).
*
*  Start by creating an instance of SoCameraInteractor by giving the camera to be
*  manipulated either to the constructor of SoCameraInteractor or to the
*  #getNewInstance() method.
*  Then, for example, to provide examinerViewer-like rotation behavior:
*   - On a mouse button down event, call #activateOrbiting().
*   - On mouse move event, if button is down, call #orbit().
*
*  To provide the viewer's automatic adjustment of near and far clip planes behavior,
*  call the #adjustClippingPlanes() method before each render traversal.
*
*  To see an example "viewer" implemented using this class, please see the
*  SceneExaminer class provided as source code in the Open Inventor SDK:
*  \par
*  \if_cpp    .../source/Inventor/ViewerComponents/nodes \endif
*  \if_dotnet .../source/OIV.Inventor.ViewerComponents/Nodes \endif
*  \if_java   .../source/com/openinventor/inventor/viewercomponents/nodes \endif
*
*  \if_cpp 
*  [C++] Reference counting: @BR
*  \par
*  This is a reference counted object, similar to a node or path.
*  It can only be destroyed by incrementing and decrementing the reference count.
*  The initial reference count is 0. You may use the ref() and unref() methods,
*  but we recommend using the SoRef "smart pointer" template.
*  An SoCameraInteractor instance increments and decrements the camera's reference
*  count when created and destroyed.
*  \endif
*
* @EXAMPLE
 * \if_cpp
 *   Create interactor and an event callback node that handles mouse events
 *   \code
 *   SoPerspectiveCamera* camera = new SoPerspectiveCamera();
 *   SoCameraInteractor*  m_interactor = SoCameraInteractor::getNewInstance( camera );
 *
 *   SoEventCallback* eventCB = new SoEventCallback;
 *   eventCB->addEventCallback( SoMouseButtonEvent::getClassTypeId(), mouseEventCB );
 *   eventCB->addEventCallback( SoLocation2Event::getClassTypeId(), mouseEventCB );
 *   \endcode
 *   Handler for mouse events that implements "orbit" camera manipulation:
 *   \code
 *   void mouseEventCB( void* userData, SoEventCallback* node ) 
 *   {
 *     static m_buttonDown = false;
 *     const SoEvent* evt = node->getEvent();
 *     if (SO_MOUSE_PRESS_EVENT( evt, BUTTON1 )) {
 *       buttonDown = true;
 *       SbVec2f position = evt->getNormalizedPosition( node->getAction()->getViewportRegion() );
 *       m_interactor->activateOrbiting( position );
 *       m_interactor->setRotationCenter( m_interactor->getFocalPoint() );
 *     }
 *     else if (SO_MOUSE_RELEASE_EVENT( evt, BUTTON1 )) {
 *       m_buttonDown = false;
 *     }
 *     else if (evt->isOfType(SoLocation2Event::getClassTypeId())) {
 *       if (m_buttonDown) {
 *         SbVec2f position = evt->getNormalizedPosition( node->getAction()->getViewportRegion() );
 *         m_interactor->orbit( position );
 *       }
 *     }
 *   }
 *   \endcode
 * \endif
 * 
 *  \htmlonly </UL> \endhtmlonly
 *
 * @SEE_ALSO
 *   SceneInteractor, SceneExaminer
 */
class INVENTOR_API SoCameraInteractor : public SoRefCounter
{
public:

  /**
  * Create a SoCameraInteractor for the specified camera.
  * The camera should be an SoPerspectiveCamera or SoOrthographicCamera object.
  * Create an empty object if camera pointer is NULL or invalid
  */
  SoCameraInteractor( SoCamera* camera );

  /**
  * Create an instance for the specified camera.
  * The camera should be an SoPerspectiveCamera or SoOrthographicCamera object.
  * Returns null if the camera parameter is null.
  */
  static SoCameraInteractor* getNewInstance( SoCamera* camera );

  /**
  * Zoom - Makes the scene appear larger or smaller.
  * Sets the camera's @I heightAngle@i field (for a perspective camera) or
  * @I height@i field (for an orthographic camera) to the specified value.
  * For a perspective camera, the value is in radians and the default is 0.785398.
  * For an orthographic camera the value is in World Coordinates and the default is 2.
  *
  * Conceptually, zoom means approximately the same thing as it does for a real
  * camera - changing the "focal length" of the lens.  Making the heightAngle (or
  * height) larger corresponds to using a wide-angle lens, so the scene appears
  * smaller. Making the heightAngle (or height) smaller corresponds to using a
  * telephoto lens, so the scene appears larger.
  *
  * NOTE: For a perspective camera, we strongly recommend implementing the visual
  * effect of "zoom" by moving the camera closer to, or farther away from, the scene
  * (see #dolly()). Very small and very large values of heightAngle can produce
  * undesirable results.
  *
  * @param zoom new zoom value
  */
  virtual void zoom( float zoom );

  /**
  * Dolly - Makes the scene appear larger or smaller.
  * For a perspective camera, moves the camera forward or backward along the view
  * vector (details below). For an orthographic camera, changes the @I height@i
  * field to give a similar visual effect (moving an orthographic camera forward
  * and backward has no visual effect).  This is the preferred way to implement the
  * visual effect of "zooming" in and out for a perspective camera. For a perspective
  * camera, dolly means the same thing it means for a camera on a movie set.
  *
  * Note: 'scaleFactor' is the inverse of what you might expect. To make the scene
  * appear 2X larger, specify a scaleFactor of 0.5.  This is because scaleFactor
  * multiplies the camera 'focalDistance' or 'height'.
  *
  * - Perspective camera: The camera has a "focal point" located @I focalDistance@i units
  *   from the camera @I position@i along the view direction vector (@I orientation@i).
  *   Unlike a real camera, neither focal point nor focal distance has effect on the
  *   appearance of the image. The focal point is just a reference point for some
  *   algorithms. You can think of it as the "look at" point. The dolly() method
  *   computes a scale factor from the 'delta' parameter. A new "focal distance" is
  *   computed as the current value of the camera's focalDistance field multiplied by this
  *   scale factor. Then the camera's @I position@i field is modified so the camera is the
  *   new focal distance away from the focal point.
  *
  * - Orthographic camera: The dolly() method computes a scale factor from the 'delta'
  *   parameter and multiplies the camera's @I height@i field by this scale factor.
  */
  virtual void dolly( float scaleFactor );

  /**
  * Change camera's view volume.
  * Allows changing the view volume height and width with a single call.
  * For a perspective camera, the height parameter is @I heightAngle@i in radians.
  * For an orthographic camera, the height parameter is @I height@i in world coordinates.
  *
  * @param height new camera height or heightAngle
  * @param aspect new camera aspect ratio
  */
  virtual void changeViewVolume( float height, float aspect );

  /**
  * Returns the associated camera.
  */
  SoCamera* getCamera() const;

#if SoDEPRECATED_BEGIN(9500)

  /**
  * Returns the viewport center.
  * On other words, returns the camera's "focal point", located
  * @I focalDistance@i units from the camera @I position@i along the view
  * direction vector (@I orientation@i).
  *
  * @return center in world coordinates
  */
  SoDEPRECATED_METHOD(9500,"Use getFocalPoint() instead.")
  SbVec3f getViewportCenter() const;

#endif /** @DEPRECATED_END */

  /**
  * Returns the camera's "focal point", located
  * @I focalDistance@i units from the camera @I position@i along the view
  * direction vector (defined by the camera's @I orientation@i field).
  *
  * This is the reference point for some algorithms. For example, orbit() is often used
  * in an "examiner" type of viewer to allow the user to rotate the camera around a
  * point of interest in the scene.  To implement orbit around the camera's focal
  * point, call the setRotationCenter() method with the focal point before calling the orbit() method.
  *
  * @return focal point in world coordinates
  */
  SbVec3f getFocalPoint() const;

  /**
  * Move camera to the specified position.
  *
  * @param pos new position in world coordinates
  */
  void setPosition( const SbVec3f& pos );

  /**
  * Move camera to the specified position.
  *
  * @param pos new position (2D point in normalized screen coordinates: 0 <= x,y <= 1)
  * @param distFromEye the distance at which the point would be back projected.
  * @param vpRegion the actual viewport region
  */
  void setPosition( const SbVec2f& pos, float distFromEye, const SbViewportRegion& vpRegion );

  /**
  * Move camera by the specified 3D displacement.
  *
  * @param trans translation value in world coordinates
  */
  void translate( const SbVec3f& trans );

  /**
  * Move camera by the specified 2D displacement.
  *
  * @param displacement 2D displacement in normalized screen coordinates: 0 <= x,y <= 1
  * @param vpRegion the actual viewport region
  */
  void translate( const SbVec2f& displacement, const SbViewportRegion& vpRegion );

  /**
  * Sets the orientation of the camera so that it points toward the given target
  * point while keeping the "up" direction of the camera parallel to the positive
  * world coordinate y-axis. If this is not possible, it uses the positive z-axis
  * as "up".  Equivalent to calling the SoCamera method pointAt().
  *
  * @param targetPoint target point
  */
  void pointAt( const SbVec3f& targetPoint );

  /**
  * Returns the rotation needed to point a camera at @a position toward the
  * given @a target point while keeping the "up" direction of the camera
  * parallel to the specified @a up vector.
  * This method does NOT modify the camera node. Set the returned rotation
  * in the camera's orientation field.
  *
  * @param position of the camera
  * @param target position of the target
  * @param up up vector of the camera
  */
  SbRotation lookAt( const SbVec3f& position, const SbVec3f& target, const SbVec3f& up );

  /**
  * Returns the direction vector of the camera.
  */
  SbVec3f getDirectionVector() const;

  /**
  * Roll the camera.
  * Incremental in-place rotation around the view direction vector (camera's local Z axis).
  * Rotations are cumulative. Camera position does not change.
  * Positive values roll the camera counter-clockwise.
  *
  * @param angle angle in radians
  */
  void roll( float angle );

  /**
  * Pivot the camera.
  * Incremental in-place rotation left/right (around the camera's up vector, i.e. local Y axis).
  * Rotations are cumulative. Camera position does not change.
  * Positive values pivot the camera left.
  *
  * @param angle angle in radians
  */
  void pivot( float angle );

  /**
  * Tilt the camera.
  * Incremental in-place rotation up/down (around the camera's local X axis).
  * Rotations are cumulative. Camera position does not change.
  * Positive values tilt the camera up.
  *
  * @param angle angle in radians
  */
  void tilt( float angle );

  /**
  * Rotate the camera.
  * Incremental in-place rotation. Rotations are cumulative.
  * Camera position does not change (rotationCenter and rotationAxis do not apply).
  *
  * @param rot new rotation
  */
  void rotate( const SbRotation& rot );

  /**
  * Set the orientation of the camera.
  * Replaces any previous rotations applied to the camera. Camera position does not change.
  *
  * @param rot new orientation
  */
  void setOrientation( const SbRotation& rot );

  /**
  * Set the center of rotation.
  * Default is (0, 0, 0). This position is used by the rotate(float) and orbit() methods.
  *
  * @param pos position in world coordinates
  */
  void setRotationCenter( const SbVec3f& pos );

  /**
  * Set the axis of rotation.
  * Default is (0, 0, 0). This vector is used by the rotate(float) method.
  *
  * @param axis new rotations axis
  */
  void setRotationAxis( const SbVec3f& axis );

  /**
  * Rotate the camera around the center specified by
  * #setRotationCenter() and the axis specified by #setRotationAxis().
  *
  * The default rotation axis is (0,0,0), so calling this
  * method will not have any visible effect until a rotation axis is set.
  * The rotation axis is automatically set by some methods, e.g. orbit().
  *
  * @param angle angle in radians
  */
  void rotate( float angle );

  /**
   * Moves and rotates the camera to be aligned with the given direction vector while keeping
   * the up direction of the camera parallel to the specified up vector.
   * The camera @I position @i and @I orientation @i are changed to look at the point of interest,
   * but the @I focalDistance @i is not changed.
   *
   * @param direction The new view direction vector: this is the vector from the camera's position
   * to the target point. Any future call to the method getDirectionVector() will return the same vector.
   * @param up The new up direction vector
   */
  void viewAxis( const SbVec3f& direction, const SbVec3f& up );

  /**
  * NOTE: The camera @I position @i is not changed, so the camera may not be looking
  * toward the scene. Calling viewAll() will move the camera so that the scene
  * is visible again.
  *
  * @param reverse If false, rotate to look in the -X direction, else in +X direction.
  */
  void viewX( bool reverse = false );

  /**
  * Rotate camera to look parallel to the Y axis with +Z up.
  * NOTE: The camera position is not changed, so the camera may not be looking
  * toward the scene. Calling viewAll() will move the camera so that the scene
  * is visible again.
  *
  * @param reverse If false, rotate to look in the -Y direction, else in +Y direction.
  */
  void viewY( bool reverse = false );

  /**
  * Rotate camera to look parallel to the Z axis with +Y up.
  * NOTE: The camera position is not changed, so the camera may not be looking
  * toward the scene. Calling viewAll() will move the camera so that the scene
  * is visible again.
  *
  * @param reverse If false, rotate to look in the -Z direction, else in +Z direction.
  */
  void viewZ( bool reverse = false );

  /**
  * Move camera to the center of the specified box.
  * The near and far clipping planes will be adjusted
  * to see the inside of the box. If the box is empty, nothing happens.
  */
  void viewIn( const SbBox3f& bbox );

  /**
  * Move camera to the center of the scene defined by the specified path.
  * Compute the bounding box of the scene and move the camera to the center of
  * this box.
  * The near and far clipping planes will be adjusted
  * to see the inside of the box.
  *
  * @param path path to the scene
  * @param vpRegion the actual viewport region
  */
  void viewIn( SoPath* path, const SbViewportRegion& vpRegion );

  /**
  * Move camera to the center of the scene defined by the specified node.
  * Compute the bounding box of the scene and move the camera to the center of
  * this box.
  * The near and far clipping planes will be adjusted
  * to see the inside of the box.
  *
  * @param node root node of the scene
  * @param vpRegion the actual viewport region
  */
  void viewIn( SoNode* node, const SbViewportRegion& vpRegion );

  /**
  * Move the camera to view the scene defined by the given path.
  * Equivalent to calling the SoCamera method viewAll().
  * The camera @I position@i is changed, but not the @I orientation@i.
  * - Orthographic camera: the @I height@i field is changed.
  * - Perspective camera: the @I heightAngle@i field is NOT changed.
  */
  void viewAll( SoPath* path, const SbViewportRegion& vpRegion );

  /**
  * Move the camera to view the scene defined by the given node.
  * Equivalent to calling the SoCamera method viewAll().
  * The camera @I position@i is changed, but not the @I orientation@i.
  * - Orthographic camera: the @I height@i field is changed.
  * - Perspective camera: the @I heightAngle@i field is NOT changed.
  */
  void viewAll( SoNode* node, const SbViewportRegion& vpRegion );

  /**
  * Project specified 2D point on the plane parallel to the near plane that
  * is at @I distFromEye @i units from the eye, see also SbViewVolume::getPlanePoint.
  *
  * @param point point to project in normalized screen coordinates (0 <= x,y <= 1)
  * @param distFromEye the distance at which the point would be back projected
  * @param vpRegion the actual viewport region
  * @return projected point in world coordinates
  */
  SbVec3f projectToPlane( const SbVec2f& point, float distFromEye, const SbViewportRegion& vpRegion ) const;

  /**
  * Map a 3D point in world coordinates to a 2D point in normalized screen
  * coordinates (0 <= x,y <= 1), see also SbViewVolume::projectToScreen.
  *
  * @param point point to project in world coordinates
  * @param vpRegion the actual viewport region
  * @return 2D point in normalized screen coordinates (0 <= x,y <= 1)
  */
  SbVec2f projectToScreen( const SbVec3f& point, const SbViewportRegion& vpRegion ) const;

  /**
  * Adjust near and far clipping planes to minimize clipping of objects in the scene.
  * Equivalent to the auto clip planes feature in the viewer classes.
  * Typically called before each render traversal.
  */
  void adjustClippingPlanes( SoNode* sceneRoot, const SbViewportRegion& vpRegion );

  /**
  * Zooms to the region defined by the given corners in normalized screen coordinates.
  * - Perspective camera: This method modifies the @a heightAngle field. @BR
  *   Modifying the heightAngle field changes the projection, similar to changing
  *   the focal length of a camera. Making the heightAngle very small or very large
  *   may cause rendering artifacts.  Therefore the #dollyInWindow method is
  *   recommended when using a perspective camera.
  *
  * - Orthographic camera: This method modifies the @a height field.
  *
  * @param topLeftCorner top left corner in normalized screen coordinates (0 <= x,y <= 1)
  * @param bottomRightCorner bottom right corner in normalized screen coordinates (0 <= x,y <= 1)
  * @param vpRegion the actual viewport region
  */
  void zoomInWindow( const SbVec2f& topLeftCorner, const SbVec2f& bottomRightCorner, const SbViewportRegion& vpRegion);

  /**
  * Moves the camera forward or backward along the view
  * vector to view the region defined by the given corners in normalized screen coordinates.
  *
  * - Perspective camera: The visual result is similar to zoomInWindow(),
  *   but this method changes the camera @a position (closer or farther away)
  *   and @a focalDisance fields instead of changing the heightAngle.
  *   This is the recommended method when using a perspective camera, because it avoids
  *   the limitations of very small or very large heightAngle values.
  *
  * - Orthographic camera: This method does nothing because moving the camera
  *   closer or farther away does not change the visual result.
  */
  void dollyInWindow( const SbVec2f& topLeftCorner, const SbVec2f& bottomRightCorner, const SbViewportRegion& vpRegion );

  /**
  * Modifies the camera to ajust the view around the specified point.
  * The algorithm is the same as the #dolly() method (for a perspective camera
  * the position field is changed, for an orthographic camera the height field
  * is changed) and the delta parameter is used the same way.
  *
  * @param centerPos zoom center in normalized screen coordinates (0 <= x,y <= 1).
  *        Typically this value is obtained from the SoEvent method getNormalizedPosition().
  * @param scaleFactor zoom scale
  * @param vpRegion the actual viewport region
  */
  void dollyWithZoomCenter( const SbVec2f& centerPos, float scaleFactor, const SbViewportRegion& vpRegion );

  /**
  * Set the starting point for interactive panning.
  * This method should be called when starting an interaction.
  * Typically this is triggered by a mouse button or touch event.
  * On subsequent mouse move or touch events, call the #pan() method with the
  * new cursor or touch position.
  *
  * @param initPos initial position in normalized screen coordinates (0 <= x,y <= 1).
  *        Typically this value is obtained from the SoEvent method getNormalizedPosition().
  * @param vpRegion the actual viewport region
  */
  void activatePanning( const SbVec2f& initPos, const SbViewportRegion& vpRegion );

  /**
  * Pan the camera based on offset from starting point.
  * This method can be used to implement an interactive panning operation in
  * response to move move or touch events. The #activatePanning() method should
  * be called at the start of the operation.
  *
  * @param newPos new position in normalized screen coordinates (0 <= x,y <= 1).
  *        Typically this value is obtained from the SoEvent method getNormalizedPosition().
  * @param vpRegion the actual viewport region
  */
  void pan( const SbVec2f& newPos, const SbViewportRegion& vpRegion );

  /**
  * Set the starting point for interactive orbiting.
  * This method should be called when starting an interaction.
  * Typically this is triggered by a mouse button or touch event.
  * On subsequent mouse move or touch events, call the orbit() method with the
  * new cursor or touch position.
  *
  * Orbit is often used in an "examiner" type of viewer to allow the user to rotate
  * the camera around a point of interest in the scene.  To implement orbit around
  * the camera's "focal point", call setRotationCenter() with the value returned from
  * getFocalPoint().
  *
  * @param startPos starting position, 2D point in normalized screen coordinates (0 <= x,y <= 1).
  *        Typically this value is obtained from the SoEvent method getNormalizedPosition().
  */
  void activateOrbiting( const SbVec2f& startPos );

  /**
  * Orbit the camera by the specified rotation.
  * "Orbit" means to rotate the camera around a fixed point in 3D space.  The rotation
  * point is specified using the setRotationCenter() method.  This method automatically
  * calls the setRotationAxis() method with the axis defined by the SbRotation parameter.
  *
  * Orbit is often used in an "examiner" type of viewer to allow the user to rotate
  * the camera around a point of interest in the scene.  Orbit based on mouse or touch
  * events can be implemented using the orbit() method. This method can be used,
  * for example, to animate the camera to a new viewpoint.  To implement orbit around
  * the camera's "focal point", call setRotationCenter() with the value returned from
  * getFocalPoint().
  *
  * @param rotation Incremental rotation, i.e. this rotation is multiplied with
  *                 the current camera orientation.
  */
  void orbit( const SbRotation& rotation );

  /**
  * Orbit the camera based on offset from the previous screen position.
  * This method can be used to implement an interactive orbit operation in
  * response to mouse move or touch events. The #activateOrbiting() method should
  * be called at the start of the operation.  This method uses an SbSphereSheetProjector
  * to compute a rotation value based on the specified point and the previously
  * specified point.
  *
  * @param newPos new position, 2D point in normalized screen coordinates (0 <= x,y <= 1).
  *        Typically this value is obtained from the SoEvent method getNormalizedPosition().
  */
  void orbit( const SbVec2f& newPos );

  /**
  * Save current camera.
  * The values of the camera fields are saved and can be restored by calling popCamera().
  */
  void pushCamera();

  /**
  * Restore saved camera.
  * The values of the camera fields are restored to the last pushed values.
  * Returns true if successful, false if the vector of pushed cameras is empty.
  */
  SbBool popCamera();

  /**
  * Copy field values from the specified camera to the camera of this interactor.
  * If the two cameras are not the same type, common fields are copied and
  * the height or heightAngle field is computed.
  */
  void synchronize( SoCamera* camera );

protected:

  SoCameraInteractorImpl* m_impl;

private:

  /**
  * Private constructor
  * Use getNewInstance() function to instanciate a new SoCameraInteractor
  */
  SoCameraInteractor() {}

  ~SoCameraInteractor();

};

#endif // SO_CAMERA_INTERACTOR
