#ifndef SO_EVENT_BUILDER
#define SO_EVENT_BUILDER

#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/SbPImpl.h>

class SoMouseWheelEvent;
class SoLocation2Event;
class SoTouchManager;

SO_PIMPL_BASE_PUBLIC_DECLARATION(SoEventBuilder);

/**
* @VSGEXT Utility class that generates Open Inventor events.
*
* @ingroup ViewerComponents
*
* @DESCRIPTION
* 
* This class is used to generate an Open Inventor event object from a system independent event description.
*
* Note: The same SoEvent object is returned each time. This object is owned by the SoEventBuilder
*       instance and must not be modified or destroyed by the application.
*
*/
class INVENTOR_API SoEventBuilder
{

  SO_PIMPL_BASE_PUBLIC_HEADER(SoEventBuilder);

public:

  /** Constructor */
  SoEventBuilder();

  /** Destructor */
  ~SoEventBuilder();

  /**
   * Return an SoMouseButtonEvent corresponding to given parameters.
   * The returned object should not be destroyed.
   * @param x X coordinate in pixel.
   * @param y Y coordinate in pixel.
   * @param buttonId Which mouse button has been released.
   * @param isAltDown Specify if Alt key was down when mouse has been released.
   * @param isCtrlDown Specify if Ctrl key was down when mouse has been released.
   * @param isShiftDown Specify if Shift key was down when mouse has been released.
   */
  SoMouseButtonEvent* getMouseReleaseEvent(int x, int y, SoMouseButtonEvent::Button buttonId, bool isAltDown, bool isCtrlDown, bool isShiftDown);

  /**
   * Return an SoMouseButtonEvent corresponding to given parameters.
   * The returned object should not be destroyed.
   * @param x X coordinate in pixel.
   * @param y Y coordinate in pixel.
   * @param buttonId Which mouse button has been pressed.
   * @param isAltDown Specify if Alt key was down when mouse has been pressed.
   * @param isCtrlDown Specify if Ctrl key was down when mouse has been pressed.
   * @param isShiftDown Specify if Shift key was down when mouse has been pressed.
   */
  SoMouseButtonEvent* getMousePressEvent(int x, int y, SoMouseButtonEvent::Button buttonId, bool isAltDown, bool isCtrlDown, bool isShiftDown);

  /**
   * Return an SoMouseButtonEvent corresponding to given parameters.
   * The returned object should not be destroyed.
   * @param x X coordinate in pixel.
   * @param y Y coordinate in pixel.
   * @param buttonId Which mouse button has been double clicked.
   * @param isAltDown Specify if Alt key was down when mouse has been double clicked.
   * @param isCtrlDown Specify if Ctrl key was down when mouse has been double clicked.
   * @param isShiftDown Specify if Shift key was down when mouse has been double clicked.
   */
  SoMouseButtonEvent* getMouseDoubleClickEvent(int x, int y, SoMouseButtonEvent::Button buttonId, bool isAltDown, bool isCtrlDown, bool isShiftDown);

  /**
   * Return an SoMouseWheelEvent corresponding to given parameters.
   * The returned object should not be destroyed.
   * @param delta Number of tick mouse wheel has done. For most of mice, one tick correspond to
   * 15 degree on the wheel but it may change depending on hardware.
   * @param isAltDown Specify if Alt key was down when wheel has been activated.
   * @param isCtrlDown Specify if Ctrl key was down when when wheel has been activated.
   * @param isShiftDown Specify if Shift key was down when when wheel has been activated.
   */
  SoMouseWheelEvent* getMouseWheelEvent(int delta, bool isAltDown, bool isCtrlDown, bool isShiftDown);

  /**
   * Return an SoLocation2Event corresponding to given parameters.
   * The returned object should not be destroyed.
   * @param x X coordinate in pixel.
   * @param y Y coordinate in pixel.
   * @param isAltDown Specify if Alt key was down when mouse has been moved.
   * @param isCtrlDown Specify if Ctrl key was down when mouse has been moved.
   * @param isShiftDown Specify if Shift key was down when mouse has been moved.
   */
  SoLocation2Event* getMouseMoveEvent(int x, int y, bool isAltDown, bool isCtrlDown, bool isShiftDown);

  /**
   * Return an SoLocation2Event corresponding to given parameters. A MouseEnter event
   * correspond to a mouse entering some GUI area like a panel or a window.
   * The returned object should not be destroyed.
   * @param x X coordinate in pixel.
   * @param y Y coordinate in pixel.
   * @param isAltDown Specify if Alt key was down when mouse has entered area.
   * @param isCtrlDown Specify if Ctrl key was down when mouse has entered area.
   * @param isShiftDown Specify if Shift key was down when mouse has entered area.
   */
  SoLocation2Event* getMouseEnterEvent(int x, int y, bool isAltDown, bool isCtrlDown, bool isShiftDown);

  /**
   * Return an SoLocation2Event corresponding to given parameters. A MouseLeave event
   * correspond to a mouse leaving some GUI area like a panel or a window.
   * The returned object should not be destroyed.
   * @param x X coordinate in pixel.
   * @param y Y coordinate in pixel.
   * @param isAltDown Specify if Alt key was down when mouse has leaved area.
   * @param isCtrlDown Specify if Ctrl key was down when mouse has leaved area.
   * @param isShiftDown Specify if Shift key was down when mouse has leaved area.
   */
  SoLocation2Event* getMouseLeaveEvent(int x, int y, bool isAltDown, bool isCtrlDown, bool isShiftDown);

  /**
   * Return an SoKeyboardEvent corresponding to given parameters.
   * The returned object should not be destroyed.
   * @param key key that have been released.
   * @param isAltDown Specify if Alt key was down when key has been released.
   * @param isCtrlDown Specify if Ctrl key was down when key has been released.
   * @param isShiftDown Specify if Shift key was down when key has been released.
   */
  SoKeyboardEvent* getKeyReleaseEvent(SoKeyboardEvent::Key key, bool isAltDown, bool isCtrlDown, bool isShiftDown);


  /**
   * Return an SoKeyboardEvent corresponding to given parameters.
   * The returned object should not be destroyed.
   * @param key key that have been pressed.
   * @param isAltDown Specify if Alt key was down when key has been pressed.
   * @param isCtrlDown Specify if Ctrl key was down when key has been pressed.
   * @param isShiftDown Specify if Shift key was down when key has been pressed.
   */
  SoKeyboardEvent* getKeyPressEvent(SoKeyboardEvent::Key key, bool isAltDown, bool isCtrlDown, bool isShiftDown);

  /**
   * Return a list of SoEvent corresponding to given parameters.
   * This list may contain some recognized SoGestureEvent.
   * The returned events should not be destroyed.
   * @param x X coordinate in pixel of the finger position.
   * @param y Y coordinate in pixel of the finger position.
   * @param fingerId id of the finger used.
   */
  const std::vector<const SoEvent*>& getTouchDownEvent(int x, int y, unsigned long fingerId);

  /**
   * Return a list of SoEvent corresponding to given parameters.
   * This list may contain some recognized SoGestureEvent.
   * The returned events should not be destroyed.
   * @param x X coordinate in pixel of the finger position.
   * @param y Y coordinate in pixel of the finger position.
   * @param fingerId id of the finger used.
   */
  const std::vector<const SoEvent*>& getTouchUpEvent(int x, int y, unsigned long fingerId);

  /**
   * Return a list of SoEvent corresponding to given parameters.
   * This list may contain some recognized SoGestureEvent.
   * The returned events should not be destroyed.
   * @param x X coordinate in pixel of the finger position.
   * @param y Y coordinate in pixel of the finger position.
   * @param fingerId id of the finger used.
   */
  const std::vector<const SoEvent*>& getTouchMoveEvent(int x, int y, unsigned long fingerId);

  /**
   * Get the touch manager instance used by this event builder to build touch events.
   */
  SoTouchManager& getTouchManager() const;

};

#endif // SO_EVENT_BUILDER
