/*=======================================================================
** VSG_COPYRIGHT_TAG
**=======================================================================*/

#ifndef RenderArea_H
#define RenderArea_H

#include <Inventor/sys/port.h>

#if defined(_WIN32) && !defined(OIV_DISABLE_AUTOLINK) && !defined(WinViewerComponents_EXPORTS)
#  if !defined(__WIN_VIEWER_COMPONENTS_LIB)
#    if defined(_DEBUG)
#      define __WIN_VIEWER_COMPONENTS_LIB "fei_inventor_WinViewerComponentsD.lib"
#    else
#      define __WIN_VIEWER_COMPONENTS_LIB "fei_inventor_WinViewerComponents.lib"
#    endif
#  endif
#  pragma comment(lib, __WIN_VIEWER_COMPONENTS_LIB)
#endif

#include <Inventor/misc/SoRef.h>
#include <Inventor/sys/SoGL.h>
#include <Inventor/ViewerComponents/SoRenderAreaCore.h>
#include <Inventor/SbColorRGBA.h>

#include <Inventor/sys/port.h>

class SoNode;

/**
 * Class to render an OpenInventor scene in a Win32 window.
 *
 * @ingroup ViewerComponentsWin
 *
 * @DESCRIPTION
 *
 * RenderArea creates a Win32 window, either stand-alone or as a child of another window.
 * Use the initializeGL() method to create the 3D rendering window and the
 * OpenGL context. Use the setSceneGraph() method to specify the scene graph to be
 * rendered.
 *
 * A SceneExaminer node is typically used to allow the user to manipulate the scene camera.
 *
 * Note: The SoRenderAreaCore object is not created until initializeGL() is called. @BR
 * Generally the application should not call methods like getClearColor() and setClearColor()
 * until @I after @i the call to initializeGL().
 *
 * @SEE_ALSO
 *    RenderAreaInteractive, RenderAreaExaminer, SceneInteractor, SceneExaminer, SoRenderAreaCore
 */
class WINVIEWERCOMPONENTS_API RenderArea : public SiRenderArea, public SiRenderAreaAntialiasing, public SiRenderAreaStillSuperSampling
{
public:
  /**
   * Constructor
   */
  RenderArea();

  /**
   * Constructor with parent window
   */
  RenderArea(HWND parentHWND);

  /**
   * Destructor
   */
  virtual ~RenderArea();

  /**
   * Return the render area associated to given HWND
   */
  static RenderArea* getRenderArea( HWND hWnd );

  /**
   * Sets the open inventor scene graph to display.
   */
  virtual void setSceneGraph( SoNode* sceneGraph );

  /**
   * Gets the scene graph to render.
   */
  virtual SoNode* getSceneGraph() const;

  /**
   * Defines the color buffer and depth buffer clear policy.
   * @useenum{ClearPolicy} Default is COLORBUFFER_AND_DEPTHBUFFER.
   */
  virtual void setClearPolicy(ClearPolicy policy);

  /**
   * Gets the color buffer and depth buffer clear policy.
   */
  virtual ClearPolicy getClearPolicy() const;

  /**
   * Defines the RGBA value used when the color buffer is cleared.
   * Default is transparent black (0,0,0,0).
   */
  virtual void setClearColor(const SbColorRGBA& color);

  /**
   * Gets the RGBA value used when the color buffer is cleared.
   */
  virtual SbColorRGBA getClearColor() const;

  /**
   * Defines the depth value used when the depth buffer is cleared.
   * Default is 1. Range is 0..1.
   */
  virtual void setClearDepth(float depth);

  /**
   * Gets the depth value used when the depth buffer is cleared.
   */
  virtual float getClearDepth() const;

  /**
   * Defines the size used for rendering.
   */
  virtual void setSize(const SbVec2i32& size);

  /**
   * Gets the size used for rendering.
   */
  virtual SbVec2i32 getSize() const;

  /**
   * Defines the render action used for rendering.
   */
  virtual void setGLRenderAction( SoGLRenderAction* glAction );

  /**
   * Returns the render action used for rendering.
   */
  virtual SoGLRenderAction* getGLRenderAction() const;

  /**
  * Define the antialiasing mode.
  * @param mode The antialiasing algorithm. Default is NO_ANTIALIASING which turns off antialiasing.
  */
  virtual void setAntialiasingMode( SoSceneManager::AntialiasingMode mode ) override;

  /**
  * @see setAntialiasingMode().
  */
  virtual SoSceneManager::AntialiasingMode getAntialiasingMode() const override;

  /**
  * Define the antialiasing quality value.
  * @param quality The quality is a factor in the range [0.0,1.0]. @BR
  *        Default is 0.0. The value 0.0 turns off antialiasing.
  */
  virtual void setAntialiasingQuality( float quality ) override;

  /**
  * @see setAntialiasingQuality().
  */
  virtual float getAntialiasingQuality() const override;

  /**
  * Set quality for supersampling when "still" (not interacting).
  * When quality is greater than 0, still images will be automatically supersampled.
  * @param quality The quality is a factor in the range [0.0,1.0]. @BR
  *        Default value is 0.0. Use the value 0.0 to turn off still supersampling. 0.5 is a typical value.
  * See also setStillSuperSamplingDelay.
  */
  virtual void setStillSuperSamplingQuality( float quality ) override;

  /**
  * @see setStillSuperSamplingQuality().
  */
  virtual float getStillSuperSamplingQuality() const override;

  /**
  * Set delay for supersampling when "still" (not interacting).
  * If greater than 0, images will be supersampled after the specified delay.
  * @param delay The delay is in milliseconds. @BR
  *        If greater than 0, images will be supersampled after the specified delay
  * See also setStillSuperSamplingQuality.
  */
  virtual void setStillSuperSamplingDelay( unsigned int delay ) override;

  /**
  * @see setStillSuperSamplingDelay().
  */
  virtual unsigned int getStillSuperSamplingDelay() const override;

  /**
   * Returns the event handler that raises when a new render starts.
   */
  virtual SbEventHandler<RenderEventArg&>& onStartRender();

  /**
   * This method is called in order to create window and to create OpenGL context.
   */
  virtual void initializeGL( int width, int height );

  /**
   * Release the OpenGL context.
   */
  virtual void uninitializeGL();

  /**
   * This method is called when the size of the widget is modified.
   * We need to know when the size is modified in order to update the viewport for
   * the internal computation of the clipping planes, aspect ratio, mouse coordinates...
   *
   * It is called when the widget is displayed for the first time and when the size changes.
   */
  virtual void resizeGL( int width, int height );

  /**
   * This method is called when we want to render the OpenGL surface.
   * It is called also when OpenInventor needs an update through the oivRenderCB function.
   * OpenInventor must be able to inform us of the required update when the scenegraph is
   * modified.
   */
  virtual void paintGL();

  /**
   * Return true if the window is active
   */
  bool isActive()
  {
    return m_active;
  }

  /**
   * This method is call to process windows events
   */
  virtual void processEvents( UINT uMsg, WPARAM wParam, LPARAM lParam );

  /**
   * Return the window used by this render area
   */
  HWND getHWND()
  {
    return m_hWnd;
  }

  /**
   * Return the width of the window
   */
  int getWidth() const
  {
    return m_width;
  }

  /**
   * Return the height of the window
   */
  int getHeight() const
  {
    return m_height;
  }

protected:

  /**
   * Render the scene graph.
   */
  virtual RenderStatus render();

  /**
   * Instance of the class which provides the algorithms for viewing and managing
   * OpenInventor scene graphs.
   */
  SoRef<SoRenderAreaCore> m_renderAreaCore;

  /**
   * Scene graph used by this render area
   */
  SoNode* m_sceneGraph;

  /**
   * Clear color used by this render area
   */
  SbColorRGBA m_color;

  /**
   * Clear depth used by this render area
   */
  float m_depth;

  /**
   * Size used by this render area
   */
  SbVec2i32 m_size;

  /**
   * GLRenderAction used by this render area
   */
  SoGLRenderAction* m_glRenderAction;

  /**
   * Antialiasing mode used by this render area
   */
  SoSceneManager::AntialiasingMode m_antialiasingMode;

  /**
   * Antialiasing quality used by this render area
   */
  float m_antialiasingQuality;

  /**
   * Delay for still supersampling
   */
  unsigned int m_stillAntialiasingDelay;

  /**
   * Quality of still supersampling
   */
  float m_stillAntialiasingQuality;

  // ********************************
  // windows stuff
  // ********************************

  HINSTANCE m_hInstance;

  /**
   * The device context for OpenGL
   */
  HDC m_hDC;

  /**
   * The window
   */
  HWND m_hWnd;

  /**
   * The parent window
   */
  HWND m_parentHWND;

  /**
   * Specify if the window is active
   */
  bool m_active;

  /**
   * Map to associate a HWND to a specific render area instance
   */
  static std::map<HWND, RenderArea*> m_hWndToRenderAreaMap;

  /**
   * Store the width of the window
   */
  int m_width;

  /**
   * Store the height of the window
   */
  int m_height;

  /**
   * Store the clear policy
   */
  ClearPolicy m_clearPolicy;
};

#endif // RenderArea_H
