/*=================================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                 ***
 ***                                                                            ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS   ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR   ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                       ***
 ***                                                                            ***
 ***                        RESTRICTED RIGHTS LEGEND                            ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS  ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN  ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT  ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN  ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         ***
 ***                                                                            ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                    ***
 ***                              BORDEAUX, FRANCE                              ***
 ***                            ALL RIGHTS RESERVED                             ***
 =================================================================================*/

#pragma once

#include <Inventor/SbPImpl.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/fields/SoSFNode.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFColor.h>
#include <Inventor/fields/SoSFVec2f.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFFilePathString.h>

SO_PIMPL_PUBLIC_DECLARATION( SoViewingCube )

/**
 * Interactive cubic shape to control the orientation of a camera.
 *
 * @VSGEXT @PREVIEWTAG
 *
 * @ingroup ViewerComponentsNodes
 *
 * @DESCRIPTION
 * The viewing cube is an interactive shape that indicates the current orientation
 * of a camera (like a "compass" or "gnomon") and also responds to user actions by
 * re-orienting the associated camera to one of the predefined orientations.
 * These orientations are defined so that the camera will be in front
 * of a selected part of the bounding cube of the scene.
 *
 * @B Interactions @b
 *
 * When the mouse cursor is over the viewing cube, the part (face, edge, or corner)
 * that is under the cursor is highlighted with the #selectionColor. Then if the
 * left mouse button (button 1) is clicked (pressed and released), the selected part
 * of the viewing cube is used to define the new orientation.
 *
 * The viewing cube can be used as a replacement for the X/Y/Z camera orientation
 * buttons implemented in many graphical user interfaces. It provides even more
 * possibilities thanks to the pickable edges and corners of the viewing cube
 * which offer 26 different viewing positions. However, if the #edgeSize is set
 * to 0, the edges and corners are not pickable. In this case the user can only
 * select six different positions.
 *
 * The camera controlled by the viewing cube is defined by the field #sceneCamera.
 * This is normally the same camera that the 3D viewer is controlling.
 * In this case both the viewer and the viewing cube control the camera and
 * the viewing cube orientation is synchronized with the #sceneCamera.
 *
 * In principle, as SoViewingCube is a standard class derived from SoNode,
 * it can be inserted in any scene graph and used with a classical viewer
 * or SceneExaminer class. In that case the viewing cube will correctly
 * indicate the camera orientation, but it will not be convenient for the
 * user to use the viewing cube's interaction features. These viewers are
 * always in either viewing mode (mouse events control the camera) or in
 * selection mode (mouse events are sent to the scene graph). In viewing
 * mode, the viewing cube will not respond to mouse clicks, but in selection
 * mode the viewer will not respond to mouse clicks.
 *
 * Therefore we recommend to use the SceneOrbiter class as the viewer when
 * using a viewing cube. SceneOrbiter is a "mode less" viewer. A mouse click
 * without moving the mouse is interpreted as a selection (for example
 * triggering the viewing cube behavior), but a mouse click and "drag" is
 * interpreted as controlling the camera. For convenience, the SceneOrbiter
 * automatically adds a viewing cube to the scene graph.
 *
 * The viewing cube is rendered in a corner of the render area. Its size
 * and position in the render area are defined by the #size and #position fields.
 *
 * <center>
 *   @TABLE_0B
 *       @TR @IMAGE vcube_top_right.png
 *       @TR @I viewing cube at the top right corner of a render area @i
 *   @TABLE_END
 * </center>
 *
 * @B Rendering customization @b
 *
 * When the mouse cursor is moved on top of the viewing cube,
 * the part that is under the cursor is highlighted using the color defined
 * by the field #selectionColor.
 * The colors of the cube can also be personalized with 3 fields: #faceColor, #edgeColor and #cornerColor.
 *
 *
 * @B Cubic shape customization @b
 *
 * The shape of the viewing cube can be customized by several fields.
 * The shape of the edges and corners is defined by the field #edgeStyle.
 * The edges can form a sharp corner, be rounded or be flat (beveled).
 * The size of the edges and corners relative to the cube face is defined by the field #edgeSize.
 *
 * <center>
 *   @TABLE_0B
 *       @TR @IMAGE vcube_edgesize01_round.png
 *       @TR @I #edgeStyle=ROUND and #edgeSize=0.1 @i
 *   @TABLE_END
 * </center>
 *
 * Each flat face of the viewing cube can also be customized. By default
 * the text Top/Bottom/Front/Back/Left/Right appears on the center of each face
 * according to its orientation. The six fields #facePosX ... #faceNegZ allow
 * the application to replace the default appearance of each face with an image.
 * Note that these images should have a transparent background so the highlight
 * color is visible when the cursor is over the face.
 *
 * <center>
 *   @TABLE_0B
 *       @TR @IMAGE vcube_edgesize025_round.png
 *       @TR @I viewing cube custom faces with F/R/B textures @i
 *   @TABLE_END
 * </center>
 *
 * @B Opacity @b
 *
 * The viewing cube's opacity may vary depending on the distance between it and the cursor.
 *    - if the cursor is far from the center of the cube, the #opacityMin is applied.
 *    - if the cursor is very close or over the cube, the #opacityMax is applied.
 *    - otherwise the opacity varies in the range [opacityMin, opacityMax]
 *
 * When the cursor moves in an area close to the cube but not over the cube
 * and if the opacityMin and opacityMax differ, a rendering of the scene occurs
 * each time the mouse is moved in this area, which can lead to slowing down of the application.
 *
 * By default, #opacityMin and #opacityMax are set to 1, which means the cube is always opaque.
 *
 * @B Limitations: @b
 *    - Some visual artifact on semi transparent ViewingCube when you use NO_SORT transparency type.
 *    - The opacity of the ViewingCube is only updated on a mouse move event without button pressed. Therefore the opacity won't change if you move the mouse without pressing a button.
 *
 * @B Compass @b
 * The #compass is not impacted by the varying opacity, thus independently from the fields #opacityMin and #opacityMax
 *
 * @B Animation @b
 *
 * To provide a better user experience, picking a part of the viewing cube
 * does not immediately change the camera orientation.
 * A smooth animation is done between the current camera orientation and
 * the new orientation. The duration of the animation is defined by the
 * #animationDuration field.
 *
 * Note that the animation of the camera is done in such a way that the default
 * text of each face Top/Bottom/Front/Back/Left/Right are always legible.
 *
 * @FILE_FORMAT_DEFAULT
 *    ViewingCube {
 *    @TABLE_FILE_FORMAT
 *       @TR sceneCamera       @TD NULL
 *       @TR size              @TD (150.0, 150.0)
 *       @TR position          @TD TOP_RIGHT
 *       @TR edgeStyle         @TD ROUND
 *       @TR edgeSize          @TD 0.4
 *       @TR selectionColor    @TD cyan
 *       @TR faceColor         @TD 0.8 0.8 0.8
 *       @TR edgeColor         @TD 0.8 0.8 0.8
 *       @TR cornerColor       @TD 0.8 0.8 0.8
 *       @TR animationDuration @TD 0.8
 *       @TR upAxis            @TD "Y"
 *       @TR facePosX          @TD ""
 *       @TR faceNegX          @TD ""
 *       @TR facePosY          @TD ""
 *       @TR faceNegY          @TD ""
 *       @TR facePosZ          @TD ""
 *       @TR faceNegZ          @TD ""
 *       @TR opacityMin        @TD 1
 *       @TR opacityMax        @TD 1
 *       @TR compass           @TD NULL
 *    @TABLE_END
 *    }
 *
 * @SEE_ALSO
 *    SceneOrbiter
 *
 * @PREVIEWFEATURES
 *
 */
class INVENTOR_API SoViewingCube : public SoNode
{
  SO_NODE_HEADER( SoViewingCube );
  SO_PIMPL_PUBLIC_HEADER( SoViewingCube )

public:
  /**
   * Possible positions of the viewing cube in the scene camera viewport.
   */
  enum PositionInViewport
  {
    /** The viewing cube is positioned at the top right corner */
    TOP_RIGHT,
    /** The viewing cube is positioned at the top left corner */
    TOP_LEFT,
    /** The viewing cube is positioned at the bottom right corner */
    BOTTOM_RIGHT,
    /** The viewing cube is positioned at the bottom left corner */
    BOTTOM_LEFT,
  };

  /**
   * Different types of edges.
   */
  enum EdgeStyle
  {
    /**
     * Edges are flat (beveled).
     */
    FLAT,
    /**
     * Edges have a rounded shape.
     */
    ROUND,
    /**
     * Edges form a sharp corner.
     */
    CORNER,
  };

  /**
   * Camera which is synchronized with this viewing cube.
   * When using a viewing cube with a SceneOrbiter, the method SceneOrbiter#enableViewingCube(true)
   * automatically sets the field sceneCamera.
   *
   * However, to use a viewing cube with a SceneExaminer, the following code sample gives an example to
   * set the field #sceneCamera.
   *
   * @if_cpp
   *     @code
   *     qtViewerExaminer = new ViewerExaminer(NULL);
   *     SceneExaminer* sceneExaminer = qtViewerExaminer->getRenderArea()->getSceneInteractor();
   *     SoViewingCube* viewingCube = new SoViewingCube;
   *     viewingCube->sceneCamera = sceneExaminer->getCamera();
   *     sceneExaminer->addChild(viewingCube);
   *     @endcode
   * @endif
   * @if_dotnet
   *     @code
   *     SceneExaminer sceneExaminer = RenderAreaExaminer.SceneInteractor;
   *     SoViewingCube viewingCube = new SoViewingCube();
   *     viewingCube.sceneCamera.Value = sceneExaminer.Camera;
   *     sceneExaminer.AddChild(viewingCube);
   *     @endcode
   * @endif
   * @if_java
   *     @code
   *     viewerExaminer = new ViewerExaminer();
   *     SceneExaminer sceneExaminer = viewerExaminer.getRenderArea().getSceneInteractor();
   *     SoViewingCube viewingCube = new SoViewingCube();
   *     viewingCube.sceneCamera.setValue( sceneExaminer.getCamera() );
   *     sceneExaminer.addChild( viewingCube );
   *     @endcode
   * @endif
   */
  SoSFNode sceneCamera;

  /**
   * Size of the viewport, in pixels, in which the viewing cube is drawn.
   * Default is 150 pixels width, 150 pixels height.
   */
  SoSFVec2f size;

  /**
   * Position of the viewing cube in the scene camera viewport.
   * @useenum{PositionInViewport}.
   *
   * Default is TOP_RIGHT.
   */
  SoSFEnum position;

  /**
   * Style of edges and corners.
   * @useenum{EdgeStyle}.
   *
   *   @TABLE_0B
   *       @TR @IMAGE vcube_edgestyle_round.png
   *       @TD @IMAGE vcube_edgestyle_corner.png
   *       @TD @IMAGE vcube_edgestyle_flat.png
   *       @TR @I #edgeStyle = ROUND @TD #edgeStyle = CORNER @TD #edgeStyle = FLAT @i
   *   @TABLE_END
   */
  SoSFEnum edgeStyle;

  /**
   * Color used to highlight the part of the viewing cube that is under the cursor.
   * Default is cyan (0,1,1).
   *
   * The following image shows the highlighted edge behind the cursor when #selectionColor = (1,0,0)
   *   @TABLE_0B
   *       @TR @IMAGE vcube_selectioncolor_red.png
   *   @TABLE_END
   */
  SoSFColor selectionColor;


  /**
   * Color used to render the faces of the cube.
   * Default is gray (0.8,0.8,0.8).
   */
  SoSFColor faceColor;


  /**
   * Color used to render the edges of the cube.
   * Default is gray (0.8,0.8,0.8).
   */
  SoSFColor edgeColor;


  /**
   * Color used to render the corners of the cube.
   * Default is gray (0.8,0.8,0.8).
   */
  SoSFColor cornerColor;

  /**
   * Texture to customize the face's appearance which has a "Right" label by default.
   * This field defines the name of the file which contains this texture.
   *
   * The standard image file formats are supported. See SoRasterImageRW for the list.
   * The image format must handle an alpha channel in order to allow the selected
   * face to be highlighted with the #selectionColor.
   */
  SoSFFilePathString facePosX;

  /**
   * Texture to customize the face's appearance which has a "Left" label by default.
   * See #facePosX for detail.
   */
  SoSFFilePathString faceNegX;

  /**
   * Texture to customize the face's appearance which has a "Top" label by default.
   * See #facePosX for detail.
   */
  SoSFFilePathString facePosY;

  /**
   * Texture to customize the face's appearance which has a "Bottom" label by default.
   * See #facePosX for detail.
   */
  SoSFFilePathString faceNegY;

  /**
   * Texture to customize the face's appearance which has a "Front" label by default.
   * See #facePosX for detail.
   */
  SoSFFilePathString facePosZ;

  /**
   * Texture to customize the face's appearance which has a "Back" label by default.
   * See #facePosX for detail.
   */
  SoSFFilePathString faceNegZ;

  /**
   * Size of the edges, relative to the size of the faces.
   * The size of the corners are also adjusted according to this value.
   * When the #edgeSize is 0, neither edges nor corners of the viewing cube
   * can be highlighted or selected.
   * Thus the viewing cube allows to select only 6 predefined positions
   * of the camera.
   *
   * This field's value is clamped in a range [0-1].
   * #edgeSize = 1 corresponds to 25% of the face's size.
   * Default is 0.4.
   *
   * Depending on the #edgeStyle value, the #edgeSize defines
   * either a radius (for ROUND style) or a width.
   *
   *   @TABLE_0B
   *       @TR @IMAGE vcube_edgesize0.png
   *       @TR @I Above is a viewing cube with #edgeSize = 0 @i
   *   @TABLE_END
   *
   * Below are images of a viewing cube with different #edgeSize=0.1 on the left column
   * and #edgeSize=0.25 on the right column.
   *
   *   @TABLE_0B
   *       @TR @IMAGE vcube_edgesize01_corner.png
   *       @TD @IMAGE vcube_edgesize025_corner.png
   *       @TR @I #edgeStyle = CORNER @i
   *       @TR @TR
   *       @TR @IMAGE vcube_edgesize01_round.png
   *       @TD @IMAGE vcube_edgesize025_round.png
   *       @TR @I #edgeStyle = ROUND @i
   *       @TR @TR
   *       @TR @IMAGE vcube_edgesize01_flat.png
   *       @TD @IMAGE vcube_edgesize025_flat.png
   *       @TR @I #edgeStyle = FLAT @i
   *   @TABLE_END
   */
  SoSFFloat edgeSize;

  /**
   * Duration of camera movement to reach the desired position.
   * Any negative value of this field is equivalent to 0 (no animation).
   * Default is 0.8 seconds.
   */
  SoSFFloat animationDuration;

   /**
   * Up axis of the scene. Setting the up axis allows the adjustment of the viewing cube rotations.
   * Thus, this axis remains vertical at all times relative to the viewport, it is always parallel to the vertical border of the viewport.
   * The chosen axis defines the initial position and orientation of the camera (front-top-right corner).
   * @useenum{openinventor::inventor::Axis::Type}.
   * Default is Y.
   */
  SoSFEnum upAxis;

  /**
   * Defines the viewing cube opacity when the cursor is far from it.
   * The value range is [0, 1].
   *
   * Default is 1.
   */
  SoSFFloat opacityMin;

  /**
   * Defines the viewing cube opacity when the cursor is close to it or over it.
   * The value range is [0, 1].
   *
   * Default is 1.
   */
  SoSFFloat opacityMax;

  /**
   * Defines an additional and optional scene graph to visualize a compass encapsulated in
   * the viewing cube.
   * This scene graph will have to be centered on (0, 0, 0) to be aligned with the center
   * of the viewing cube.
   * According to its size, the compass will be visible or not when the viewing cube is opaque.
   *
   * By default this field is NULL, so no compass is displayed.
   * Some examples of simple compasses are provided in the folder $OIVHOME/examples/data/Inventor/ViewingCube/Compass
   */
  SoSFNode compass;

  SoViewingCube();

  /** This node does not affect the state. Returns FALSE. */
  virtual SbBool affectsState() const;

SoEXTENDER public:
  // Implement actions
  virtual void GLRender( SoGLRenderAction* action );
  virtual void handleEvent( SoHandleEventAction* ha );

SoINTERNAL public:
  static void initClass();
  static void exitClass();
  virtual void fieldHasChanged(SoField* field);

  virtual SoChildList* getChildren() const;

protected:
  virtual ~SoViewingCube();

private:
  
  /**
   * Type of the easing function
   */
  enum EasingType
  {
    /* bezier cubic curve from points (0, 0), (0.47, 0), (0.745, 0.715), (1, 1) */
    EaseInSine,
    /* bezier cubic curve from points (0, 0), (0.39, 0.575), (0.565, 1), (1, 1) */
    EaseOutSine,
    /* bezier cubic curve from points (0, 0), (0.445, 0.05), (0.55, 0.95), (1, 1) */
    EaseInOutSine,
    /* bezier cubic curve from points (0, 0), (0.55, 0.085), (0.68, 0.53), (1, 1) */
    EaseInQuad,
    /* bezier cubic curve from points (0, 0), (0.25, 0.46), (0.45, 0.94), (1, 1) */
    EaseOutQuad,
    /* bezier cubic curve from points (0, 0), (0.455, 0.03), (0.515, 0.955), (1, 1) */
    EaseInOutQuad,
    /* bezier cubic curve from points (0, 0), (0.55, 0.055), (0.675, 0.19), (1, 1) */
    EaseInCubic,
    /* bezier cubic curve from points (0, 0), (0.645, 0.045, 0.355, 1), (1, 1) */
    EaseOutCubic,
    /* bezier cubic curve from points (0, 0), (0.645, 0.045, 0.355, 1), (1, 1) */
    EaseInOutCubic,
    /* bezier cubic curve from points (0, 0), (0.895, 0.03), (0.685, 0.22), (1, 1) */
    EaseInQuart,
    /* bezier cubic curve from points (0, 0), (0.165, 0.84), (0.44, 1), (1, 1) */
    EaseOutQuart,
    /* bezier cubic curve from points (0, 0), (0.77, 0), (0.175, 1), (1, 1) */
    EaseInOutQuart,
    /* bezier cubic curve from points (0, 0), (0.755, 0.05), (0.855, 0.06), (1, 1) */
    EaseInQuint,
    /* bezier cubic curve from points (0, 0), (0.23, 1), (0.32, 1), (1, 1) */
    EaseOutQuint,
    /* bezier cubic curve from points (0, 0), (0.86, 0), (0.07, 1), (1, 1) */
    EaseInOutQuint,
    /* bezier cubic curve from points (0, 0), (0.95, 0.05), (0.795, 0.035), (1, 1) */
    EaseInExpo,
    /* bezier cubic curve from points (0, 0), (0.19, 1), (0.22, 1), (1, 1) */
    EaseOutExpo,
    /* bezier cubic curve from points (0, 0), (1, 0), (0, 1), (1, 1) */
    EaseInOutExpo,
    /* bezier cubic curve from points (0, 0), (0.6, 0.04), (0.98, 0.335), (1, 1) */
    EaseInCirc,
    /* bezier cubic curve from points (0, 0), (0.075, 0.82), (0.165, 1), (1, 1) */
    EaseOutCirc,
    /* bezier cubic curve from points (0, 0), (0.785, 0.135), (0.15, 0.8), (1, 1) */
    EaseInOutCirc,
    /* bezier cubic curve from points (0, 0), (0.6, -0.28), (0.735, 0.045), (1, 1) */
    EaseInBack,
    /* bezier cubic curve from points (0, 0), (0.175, 0.885), (0.32, 1.275), (1, 1) */
    EaseOutBack,
    /* bezier cubic curve from points (0, 0), (0.68, -0.55), (0.265, 1.55), (1, 1) */
    EaseInOutBack,
    /* curve of elastic progression */
    EaseInElastic,
    /* curve of elastic regression */
    EaseOutElastic,
    /* curve of elastic progression then regression */
    EaseInOutElastic,
    /* curve of bounce progression */
    EaseInBounce,
    /* curve of bounce regression */
    EaseOutBounce,
    /* curve of bounce progression then regression */
    EaseInOutBounce
  };

  /**
   * Easing type of the camera
   * @useenum{EasingType}. Default is EaseInOutQuint.
   */
  SoSFEnum easingType;

  /**
   * OneShot engine activated when the internal camera moves
   * Used to add a sensor on the OneShot engine.
   */
  SoSFBool oneShotActive;

  /**
   * OpacityDistanceMin modifies the distance between the cursor and the center of the viewing cube
   * below which the viewing cube's opacity is #opacityMax.
   * The value is expressed in viewing cube's size.
   * The value is >= 0.
   *
   * Default is 0.5.
   */
  SoSFFloat opacityDistanceMin;

  /**
   * OpacityDistanceMax modifies the distance between the cursor and the center of the viewing cube
   * above which the viewing cube's opacity is #opacityMin.
   * The value is expressed in viewing cube's size.
   * The value is >= 0.
   *
   * Default is 1.5.
   */
  SoSFFloat opacityDistanceMax;
};
