/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/



#ifndef _SO_WIN_GL_WIDGET_H_
#define _SO_WIN_GL_WIDGET_H_


#include <Inventor/Win/SoWinBeginStrict.h>

#include <Inventor/Win/SoWinComponent.h>
#include <Inventor/components/SoGLGraphicConfigTemplate.h>
#include <Inventor/components/SoGLGraphicDevice.h>
#include <Inventor/components/SoGLGraphicConfig.h>

#include <Inventor/devices/SoGLContext.h>

#include <Inventor/nodes/SoFullSceneAntialiasing.h>

#include <Inventor/Gui/SoGuiGLWidget.h>

#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4251)
#endif

class SoAntialiasingParameters;

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoWinGLWidget
//
//
//////////////////////////////////////////////////////////////////////////////

/**
 * @VSGEXT Component for OpenGL rendering.
 * 
 * @ingroup Win
 * 
 * @DESCRIPTION
 *   This abstract base class provides a C++ wrapper around an OpenGL drawing window.
 *   It allows OpenGL rendering to be performed within a Windows window and is used
 *   by the SoWinRenderArea. SoWinGLWidget uses a parent window with two separate
 *   OpenGL windows (one for single buffering and one for double buffering), with
 *   routines to return the appropriate windows.
 *   
 *   Subclasses only need to redefine the redraw() routine for rendering and
 *   processEvent() routine if they are interested in receiving Windows
 *   messages.
 *
 * @SEE_ALSO
 *    SoWinComponent,
 *    SoWinRenderArea
 * 
 * 
 */
class INVENTORW_API SoWinGLWidget : public SoWinComponent 
{
 public:
   
    /** FloatColorBufferSize */
   enum FloatColorBufferSize 
   {
    /**
     * 16-bit rendering per component.
     */
    FLOAT_16_COLOR_BUFFER = SoGuiGLWidget::FLOAT_16_COLOR_BUFFER,
    /**
     * 32-bit rendering per component.
     */
    FLOAT_32_COLOR_BUFFER = SoGuiGLWidget::FLOAT_32_COLOR_BUFFER
  };

  /**
   * Gets the normal GL window, which is
   * needed as an argument to SbGlContextHelper::makeCurrent() when drawing in the normal
   * planes.
   * 
   * Note: This should @B not @b be cached by users because it will change as
   * single/double buffering changes.
   */
  virtual Window getNormalWindow();
    
  /**
   * Gets the normal context, which is
   * needed as an argument to SbGlContextHelper::makeCurrent() when drawing in the normal
   * planes.
   * 
   * Note: This should @B not @b be cached by users because it will change as
   * single/double buffering changes.
   */
  GLXContext  getNormalContext() { return m_contextNormal->getGLContext(); }

  virtual SoGLContext* getNormalSoContext() { return m_contextNormal.ptr(); }

  /**
   * Returns the device context (which is needed for SbGlContextHelper::makeCurrent).
   */
  Hdc getNormalDC() const;

#ifndef HIDDEN_FROM_DOC
  // Jim's thing for now, not really public
  Hdc copyWindowBits() ;
#endif // HIDDEN_FROM_DOC

  /**
   * By default the GLWidget "steals" focus whenever the cursor moves over it. This
   * roughly simulates the UNIX/X "focusFollowsCursor" behavior (which has no
   * equivalent in Win32) and allows the Ctrl and Shift keys to be detected for
   * 1-button mouse behaviors. Sometimes you want to this off however, like when
   * you're typing in the zoom field.
   */
  void setStealFocus( SbBool onOrOff ) { stealFocus = onOrOff; };
    
  /**
   * Gets the normal window handle, which is
   * needed as an argument to SbGlContextHelper::makeCurrent() when drawing in the normal 
   * planes.
   * 
   * Note: This should @B not @b be cached by users because it will change as
   * single/double buffering changes.
   */
  SoWidget getNormalWidget() const;
    
  /**
   * Specifies the exact pixel format descriptor for the normal
   * window. This allows the user to create all possible pixel format
   * descriptors supported by OpenGL. The PIXELFORMATDESCRIPTOR structure should be a
   * valid OpenGL pixel format descriptor returned by ChoosePixelFormat(). (The
   * methods for setting the pixel format descriptor are virtual so that derived
   * classes can know when the pixel format descriptor is changing.)
   */
  virtual void setNormalVisual( XVisualInfo* vis );

  /**
   * Returns the pixel format descriptor for the normal window.
   */
  XVisualInfo* getNormalVisual();

  /**
   * Sets the current pixel format. This method allows an application to
   * set a pixel format that has extended attributes that can't be specified in
   * a PFD struct. The specified format must support OpenGL and drawing to a window.
   * 
   * You can also use environment variable OIV_FORCE_PIXEL_FORMAT to request a pixel
   * format.
   *
   * Note: OIV_FORCE_PIXEL_FORMAT can also be used to set special convenience
   * values (-1, -2, etc.) used for troubleshooting. However, these values are
   * not actually valid pixel formats and are not supported by #setPixelFormat.
   */
  virtual void setPixelFormat( int format );

  /**
   * Returns the current pixel format. This method allow an application to
   * query a pixel format that has extended attributes that can't be specified in
   * a PFD struct. 
   */
  int getPixelFormat();

  /** 
    * @copydoc SoSceneManager::setAntialiasing(const float, const SoSceneManager::AntialiasingMode)
    */
  void setAntialiasing(const float quality, const SoSceneManager::AntialiasingMode mode = SoSceneManager::AUTO);

  /** 
    * @copydoc SoSceneManager::setAntialiasing(SoAntialiasingParameters*)
    */
  void setAntialiasing(SoAntialiasingParameters* advancedParameters);

  /** 
    * @copydoc SoSceneManager::getAntialiasingQuality()
    */
  float getAntialiasingQuality() const;

  /** 
    * @copydoc SoSceneManager::getAntialiasingMode()
    */
  SoSceneManager::AntialiasingMode getAntialiasingMode() const;

  /** 
    * @copydoc SoSceneManager::getAntialiasingParameters
    */
  SoAntialiasingParameters* getAntialiasingParameters() const;

  /**
   * Routine that dynamically changes between single and double buffering. Default
   * is double buffer off. (The SoWinRenderArea subclass makes it double buffer by
   * default.)
   */
  void setDoubleBuffer( SbBool onOrOff );

  /**
   * Returns whether double buffering is on or off.
   */
  SbBool isDoubleBuffer() { return m_guiGLWidget->isDoubleBuffer(); }
    
  /**
   * Included for portability only.
   */
  void setBorder( SbBool onOrOff );

  /**
   * Included for portability only.
   */
  int getBorderSize() { return m_guiGLWidget->getBorderSize(); }

  /**
   * Included for portability only.
   */
  SbBool isBorder() const { return m_guiGLWidget->isBorder(); }
    
  /**
   * Sets drawing to the front buffer. Controls drawing to the front buffer when
   * an obscured portion of the window is exposed. Default: FALSE.
   */
  void setDrawToFrontBufferEnable( SbBool enableFlag );

  /**
   * Returns whether drawing to the front buffer is enabled.
   */
  SbBool isDrawToFrontBufferEnable() const { return m_guiGLWidget->isDrawToFrontBufferEnable(); }

  /**
   * Sets the current cursor. If you are using a viewer class, the viewer will
   * automatically change the cursor depending on the viewer mode. These changes will
   * override the cursor set with this method unless you also call
   * SoWinViewer::setCursorEnabled(FALSE). 
   * [OIVNET-WRAPPER PROPERTY{Cursor},SETTER,SINCE{9.2.1}]
   * [OIVNET-WRAPPER-ARG ALLOC_REF{Normal}]
   */
  void setCursor( Cursor newCursor );

  /**
   * Returns the current cursor. 
   * [OIVNET-WRAPPER PROPERTY{Cursor},GETTER,SINCE{9.2.1}]
   */
  Cursor getCursor();

  /**
   * Makes the normal rendering context the current context.
   * Equivalent to makeNormalCurrent() call.
   */
  virtual SbBool bindNormalContext();

  /**
   * unbind the current context (previously bind with bindNormalContext );
   */
  virtual SbBool unbindNormalContext();


  /**
   * Swaps the normal front and back buffers.
   */
  virtual SbBool swapNormalBuffers();

  /**
   * Gets the current graphics configuration template.
   */
  SoGLGraphicConfigTemplate* getGraphicConfigTemplate();

  /**
   * Sets a new graphics configuration template.
   */
  void setGraphicConfigTemplate( SoGLGraphicConfigTemplate* gTemplate );

  /**
   * Enables/disables floating point rendering using 16- or 32-bit components.
   * If TRUE, Open Inventor will automatically render to a floating point color
   * buffer.
   *
   * To determine if floating point rendering was successfully enabled,
   * use #getFloatingColorBuffer.
   *
   * Using floating point rendering can improve image quality, particularly when
   * many objects are being blended.
   */
  virtual void setFloatingColorBuffer( SbBool enable, 
                                       FloatColorBufferSize size = FLOAT_16_COLOR_BUFFER );

  /**
   * Returns TRUE if floating point rendering is used and its precision.
   */
  virtual void getFloatingColorBuffer( SbBool& enable, FloatColorBufferSize& size );

  /**
   * Returns TRUE if floating point rendering is available.
   */
  static SbBool isFloatingColorBufferSupported();

  /** 
   * Save a snapshot of the current image displayed in the viewer.
   * The image is read back from the OpenGL framebuffer and will be the same size
   * as the viewer's drawing window. Returns true if successful.
   *
   * Notes:
   * - Depending on the operating system and graphics hardware, it is possible that 
   *   the image could be incomplete if the viewer window is overlapped by other windows.  
   * - The supported image file formats are: BMP (Windows only), JPEG, PNG and TIFF. @BR
   *   The image file format to write is determined by the extension of the specified filename,
   *   e.g. ".png" for PNG format.  
   * - For BMP and JPEG formats an RGB image is written.
   * - For PNG and TIFF formats, an RGBA (RGB plus alpha channel) image is always written.
   * - Writing an RGBA image means the image background is transparent. Actually an RGBA snapshot
   *   can not be saved without transparency. To get the displayed image without transparency, use 
   *   the BMP or the JPEG format.
   * - To generate an image with a different size or different options, use SoOffscreenRenderArea instead.
   * - The specified filename @I must@i include a directory or nothing will be written.
   *   Can be simply "./filename.png", but just "filename.png" won't work.
   * 
   * @param filename Fully qualified file path for the snapshot file. @BR
   * The specified filename must end with one of the extensions supported by SoJPEGImageRW 
   * (jpg, jpeg, ...), SoPNGImageRW (png), SoBMPImageRW (bmp), or SoTIFFImageRW (tif, ...).
   * @param overwrite If true, overwrite any existing file with the same name (default is true).
   */
  bool saveSnapshot( const SbString& filename, bool overwrite = true );

SoEXTENDER public:

  // Starting with version 3.0, subclasses should no longer make direct
  // calls to glXMakeCurrent, glXSwapBuffers and glFlush.
  // For RemoteRendering we need to know when MakeCurrent is called
  // (marking the beginning of a frame) and when SwapBuffers or glFlush
  // is called (marking the end of a frame).
  // Subclasses should always use these generic methods.
  void   flush();

SoINTERNAL public:

  SoGuiGLWidget* getGuiGLWidget() const;
    
  // Synchronise current properties from this object with the current configuration.
  void syncFromCurrentConfig();
  // Synchronise the current template with current properties from this object.
  void syncToCurrentConfigTemplate( PIXELFORMATDESCRIPTOR* newPFD );

  SbBool currentGraphicConfigTemplateIsUsed;
  SoGLGraphicConfigTemplate currentGraphicConfigTemplate;

  // Gets the current graphic device.
  SoGLGraphicDevice* getGraphicDevice();

  // Gets the current graphic configuration.
  SoGLGraphicConfig* getGraphicConfig();

  virtual void posChanged( const SbVec2i32&, const SbVec2i32& );

  // These variables are  public for retro compatibility purposes only.
  // Do not use them directly but call their accessors instead;
  SbBool drawToFrontBuffer;

  // These two function are used by SoGui, to call the system dependent version of 
  // these functions. data is a SoWinGLWidget handle.
  static SbBool swapNormalBuffersCB(void *data);
  static SbBool bindNormalCurrentCB(void *data);
  static SbBool unbindNormalCurrentCB(void *data);
  static void* getNormalWindowCB(void *data);

protected:

  // Update the current device and disable the current configuration.
  // If the given hdc is the same than the current device, the current
  // device is kept and the current configuration is updated with the 
  // current template.
  // If the given hdc is NULL, the current device and configuration are
  // freed and set to NULL.
  void updateCurrentDeviceAndConfig( HDC hdc );
    
  SbBool currentGraphicConfigIsUsed;
  SoGLGraphicConfig* currentGraphicConfig;
  SbBool currentGraphicDeviceIsUsed;
  SoGLGraphicDevice* currentGraphicDevice;

  // Subclasses can pass in a bitwise OR specifying the GL modes
  // (e.g. SO_GLX_RGB | SO_GLX_DOUBLE | SO_GLX_ZBUFFER)
  // If buildNow is FALSE, this will not build its widget tree until
  // buildWidget() is explicity called; else, buildWidget() is called here.
  SoWinGLWidget( SoWidget parent = NULL,
                 const char* name = NULL, 
                 SbBool buildInsideParent = TRUE, 
                 int glModes = SO_GLX_RGB, 
                 SbBool buildNow = TRUE,
                 SbBool sync = TRUE );

  SoWinGLWidget( SoWidget parent,
                 const char* name, 
                 SbBool buildInsideParent, 
                 int glModes, 
                 SbBool buildNow,
                 SbBool sync,
                 SoGuiGLWidget* guiGLWidget );

  virtual ~SoWinGLWidget();
    
  // subclasses MUST redefine redraw() to draw in the normal bit planes.
  virtual void redraw() = 0;
  virtual void processEvent( XAnyEvent* anyevent );
    
  // subclasses can redefine these to do something useful.
  // initGraphic() is called whenever a GLX window gets created
  // sizeChanged() is called whenever the window changes size
  // widgetChanged() is called whenever the widget is changed (i.e. at
  //    initialization or after switching from single->double buffer)
  virtual void initGraphic();
  virtual void sizeChanged( const SbVec2s& newSize );
  virtual void widgetChanged( SoWidget newWidget );
     
  // sets/gets the size of the glx widget(s) - Note this size could be
  // different from the SoWinComponent::getSize() method which return
  // the size of the component, not necessary the same as the glx widget
  // window (because of extra stuff like the decoration in the viewers).
  void setGlxSize( SbVec2s newSize );
  const SbVec2s& getGlxSize() const { return m_guiGLWidget->getGlxSize(); }
    
  // subclasses can use this routine as the event handler for any
  // devices they wish to get input from.
  static void eventHandler( SoWidget w, SoWinGLWidget* p, XAnyEvent* e, Boolean* b );
    
  // set/get stereo buffering visual. This routine (like setDoubleBuffer)
  // can be called interactively, althought slower since a new window
  // with the needed visual is always created on the fly.
  void setStereoBuffer( SbBool flag );
  SbBool isStereoBuffer() const;
    
  // returns TRUE if main window is in rgb mode, else FALSE for color index mode
  SbBool isRGBMode() const;
    
  // returns the display lists share group for given context:
  int getDisplayListShareGroup( SoGLContext* ctx );

  // set when color index is used
  Colormap getColorMap() const
  { return m_guiGLWidget->getColorMap(); }


  SbBool waitForExpose; // prevent redraws until an expose is received
    
  // make those methods protected so enable the SoWinRenderArea to use them
  // directly when it needs to build the widget and get its resources.
  SoWidget buildWidget( SoWidget parent );
  SoWidget getGlxMgrWidget() { return mgrWidget; }

  // _WIN32
  // Add some convenience functions that reduce the number of ifdef's in
  // viewers by hiding these platform dependent operations.
  // We should add these to the UNIX version too.
  void changeCursor( Cursor newCursor );

  // Set/get thread this context belongs to
  DWORD getThreadId() { return dwThreadId; };
  void setThreadId( DWORD id ) { dwThreadId=id; };

  ////////////////////////////////////////////////////////////////////////////////

  virtual void onExpose();

  void destroyNormalWindows(SbBool normalCall=TRUE);

 private:

   void constructorCommon( int glModes, SbBool buildNow, SoWidget parent );

   static SbBool onGLFormatChangeCallback( SoGLFormat& format, void* userData );

   HWND createWindow( SoGLFormat* format );

   void destroyWindow( HWND handle );

   void initGLXWidget( SoWidget glx, SoGLFormat* format );
  
  // local vars
  SoWidget mgrWidget;
  SoWidget m_widgetNormal;
  
  SoRef<SoGLContext> m_contextNormal;

  SoGuiGLWidget* m_guiGLWidget; //Implementation class for SoXxGLWidget

  // Note for _WIN32:
  // 1) All the "SoWidget" values above are actually type "HWND".
  // 2) The "GLXContext" values above are actually type "HGLRC".

  // For _WIN32 we need a GL context *and* a device context for each
  // GL drawing window (we don't need one for the mgr because we don't
  // plan to ever draw in it).  These correspond to the "ctx..." vars.
  // Note: This plan depends on use of the CS_OWNDC style.
  Hdc m_hdcNormal;

  // For _WIN32 we also need a logical color palette for each drawing
  // area if we happen to running on an 8bit device...
  HPALETTE palNormal;

  // For _WIN32 we also need to remember our parent and our
  // "ancestor" (the toplevel window that we're descended from).
  SoWidget parent;
  SoWidget ancestor;

  // For _WIN32 we have to simulate the "focus follows pointer" behavior
  // that X provides for free.  Otherwise the app would have to be
  // responsible for giving focus to the GL window and/or the user
  // would have to keep clicking to get focus in the GL window.  Focus
  // is required for the 1-button mouse viewer behaviors like Ctrl-Left
  // Button and so on.  This variable tracks whether we have focus
  // based on the WM_SETFOCUS/WM_KILLFOCUS messages (see glxWindowProc).
  int haveFocus;

  // This flag tracks whether "focus follows pointer" is enabled.
  int stealFocus;

  // For _WIN32 we have to simulate the behavior of X where getting a
  // mouse button down event guarantees you will also get the
  // corresponding button up event.  Otherwise examiner viewer windows
  // get out of sync if a mouse button is released outside the window.
  UINT mouseCaptured;

  // For _WIN32 we have to know the handle of the cursor that's supposed
  // to be displayed currently in the drawing window.  The explanation
  // is too long to fit here...  :-)  OK, actually it's because the
  // viewers want to change the cursor periodically but we can't change
  // the cursor in the class defn or it will change for *all* the GL
  // windows (yuck).  So we have to process the WM_SETCURSOR message in
  // the glxWindowProc.
  HCURSOR currentCursor;

  // We also need (globally for the class) the default cursor to use.
  static HCURSOR defaultCursor;

  DWORD dwThreadId; // thread to which this context belongs -- mmh Apr-99

  int* attribList;
    
  // specify if sizeChanged() should be called when an expose event
  // is received (as opposed to when a resize event is received, where
  // the window hasn't yet been maped to the new size).
  // ??? a GlxDraw bug ?
  SbBool windowResized;

  bool m_isMouseTracking;
    
  // creates a GLX widget of the correct current type and get the current
  // set of windows, color maps, etc...
  void buildNormalGLXWidget( SoGLFormat* format = NULL );
  void destroyGLXWidget( SoWidget& w, SoRef<SoGLContext>& ctx, SbBool normalWindow );
    
  // callbacks from glx widget
  static void ginitCB( SoWidget w, SoWinGLWidget* p, XtPointer );
  static void exposeCB( SoWidget w, SoWinGLWidget* p, XtPointer ptr );
  static void resizeCB( SoWidget w, SoWinGLWidget* p, XtPointer ptr );
  static void mgrStructureNotifyCB( SoWidget w, SoWinGLWidget* p, XAnyEvent* e, Boolean* b );

  // Window proc for SoWinGL "manager widget" windows
  static LRESULT CALLBACK mgrWindowProc( Hwnd hwnd, UINT message,
                                         WPARAM wParam, LPARAM lParam );

  // Window proc for SoWinGL drawing windows
  static LRESULT CALLBACK glxWindowProc( Hwnd hwnd, UINT message,
                                         WPARAM wParam, LPARAM lParam );
};

inline void 
SoWinGLWidget::posChanged( const SbVec2i32&, const SbVec2i32& )
{};


#include <Inventor/Win/SoWinEndStrict.h>

#ifdef _WIN32
#pragma warning(pop)
#endif

#endif // _SO_WIN_GL_WIDGET_H_


