/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#ifndef _SO_WIN_RENDER_AREA_H_
#define _SO_WIN_RENDER_AREA_H_

#include <Inventor/Win/SoWinBeginStrict.h>
#include <windows.h>
#include <Inventor/Win/SoWinDef.h>

#include <climits>

#include <Inventor/SbColor.h>
#include <Inventor/sensors/SoNodeSensor.h>
#include <Inventor/SoSceneManager.h>
#include <Inventor/Win/SoWinGLWidget.h>
#include <Inventor/actions/SoGLRenderAction.h>

#include <Inventor/nodes/SoCamera.h> // stereo settings
#include <Inventor/elements/SoStereoElement.h>
#include <Inventor/nodes/SoFullSceneAntialiasing.h>
#include <Inventor/SbElapsedTime.h>


#include <Inventor/Gui/SoGuiRenderArea.h>

class SoHandleEventAction;
class SoWinDevice;
class SoNode;
class SoWinMouse;
class SoWinKeyboard;
class SoSelection;
class SoWinRenderArea;
class SoGuiAlgoViewers;
class SoAntialiasingParameters;

/**
 * @memberof SoWinRenderArea
 *
 * [OIV-WRAPPER NAME{EventCallback}]
 */
typedef SbBool SoWinRenderAreaEventCB( void* userData, XAnyEvent* anyevent );

/**
 * @memberof SoWinRenderArea
 *
 * [OIV-WRAPPER NAME{RenderCallback}]
 */
typedef SbBool SoWinRenderAreaRenderCB( void* userData, SoWinRenderArea* rendArea );

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoWinRenderArea
//
//  Class to do Inventor rendering in a GLX Motif widget.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * @VSGEXT Component for rendering Open Inventor scene graphs.
 *
 * @ingroup Win
 *
 * @DESCRIPTION
 *   This class provides Open Inventor rendering and message handling inside a
 *   Windows window. There is a routine to specify the scene to render. The scene is
 *   automatically rendered whenever anything under it changes (a data sensor is
 *   attached to the root of the scene), unless explicitly told not to do so (manual
 *   redraws). Users can also set Open Inventor rendering attributes such as the
 *   transparency algorithm, antialiasing on or off, etc. This class employs an
 *   SoSceneManager to manage rendering and message handling.
 *
 *   Windows messages that occur in the render area can be handled by the application,
 *   by the viewer (if this is really a viewer), or by the nodes in the scene graph.
 *   When a message occurs, it is first passed to the application message \if_dotnet delegate \else callback \endif
 *   function registered with the setEventCallback() method on
 *   SoWinRenderArea. If this function does not exist or returns FALSE, the message
 *   is either used directly by the viewer or translated to an SoEvent for further
 *   scene graph processing. If the viewer does not handle the message, the SoEvent is
 *   sent to the scene graph by way of an SoHandleEventAction.
 *
 * @SEE_ALSO
 *    SoWinGLWidget,
 *    SoWinComponent,
 *    SoWinViewer,
 *    SoSceneManager,
 *    SoBoxHighlightRenderAction,
 *    SoHaloHighlightRenderAction,
 *    SoLineHighlightRenderAction
 *
 *
 */
class INVENTORW_API SoWinRenderArea : public SoWinGLWidget
{
 public:

  /**
   * Constructor which is passed arguments which tell it whether to register the
   * mouse and keyboard devices by default (SoWinMouse and SoWinKeyboard).
   * @NOTES
   * On Windows, the constructor will not create a new top level window -- you
   * must pass a valid window handle for the @B parent @b parameter and pass TRUE
   * for @B buildInsideParent @b.
   */
  SoWinRenderArea( SoWidget parent = NULL,
                   const char* name = NULL,
                   SbBool buildInsideParent = TRUE,
                   SbBool getMouseInput = TRUE,
                   SbBool getKeyboardInput = TRUE );
  /**
   * Destructor.
   */
  ~SoWinRenderArea();

  /**
   * Sets the scene graph to be rendered in this component's window.
   * [OIVJAVA-WRAPPER CUSTOM_CODE]
   * [OIVNET-WRAPPER-ARG ALLOC_REF{Normal}]
   */
  virtual void setSceneGraph( SoNode *newScene );

  /**
   * Gets the scene graph to be rendered in this component's window.
   */
  virtual SoNode* getSceneGraph();

  /**
   * Registers interest in devices. When a device is registered, messages
   * from that device will be processed by the render area, and passed into the scene
   * graph. Messages from unregistered devices will be ignored.
   * [OIV-WRAPPER CUSTOM_CODE]
   */
  void registerDevice( SoWinDevice* d );
  /**
   * Unregisters interest in devices. When a device is registered, messages
   * from that device will be processed by the render area, and passed into the scene
   * graph. Messages from unregistered devices will be ignored.
   * [OIV-WRAPPER CUSTOM_CODE]
   */
  void unregisterDevice( SoWinDevice* d );

  /**
   * Sets the background color for this window. Default is black (0,0,0). @BR
   * @BR
   * The default value can be set using the environment variable
   * OIV_BACKGROUND_COLOR (3 floats representing RGB values of the colors separated by space).
   */
  void setBackgroundColor( const SbColor& c );

  /**
   * Gets the background color for this window.
   */
  SbColor getBackgroundColor() const { return m_guiRenderArea->getBackgroundColor(); }

  /**
   * Sets the window background color when in color index mode. Default is black
   * (index 0)).
   */
  void setBackgroundIndex( int index ) { m_guiRenderArea->setBackgroundIndex( index ); }

  /**
   * Gets the window background color when in color index mode.
   */
  int getBackgroundIndex() const { return m_guiRenderArea->getBackgroundIndex(); }

  /**
   * Sets the colors to use when displaying in color index mode. This will load the
   * color map with the given colors at the starting index.
   * [OIV-WRAPPER-ARG IN,IN,ARRAY]
   */
  void setColorMap( int startIndex, int num, const SbColor* colors );

  /**
   * Sets viewport region to use for rendering.
   */
  void setViewportRegion( const SbViewportRegion& newRegion )
  { m_guiRenderArea->setViewportRegion( newRegion ); }

  /**
   * Gets current viewport region to use for rendering.
   */
  const SbViewportRegion &getViewportRegion() const
    { return m_guiRenderArea->getViewportRegion(); }

  /**
   * Sets the algorithm for rendering transparent objects. 
   * Default is NO_SORT. See SoGLRenderAction for possible transparency types.
   * See also SoGLRenderAction::TransparencyType.
   *
   * Note: When using OPAQUE_FIRST, SORTED_OBJECT or SORTED_PIXEL transparency, the depth buffer is
   *   not updated (depth buffer writes are disabled) while rendering transparent objects.
   *   As a result complex 3D shapes may not be rendered correctly.
   */
  void setTransparencyType( SoGLRenderAction::TransparencyType type );

  /**
   * Gets the algorithm for rendering transparent objects. See
   * SoGLRenderAction for possible transparency types.
   */
  SoGLRenderAction::TransparencyType  getTransparencyType() const
    { return m_guiRenderArea->getTransparencyType(); }

  /**
   * Sets fast editing save policy to use when rendering. The default is WHEN_NEEDED.
   * Valid values are DISABLE, EACH_FRAME, and WHEN_NEEDED. See SoSeparator.
   * 
   * If fastEditDelayedObjects is set to TRUE, delayed objects won't be redrawn
   * when editing the scene graph. It means that composition between delayed transparent objects
   * and fast edit scene graph won't be correct but redrawing may be much faster. Default is FALSE.
   */
  void setFastEditSavePolicy( SoGLRenderAction::FastEditSavePolicy policy,
                              SbBool fastEditDelayedObjects = FALSE);

  /**
   * Returns fast editing save policy used when rendering.
   */
  SoGLRenderAction::FastEditSavePolicy  getFastEditSavePolicy() const
    { return m_guiRenderArea->getFastEditSavePolicy(); }

  /**
   * Enables or disables the invalidation of render caches.
   * - When set to ALWAYS, the caches are invalidated for each SoSeparator node before
   *   its children are traversed. No new caches will be built. This value forces
   *   all nodes to be visited during each render traversal.
   *
   * - When set to ONCE, the caches are invalidated for each SoSeparator node before
   *   its children are traversed. The
   *   invalidate cache mode is automatically changed to OFF at the end of the traversal.
   *
   * - When set to OFF (default), caches are managed by their respective render caching
   *   nodes in the usual way.
   *
   * This method is useful to force all nodes to be visited during render traversal.
   */
  void setInvalidateCacheMode( SoGLRenderAction::InvalidateCacheMode icm );

  /**
   * Returns the current cache invalidation mode.
   */
  SoGLRenderAction::InvalidateCacheMode getInvalidateCacheMode();

  /**
   * Enables/prevents window clearing from happening before a rendering starts (default
   * is clear TRUE). This can be useful to limit flickering when doing single
   * buffering and geometry covers the entire window (used in the material editor).
   * Also controls whether the depth buffer (sometimes called the Z buffer) is
   * cleared before rendering.
   */
  void setClearBeforeRender( SbBool trueOrFalse, SbBool zbTrueOrFalse = TRUE )
  { m_guiRenderArea->setClearBeforeRender( trueOrFalse, zbTrueOrFalse ); }

  /**
   * Queries whether the window will be cleared before rendering starts.
   */
  SbBool isClearBeforeRender() const { return m_guiRenderArea->isClearBeforeRender(); }

  /**
   * Queries whether the depth buffer (sometimes called the Z buffer) will be
   * cleared before rendering starts.
   */
  SbBool isClearZBufferBeforeRender() const { return m_guiRenderArea->isClearZBufferBeforeRender(); }

  /**
   * The render area will automatically redraw whenever something in the scene graph
   * changes. Passing FALSE will disable this feature.
   * NOTE: the render area will always
   * redraw in response to window system events (e.g. resize, exposure)
   * regardless of the setting of the auto redraw flag.
   */
  void setAutoRedraw( SbBool trueOrFalse );

  /**
   * Queries whether the render area will automatically redraw whenever something
   * in the scene graph changes.
   */
  SbBool isAutoRedraw() const { return m_guiRenderArea->isAutoRedraw(); }

  /**
   * Sets the priority of the redraw sensor.
   */
  void setRedrawPriority( unsigned long priority ) { m_guiRenderArea->setRedrawPriority(priority); }

  /**
   * Gets the priority of the redraw sensor.
   */
  unsigned long getRedrawPriority() const { return m_guiRenderArea->getRedrawPriority(); }

  /**
   * Gets the default priority number of the redraw sensor.
   */
  static unsigned long getDefaultRedrawPriority() { return SoSceneManager::getDefaultRedrawPriority(); }

  /**
   * Calling this forces the render area to be redrawn now. It is not necessary to
   * call this method if auto redraw is enabled (which is the default).
   */
  void render() { redraw(); }

  /**
   * Schedules a redraw to happen sometime soon (as opposed to immediately). This can
   * be used to compress multiple redraws.
   */
  void scheduleRedraw();

  /**
   * Call this convenience method to have this render area redraw whenever the
   * selection list changes in the passed node. This is useful if using a highlight
   * render action like the SoHaloHighlightRenderAction to correctly render whenever
   * the selection changes. Pass NULL to turn this off.
   *
   * This call increases the ref counter of the given SoSelection node, it will be
   * automatically decreased when the selection node is replaced by another one, or if
   * it is turned off.
   */
  void redrawOnSelectionChange( SoSelection* s );

  /**
   * Windows messages which occur in the render area window are either directly
   * handled by the viewer (when this is really a viewer) or automatically translated
   * to SoEvents, then passed into the scene graph (via the SoHandleEventAction) so
   * that live scene graph objects can handle the message (when viewers are not in
   * viewing mode).
   *
   * This method allows the application to register a \if_dotnet delegate \else callback \endif for
   * handling messages that occur in the window, instead of sending them to the
   * viewers or down the graph. The \if_dotnet delegate \else callback \endif is passed the Windows message, and
   * should return TRUE if it handled the message. In this case the message will not be handled
   * by the viewer and will not be sent to the scene graph. If the \if_dotnet delegate \else callback \endif
   * returns FALSE, then the message will be handled by the view and/or sent to the scene graph.
   *
   * The following Windows events are sent to setEventCallback:
   * - WM_LBUTTONDOWN@BR
   * - WM_MBUTTONDOWN@BR
   * - WM_RBUTTONDOWN@BR
   * - WM_LBUTTONUP@BR
   * - WM_MBUTTONUP@BR
   * - WM_RBUTTONUP@BR
   * - WM_MOUSEMOVE@BR
   * - WM_MOUSEWHEEL@BR
   * - WM_LBUTTONDBLCLK@BR
   * - WM_MBUTTONDBLCLK@BR
   * - WM_RBUTTONDBLCLK@BR
   * - WM_SETFOCUS@BR
   * - WM_KILLFOCUS@BR
   * - WM_COMMAND@BR
   * - WM_KEYDOWN@BR
   * - WM_SYSKEYDOWN@BR
   * - WM_KEYUP@BR
   * - WM_SYSKEYUP@BR
   * - WM_SIZE@BR
   * - WM_PAINT@BR
   * [OIV-WRAPPER-CUSTOM-CODE]
   */
  void setEventCallback( SoWinRenderAreaEventCB* fcn, void* userData = NULL );

  /**
   * Sets the normal scene manager.
   *
   * Note: For convenience most of the SoSceneManager methods have already been added
   * to this class.
   */
  void setSceneManager( SoSceneManager* sm );

  /**
   * Gets the normal scene manager.
   *
   * Note: For convenience most of the SoSceneManager methods have already been added
   * to this class.
   */
  SoSceneManager* getSceneManager() const { return m_guiRenderArea->getSceneManager(); }

  /**
   * Sets the GL render action to use. This is used for example to set selection
   * highlighting with the SoBoxHighlightRenderAction, SoHaloHighlightRenderAction
   * and SoLineHighlightRenderAction classes.
   *
   * Note: Properties of the previous render action, for example transparency type,
   * are not automatically copied to the new render action.  The application must
   * explicitly set the desired properties of the new render action.
   */
  void setGLRenderAction( SoGLRenderAction* ra );

  /**
   * Gets the current GL render action.
   */
  SoGLRenderAction* getGLRenderAction() const { return m_guiRenderArea->getGLRenderAction(); }

  /**
   * Returns the information needed to make OpenGL render contexts
   * share OpenGL objects, for example, display lists and texture objects.
   */
  const SbGLShareContext getShareContext();

  /**
   * Specifies a function to be called after the Open Inventor render
   * traversal and immediately before the OpenGL buffer swap.
   *
   * Generally the application should not modify Open Inventor state in these
   * callbacks, but they can be useful for special effects using OpenGL calls.
   *
   * When the \if_dotnet delegate \else callback \endif function is called, Open Inventor has completed normal
   * rendering (including delayed transparent objects, multi-pass, etc),
   * but no "end of frame" calls (glFlush, glFinish, SwapBuffers...) have
   * been made.  The function should return TRUE if "end of frame" handling
   * has been done by the application (Open Inventor will do nothing in this
   * case).  If FALSE is returned, Open Inventor will do its normal end of
   * frame calls (normally calling SwapBuffers).
   */
  void setPostRenderCallback( SoWinRenderAreaRenderCB* fcn, void* userData = NULL )
    { appPostRenderCB = fcn; appPostRenderData = userData; }

  /**
   * Gets the post-render callback function and data.
   * Returns NULL if no callback has been specified by the application.
   * [OIV-WRAPPER-NOT-WRAP]
   */
  SoWinRenderAreaRenderCB* getPostRenderCallback( const void* &userData ) const
    { userData = appPostRenderData; return appPostRenderCB; }

  /** 
   * Sends the event to be processed by the renderArea.
   */
  void sendEvent( XAnyEvent* anEvent );

  /** 
   * Sets the recorder used for MPEG encoding. Default is none.
   *
   * @param recorder the recorder to be used.
   */
  virtual void setMPEGRecorder( SoMPEGRenderer* recorder )
  { m_guiRenderArea->setMPEGRecorder( recorder ); }

  /** 
   * Returns the recorder used for MPEG encoding.
   */
  virtual SoMPEGRenderer* getMPEGRecorder() const
  { return m_guiRenderArea->getMPEGRecorder(); }

SoEXTENDER public:
  SoWinRenderArea( SoWidget parent,
                   const char* name, 
                   SbBool buildInsideParent, 
                   SbBool getMouseInput,
                   SbBool getKeyboardInput,
                   SoGuiAlgoViewers* guiAlgos );

  //Equivalent to the SoExtender constructor
  SoWinRenderArea( SoWidget parent,
                   const char* name, 
                   SbBool buildInsideParent, 
                   SbBool getMouseInput,
                   SbBool getKeyboardInput, 
                   SbBool buildNow,
                   SbBool sync,
                   SoGuiAlgoViewers* guiAlgos );

  //Equivalent to the protected constructor
  SoWinRenderArea( SoWidget parent,
                   const char* name, 
                   SbBool buildInsideParent, 
                   SbBool getMouseInput,
                   SbBool getKeyboardInput, 
                   SbBool buildNow,
                   SoGuiAlgoViewers* guiAlgos );

  /**
   * Returns the viewer algorithm implementation class.
   */
  SoGuiAlgoViewers* getGuiAlgoViewers() const { return (SoGuiAlgoViewers*)m_guiRenderArea; }

SoINTERNAL public:

  SoGuiRenderArea* getGuiRenderArea() const;

  void setStereoMode( SoCamera::StereoMode stMode );
  void setStereoElement();

  virtual void setDoubleBuffer( SbBool db );

  virtual SbBool isInteractive() const;

  SoWinRenderArea( SoWidget parent,
                   const char* name,
                   SbBool buildInsideParent,
                   SbBool getMouseInput,
                   SbBool getKeyboardInput,
                   SbBool buildNow,
                   SbBool sync );

  // Retro compatibility only.
  // These members shouldn't be used directly, instead call their accessors
  float stereoBalance, stereoOffset;
  SbBool stereoAbsoluteAdjustments;
  SoCamera::StereoMode stereoMode;
  SbBool stereoReversed;

protected:

  //
  // This constructor takes a boolean whether to build the widget now.
  // Subclasses can pass FALSE, then call SoWinRenderArea::buildWidget()
  // when they are ready for it to be built.

  SoWinRenderArea( SoWidget parent,
                   const char* name,
                   SbBool buildInsideParent,
                   SbBool getMouseInput,
                   SbBool getKeyboardInput,
                   SbBool buildNow );

  //Constructors provided to avoid multiple instance of implementation objects
  //Equivalent to the public constructor
  SoWinRenderArea( SoWidget parent,
                   const char* name,
                   SbBool buildInsideParent,
                   SbBool getMouseInput,
                   SbBool getKeyboardInput,
                   SoGuiRenderArea* guiRenderArea );

  //Equivalent to the SoExtender constructor
  SoWinRenderArea( SoWidget parent,
                   const char* name,
                   SbBool buildInsideParent,
                   SbBool getMouseInput,
                   SbBool getKeyboardInput,
                   SbBool buildNow,
                   SbBool sync,
                   SoGuiRenderArea* guiRenderArea );

  //Equivalent to the protected constructor
  SoWinRenderArea( SoWidget parent,
                   const char* name,
                   SbBool buildInsideParent,
                   SbBool getMouseInput,
                   SbBool getKeyboardInput,
                   SbBool buildNow,
                   SoGuiRenderArea* guiRenderArea );

  // redraw() calls actualRedraw(), followed by swapbuffers if necessary.
  // actualRedraw will have the scene manager render the scene. Rendering
  // is broken up into two routines like this so that subclasses can
  // redefine or simply add to rendering (in actualRedraw) without having
  // to worry about being visible, seting up the window or
  // single/double buffer swapping.
  //
  virtual void redraw();
  virtual void actualRedraw();

  //
  // Redefine these to do Inventor-specific things
  //
  virtual void processEvent( XAnyEvent* anyevent );
  virtual void initGraphic();
  virtual void sizeChanged( const SbVec2s& );
  virtual void posChanged( const SbVec2i32&, const SbVec2i32& );
  virtual void widgetChanged( SoWidget );

  SoWidget buildWidget( SoWidget parent );

  // redefine these
  virtual SbString getDefaultWidgetName() const;
  virtual SbString getDefaultTitle() const;
  virtual SbString getDefaultIconTitle() const;

  // subclasses have access to the device list for event processing
  SbPList* m_deviceList;    // list of devices

  // application event callbacks variables
  SoWinRenderAreaEventCB *defaultAppEventHandler;
  void                   *defaultAppEventHandlerData;
  SoWinRenderAreaEventCB *appEventHandler;
  void                   *appEventHandlerData;
  // invoke the application event callback - returns what the app cb returns
  SbBool invokeAppCB( XAnyEvent* anyevent );
  SbBool processInventorEvent( XAnyEvent* anyevent );
  const SoEvent* translateAnyEvent( XAnyEvent* anyevent );

  // application pre/post render callback variables
  SoWinRenderAreaRenderCB *appPostRenderCB;
  void                    *appPostRenderData;

private:
#if defined(__linux__)
  XColor* m_mapColors; // saved colors
  int m_mapColorNum; // number of saved colors
#endif

  SoWinMouse* m_mouseDevice;
  SoWinKeyboard* m_keybdDevice;
  void reinstallDevices( SoWidget newWidget );

  static void selectionChangeCB( void* userData, SoSelection* s );
  SoWidget m_deviceWidget;

  // static callbacks
  SbBool m_firstEvent;

protected:
  static void windowEventCB( SoWidget w, SoWinRenderArea* wra, XAnyEvent* e, Boolean* b );
private:
  static void renderCB( void* v, SoSceneManager* sm );

  static void visibilityChangeCB( void* pt, SbBool visible );

  // this is called by constructors
  void constructorCommon( SbBool getMouseInput,
                          SbBool getKeyboardInput,
                          SbBool buildNow );

  // this is called by constructorCommon
  void constructorCommon2( SbBool getMouseInput,
                          SbBool getKeyboardInput,
                          SbBool buildNow );

  void activate();
  void deactivate();

  SbVec2i32 getWindowPosition() { return SbVec2i32( INT_MAX, INT_MAX ); }

  int inRedraw;

  SoGuiRenderArea* m_guiRenderArea; // Implementation class for SoXxRendeArea

  // this is the default implementation of the hasRenderAbortCallback passed to the scenemanager
  // at construction. Implementation is common to all created renderArea.
  // If an event is received to one renderArea, all renderArea of the process will consider it as
  // a candidate for abortion (only implemented on Windows)
  static SbBool s_abortRenderCallback(SoAction*,void*);

  SoAntialiasingParameters* m_accumulationParameters;

  bool m_firstStdRedraw;
  bool m_firstConnectedRedraw;
  SoTimerSensor* m_viewerUpdaterSensor;
  static void viewerUpdaterCB( void*, SoSensor* );
  void updateFirstFrame();

};

#include <Inventor/Win/SoWinEndStrict.h>

#endif /* _SO_WIN_RENDER_AREA_H_ */


