/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_SEARCH_ACTION_
#define  _SO_SEARCH_ACTION_

#include <Inventor/actions/SoSubAction.h>

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoSearchAction
//
//  For searching for specific nodes in scene graphs. Nodes can be
//  searched for by pointer, by type (exact or derived), by name, or
//  by a combination of these. You can also specify whether you are
//  interested in only the first match, only the last match, or all
//  matches. You can also control whether normal traversal rules are
//  followed (switching, separators, etc.) or whether every single
//  node is to be searched.
//
//////////////////////////////////////////////////////////////////////////////

/**
* Searches for nodes in a scene graph.
* 
* @ingroup actions
* 
* @DESCRIPTION
*   This class is used to search scene graphs for specific nodes, nodes of a
*   specific type, nodes with a specific name, or any combination of these. It can
*   search for just the first or last node satisfying the criteria or for all such
*   nodes. The action returns a path (SoPath) to each node found.  The searched for
*   node is the "tail" of each path.
*
*   Note that the search criteria are @I cumulative@i. For example, if you do a
*   search by name (setName()), then reuse the same SoSearchAction object to do a
*   search by type (setType()), the action will actually search for a node that
*   satisfies @I both@i criteria. To avoid this problem, call reset() before reusing
*   the action object.
*
*   By default SoSearchAction only searches nodes that are actually traversed.
*   For example it would not search all the children of an SoSwitch node unless
*   the whichChild field is set to SO_SWITCH_ALL.  To search all nodes in the
*   scene graph (except nodekits - see next paragraph) call setSearchingAll(TRUE).
*   
*   Nodekits: @BR
*   - By default SoSearchAction will not search inside nodekits even when
*     setSearchingAll is true.  This is because nodekits try to keep their
*     children hidden.
*     To allow searching inside nodekits call the static
*     method SoBaseKit::setSearchingChildren(TRUE). @BR
*   - A common problem is that when the searched for node is found inside
*     a nodekit, the SoPath method \if_java regular.getTail() \else getTail() \endif
*     does not return the found node. This is also because nodekits try to keep
*     their children hidden. To avoid this problem \if_java use full.getTail()
*     instead. \endif \if_dotnet use FullPath.GetTail() instead. \endif \if_cpp cast
*     the returned SoPath to SoFullPath. \endif @BR
*
*   Hidden references:
*   \if_cpp
*   \par
*   SoSearchAction creates one or more SoPath objects when applied to the scene graph.
*   An SoPath object calls ref() on each node in the path. This reference will prevent
*   the node from being destroyed for as long as the SoPath object exists. These SoPath
*   objects are stored internally in the action and exist until the action object itself
*   is destroyed or reset.
*   \else
*   \par
*   SoSearchAction creates one or more SoPath objects when applied to the scene graph.
*   The SoPath object references each node in the path. This reference will prevent the
*   node and its associated memory from being reclaimed for as long as the SoPath object
*   exists. These SoPath objects are stored internally in the action and exist until the
*   action object itself is reclaimed or reset.
*   \endif @BR
*
*  Efficiency:
*  \par
*  SoSearchAction is convenient for finding one or many nodes in the scene graph.
*  However it may be an inefficient solution for finding a large number of nodes because
*  it uses CPU time and memory to create an SoPath for every node found.  If you expect
*  to find many nodes, especially if you just need the node object and not a path, then
*  you should consider using SoCallbackAction instead.  
*  \par
*  For example, if you want to
*  count all the shape nodes in the scene graph, you could use an SoSearchAction similar
*  to the second example below. The number of shapes would conveniently be the number of
*  paths created by the action, but you wouldn't actually make any use of the path information.
*  Using SoCallbackAction would be a little more work, because you have to implement a
*  counter in a \if_cpp callback method. \endif \if_dotnet delegate method. \endif
*  \if_java callback class. \endif  But it would be much more efficient because
*  the action simply calls your \if_dotnet delegate \else callback \endif when each shape
*  node is visited during the traversal.
*
*  @EXAMPLE
*   Example 1: Given an instance of a node, create a path to the
*   location of that node in the scene graph:
*   \if_cpp
*     \code
*     SoSearchAction sa;
*       sa.setNode( coneNode );
*       sa.setSearchingAll( TRUE );              // Optional: Search all nodes
*       SoBaseKit::setSearchingChildren( TRUE ); // Optional: Search inside nodekits
*       sa.apply( root );
*       SoFullPath* path = (SoFullPath*)sa.getPath(); // Get path
*       if (path != NULL) {
*         SoNode* node = path.getTail();
*       }
*     \endcode
*   \endif
*   \if_dotnet
*     \code
*     SoSearchAction sa = new SoSearchAction();
*       sa.SetNode( coneNode );
*       sa.SetSearchingAll( true );              // Optional: Search all nodes
*       SoBaseKit.SetSearchingChildren( true );  // Optional: Search inside nodekits
*       sa.Apply( root );
*       SoPath path = sa.GetPath();
*       if (path != null) {
*         SoNode node = path.FullPath.GetTail();
*       }
*     \endcode
*   \endif
*   \if_java
*     \code
*     SoSearchAction sa = new SoSearchAction();
*         sa.setNode( cone );
*         sa.setSearchingAll( true );              // Optional: Search all nodes
*         SoBaseKit.setSearchingChildren( true );  // Optional: Search inside nodekits
*         sa.apply( root );
*         SoPath path = sa.getPath();
*         if (path != null) {
*             SoNode node = path.regular.getTail();
*         }
*     \endcode
*   \endif
*
*   Example 2: Find all the SoFont nodes in the scene graph:
*   \if_cpp
*     \code
*     SoSearchAction sa;
*       sa.setType( SoFont::getClassTypeId() );
*       sa.setInterest( SoSearchAction::ALL );   // Find ALL instances
*       sa.setSearchingAll( TRUE );              // Optional: Search all nodes
*       SoBaseKit::setSearchingChildren( TRUE ); // Optional: Search inside nodekits
*       sa.apply( rootNode );
*       SoPathList& pathList = sa.getPaths();    // Get list of paths
*       int numPaths = pathList.getLength();     // How many found?
*       for (int i = 0; i < numPaths; i++) {
*         SoFullPath* path = (SoFullPath*)pathList[i];
*         SoFont* fontNode = (SoFont*)path->getTail();
*         . . .
*       }
*     \endcode
*   \endif
*   \if_dotnet
*     \code
*     SoSearchAction sa = new SoSearchAction();
*       sa.SetType( typeof(SoFont) );
*       sa.SetInterest( SoSearchAction.Interests.ALL );   // Find ALL instances
*       sa.SetSearchingAll( true );              // Optional: Search all nodes
*       SoBaseKit.SetSearchingChildren( true );  // Optional: Search inside nodekits
*       sa.Apply( rootNode );
*       SoPathList pathList = sa.GetPaths();     // Get list of paths
*       foreach (SoPath path in pathList)
*       {
*         SoFont fontNode = (SoFont)path.FullPath.GetTail();
*         . . .
*       }
*     \endcode
*   \endif
*   \if_java
*     \code
*     SoSearchAction sa = new SoSearchAction();
*       sa.setNodeClass( SoFont.class );
*       sa.setInterest( SoSearchAction.Interests.ALL );   // Find ALL instances
*       sa.setSearchingAll( true );              // Optional: Search all nodes
*       SoBaseKit.setSearchingChildren( true );  // Optional: Search inside nodekits
*       sa.apply( rootNode );
*       java.util.Vector<SoPath> pathList = sa.getPaths();
*       if (! pathList.isEmpty()) {
*           SoPath path = pathList.get( 0 );
*           SoFont fontNode = (SoFont)path.regular.getTail();
*       }
*     \endcode
*   \endif
* 
* @SEE_ALSO
*    SoPath,
*    SoBaseKit,
*    SoSearchPathAction,
*    SoSearchStepAction
*
*
*/
class INVENTOR_API SoSearchAction : public SoAction {

  SO_ACTION_HEADER(SoSearchAction);

 public:

  /** Enum that defines the search criterion. */
  enum LookFor {
	  /**
	  *  Search for a particular node instance
	  */
	  NODE = 0x01,
    /**
     *  Search for a particular type of node 
     */
    TYPE = 0x02,
    /**
     *  Search for a node with a particular name 
     */
    NAME = 0x04
  };

  /** Enum that defines which paths to return. */
  enum Interest {
    /**
     *  Return only the first path found (default)
     */
    FIRST,
    /**
     *  Return only the last path found 
     */
    LAST,
    /**
     *  Return all paths found 
     */
    ALL
  };

  /**
   * Constructor.
   */
  SoSearchAction();

  // Destructor
#ifndef HIDDEN_FROM_DOC
  virtual ~SoSearchAction();
#endif // HIDDEN_FROM_DOC

  /**
   * Resets options back to default values; clears list of returned paths. This can
   * be used to apply the action again with a different set of search criteria.
   * See also #clearApplyResult().
   */
  virtual void reset();

  /** @copydoc SoAction::clearApplyResult()
   *
   * See also #reset()
   */
  virtual void clearApplyResult();

  /**
   * Sets what to look for; @B what @b is a bitmask of @B LookFor @b enum
   * values. Default is no flags at all. Note that setting a node, type, and/or name
   * to search for activates the relevant flag, so you may never need to call this
   * method directly.
   */
  void setFind(int what) { lookingFor = what; }

  /**
   * Returns what to look for.
   */
  int getFind() { return lookingFor; }

  /**
   * Returns the node to search for.
   */
  SoNode *getNode() const { return node; }

  /**
   * Sets the node to search for.
   */
  void setNode(SoNode *n);

  /**
   * Returns the node type to search for. 
   * [OIVJAVA-WRAPPER NAME{getNodeClass},DUPLICATE{isSearchingExtendedClass}]
   * [OIVJAVA-WRAPPER-ARG NO_WRAP]
   * [OIVJAVA-WRAPPER-RETURN-TYPE APPLY_TO{isSearchingExtendedClass} NO_WRAP]
   */
  SoType getType(SbBool &derivedIsOk) const
    { derivedIsOk = derivedOk; return type; }     

  /**
   * Sets the node type to search for. If @B derivedIsOk @b is TRUE, a node
   * that is of a type that is derived from @B t@b will pass this search criterion.
   * [OIVJAVA-WRAPPER NAME{setNodeClass}]
   */
  void setType(SoType t, SbBool derivedIsOk = TRUE);

  /**
   * Returns the name of the node to search for.
   */
  const SbName &getName() const { return name; }

  /**
   * Sets the name of the node to search for.
   */
  void setName(const SbName &n);

  /**
   * Returns which paths to return. 
   */
  Interest getInterest() const { return interest; }

  /**
   * Sets which paths to return. Default is FIRST.
   */
  void setInterest(Interest i) { interest = i; }

  /**
   * Returns FALSE if searching uses regular traversal, TRUE if it traverses
   * every single node. Default is FALSE.
   */
  SbBool isSearchingAll() const { return searchingAll; }

  /**
   * Sets whether searching uses regular traversal or whether it traverses
   * every single node. For example, if this flag is FALSE, an SoSwitch node will
   * traverse only the child or children it would normally traverse for an action. If
   * the flag is TRUE, the switch would always traverse all of its children. The
   * default is FALSE.
   */
  void setSearchingAll(SbBool flag) { searchingAll = flag; }

  /**
   * Returns resulting path, or NULL if no path was found. This should be used if the
   * interest is FIRST or LAST.
   */
  SoPath *getPath() const { return retPath; }

  /**
   * Returns resulting path list. This should be used if the interest is ALL.
   */
  SoPathList &getPaths() { return retPaths; }
  
 SoEXTENDER public:

  // Sets/returns whether action has found all desired nodes
  // (implementation moved into source file for v2.6)
  virtual void setFound();
  SbBool isFound() const { return hasTerminated(); }
  
  // Sets found path or adds to list of found paths (depending on interest)
  void addPath(SoPath *path);
  
 SoINTERNAL public:

  static void initClass();
  static void exitClass();
 
  // This flag is used by the SoSwitch node, which must return a
  // different result from its 'affectsState' method when called
  // during a SearchAction that is searching all children.
  static SbBool duringSearchAll;
  
 protected:

  // Initiates action on graph
  virtual void beginTraversal(SoNode *node);
  
 private:

  // Node type id to search for
  SoType type;

  // Exact match or derived ok
  SbBool derivedOk;

  // Node to search for
  SoNode *node;

  // Name of node to search for 
  SbName name;

  // What to search for
  int lookingFor;

  // Which paths to return
  Interest interest;

  // Search all children or follow normal traversal rules.
  SbBool searchingAll;

  // Found node (if interest != ALL)
  SoPath *retPath;

  // Found nodes (if interest == ALL)
  SoPathList retPaths;
};

#endif /* _SO_SEARCH_ACTION_ */

