/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : R. ALBOU (Sep 1998)
**=======================================================================*/

#ifndef  _SO_TO_HTML_ACTION_
#define  _SO_TO_HTML_ACTION_

#include <Inventor/SbColor.h>
#include <Inventor/SbViewportRegion.h>
#include <Inventor/actions/SoAction.h>
#include <Inventor/SoOffscreenRenderArea.h>

#include <Inventor/actions/SoSubAction.h>

#include <Inventor/SoLists.h>
#include <Inventor/SbBox.h>

class SoNode ;
class SoAction ;

/*----------------------------------------------------------------------------*/

/**
 * @VSGEXT Generates an HTML image map and an image from a scene graph.
 * 
 * @ingroup actions
 * 
 * @DESCRIPTION
 *   This action generates an image file and an HTML file containing the image
 *   (&lt;IMG&gt;) and image map (\<MAP>) tags. Each region of the image map will have a
 *   URL associated with it. 
 *   
 *   With HTML, image maps allow to specify regions of an image and assign a specific
 *   action to each region (a link can be associated to each map). When the region is
 *   activated by the user, the action is executed.
 *   
 *   Each region of the image is determined by a subgraph which is under an
 *   SoWWWAnchor group node.
 *   
 *   The HTML file generated contains, in the following order:
 *     -# An image map tag that defines the regions (maps) corresponding to each
 *        subgraph that is under an SoWWWAnchor group node.
 *        @P
 *        For each region, the associated link is the URL defined either by the SoWWWAnchor node.
 *     -# An image tag that references the image generated by SoToHTMLAction.
 * 
 * @SEE_ALSO
 *    SoAction
 * 
 * 
 */
class INVENTOR_API SoToHTMLAction : public SoAction {

  SO_ACTION_HEADER(SoToHTMLAction);

 public: 
  typedef void* SoToHTMLReallocCB(void *ptr, size_t newSize) ;

    /** Defines the different shape type for maps.  */
  enum ShapeType { 
    /** NONE */
    NONE, 
    /** RECTANGLE */
    RECTANGLE,  
    /** CIRCLE */
    CIRCLE,
    /** POLYGON(default) */
    POLYGON         
  } ;

    /** Components used to generate the image  */
  enum Components { 
    /** LUMINANCE */
    LUMINANCE = 1, 
    /** LUMINANCE_TRANSPARENCY */
    LUMINANCE_TRANSPARENCY = 2,
    /** RGB(Default) */
    RGB = 3, 
    /** RGB_TRANSPARENCY */
    RGB_TRANSPARENCY = 4
  };

    /** Defines the format of the image  */
  enum ImageFormat { 
    /** BMP_FORMAT */
    BMP_FORMAT, 
    /** JPEG_FORMAT */
    JPEG_FORMAT
  };
        
  /**
   * Constructor.
   */
  SoToHTMLAction() ;
        
  /**
   * Destructor.
   */
  ~SoToHTMLAction() ;
        
 /**
  * Opens named file \if_cpp (sets file pointer to result) \endif\.
  * This returns FALSE on error.
  * By default, output goes to @B stdout @b.
  *
  * @UNICODE_WARNING
  */
  SoNONUNICODE SbBool openHTMLFile(const char *filename) ;


 /**
  * Opens named file \if_cpp (sets file pointer to result) \endif\.
  * This returns FALSE on error.
  * By default, output goes to @B stdout @b.
  */
  SbBool openHTMLFile( const SbString& filename );

  /**
   * Closes the HTML output file.
   */
  void closeHTMLFile() ;
        
  /**
   * Sets file pointer to write to.
   */
  void setHTMLFilePointer(FILE *newFP) ;
  /**
   * Returns the file pointer in use, or NULL if using a buffer.
   */
  FILE* getHTMLFilePointer() const ;

  /**
   * Sets up memory buffer for writing HTML output, its initial size, a reallocation
   * function (which is called if there is not enough room in the buffer), and an
   * offset in the buffer at which to begin writing. If the reallocation function
   * returns NULL, writing will be disabled.
   */
  void setHTMLBuffer(void *bufPointer, size_t initSize,
                     SoToHTMLReallocCB *reallocFunc, int32_t offset=0) ;
  /**
   * Returns pointer to memory buffer of the HTML output being written to and the new
   * size of the buffer. Returns FALSE if not writing into a buffer.
   * [OIV-WRAPPER-NOT-WRAP]
   */
  SbBool getHTMLBuffer(void *&bufPointer, size_t &nBytes) const ;
  /**
   * Resets buffer of the HTML output for output again. Output starts over at
   * beginning of buffer.
   */
  void   resetHTMLBuffer() ;

  /**
   * Sets the URL name of image in the HTML file, ("image.jpg" by
   * default). 
   *
   * This is the first way to specify the image file name.
   */
  void setImageURLName(const char *name) ;
  /**
   * Returns the URL name of image in the HTML file.
   */
  const char *getImageURLName() const ;

  /**
   * Sets the file pointer to use for the generated image file. 
   * If @B newFP@b is NULL, then
   * the filename is the basename of the image URL name.
   * 
   * This is the second way to define the image file name.
   */
  void setImageFilePointer(FILE *newFP) ;
  /**
   * Returns the file pointer for the generated image file.
   */
  FILE* getImageFilePointer() const ;
        
  /**
   * Sets the region shape type. POLYGON by default.
   */
  void setRegionShapeType(ShapeType type) ;
  /**
   * Returns the region shape type.
   */
  ShapeType getRegionShapeType() const ;

  /**
   * Sets the image background color. SbColor(0,0,0) by default.
   */
  void setImageBackgroundColor(const SbColor &c);
  /**
   * Returns the image background color.
   */
  const SbColor & getImageBackgroundColor() const ;

  /**
   * Sets the components of image. RGB by default.
   */
  void setImageComponents( Components components );
  /**
   * Returns the components of image.
   */
  Components getImageComponents() const ;

  /**
   * Sets the image format. JPEG_FORMAT by default.
   */
  void setImageFormat(ImageFormat format) ;
  /**
   * Returns the image format.
   */
  ImageFormat getImageFormat() const ;

  /**
   * Sets the image quality. Used for JPEG image. Quality is a value from 0.
   * to 1., with 0 being the worst quality (maximum compression) and 1 being the best
   * quality (minimum compression). The default is 1.
   */
  void setImageQuality(float quality) ;
  /**
   * Returns the image quality.
   */
  float getImageQuality() const ;

  /**
   * Specifies viewport region for the image. Allows the user to specify the image
   * size.
   */
  void setViewportRegion(const SbViewportRegion &region);
  /**
   * Returns viewport region for the image.
   */
  const SbViewportRegion& getViewportRegion() const;

  /**
   * Sets anchor highlighting. TRUE by default.
   */
  void setMapHighlight(SbBool onOff);
  /**
   * Returns anchor highlighting.
   */
  SbBool isMapHighlight();

  /** Generate HTML output for the specified root node */
  virtual void apply(SoNode *scene) ;
  /** Generate HTML output for the specified path */
  virtual void apply(SoPath *path) ;
  /** Generate HTML output for the specified list of path */
  virtual void apply(const SoPathList &pathList, SbBool flag = FALSE) ;

 SoINTERNAL public:
  // Internally used to initialize the class.
  static void initClass() ;
  static void exitClass() ;

 private:
  // Callbacks
  static void stencilCB(void *d, SoAction* action) ;
  static void stencilEndCB(void *d, SoAction* action) ;
  static void initStencilCB(void *d, SoAction* action) ;

  // Compute bounding for each anchor from the stencil
  // buffer
  void computeAnchorsBBoxList() ;

  // Adds/removes a callback node as first child of 
  // each anchor group node
  void addCallbackAtFirstForAnchors() ;
  void removeAnchorsCallbacks() ;

  // Adds/removes a callback as first child of
  // the root scene graph
  void addInitStencilCallback() ;
  void removeInitStencilCallback() ;

  // Retreives anchor URL at a specific index
  // From the scene graph
  const char* retreiveAnchorURL(int index) ;

  // Read the stencil buffer
  void readStencilBuffer() ;

  // HTML Instructions
  void writelnHTMLTag(const char *tag) ;
  void writeHTMLTag(const char *tag)  ;
  void writeHTMLMaps() ;
  void writeHTMLImage() ;
  void writeHTML(const char *str) ;

  // Generate the HTML output
  void generateHTMLOutput() ;

  // Seachs all SoWWWAnchor
  // and updates the anchor list.
  void updateAnchorsList() ;

  // Compute an approximation from a polygon with angle of two consecutive
  // segments inferior to the chordal deviation.
  // The indices and their number corresponding to approximation of the original 
  // polygon are returned.
  int* computeApproximatePolygon(const SbVec2s *polygon, int numPolyPoints,
                                 int &numApproxIndices) ;

  // Fill a bounding box of the stencil buffer with all values == valueToReplace 
  // by replaceValue.
  void fillStencilBuffer(const SbBox2s &box, int valueToReplace, int replaceValue) ;

  // Return the base name of an URL
  // ex : https://www.openinventor.com//file.jpg return file.jpg
  const char* getImageURLBaseName() const ;

  // File descriptor for the image and the HTML file
  FILE* m_HTMLFP ;
  FILE* m_imageFP ;

  // Indicates if the stencil has been initialize
  // in order to prevent from several initializations 
  // (SoAnnotation node could cause this problem)
  SbBool m_isStencilInit ;

  // Image URL name
  SbString m_imageURLName ;

  // Nodelist for SoWWWAnchor
  SoNodeList m_anchorList ;

  // Array of bounding box for each anchors
  SbBox2s *m_boxAnchorList ;

  // Current scene graph
  SoNode *m_scene ;

  SoRef<SoOffscreenRenderArea> m_osRenderArea;

  // Current stencil buffer
  unsigned char* m_stencilBuffer ;

  // Viewport region for image
  SbViewportRegion m_vpRegion ;

  // Image components
  Components       m_imageComponents ;

  // Image background color
  SbColor m_imageBackgroundColor ;

  // Image format
  ImageFormat m_imageFormat ;

  // Image quality
  float m_imageQuality ;

  // Region shape type
  ShapeType m_regionShapeType ;

  // Map number
  static int m_mapNumber ;

  // Indicate whether an HTML file has been
  // opened with openHTMLFile().
  SbBool m_isOpenHTMLFile ;

  // Map highlighting
  SbBool m_isMapHighlight;

  /* Management of a buffer for HTML file */
  SbBool m_isHTMLBuffer ;
  void*  m_HTMLBuffer ;
  char*  m_curHTMLBuffer ;
  size_t m_HTMLBufferSize ;
  SoToHTMLReallocCB *m_reallocHTMLFunc ;

  SbBool makeRoomInBuf(size_t nBytes) ;

};

#endif // _SO_TO_HTML_ACTION_



