/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_CACHE
#define  _SO_CACHE

////////////////////////////////////////////////////////////////////////
//
// @ingroup Caches
//
//  Class SoCache:
//
//  This is the base class for all types of caches. Each cache
//  maintains the following information:
//
//      A reference count, used to allow sharing of cache instances.
//      ref() and unref() methods increment and decrement the count.
//
//      An elements-used list. This is a list of elements used in the
//      cache that are set outside it. A cache is invalidated if any
//      of these elements has changed since the cache was created.
//      There is also an elements-flag array so we can quickly tell if
//      an element type has already been added to the elements-used
//      list.
//
////////////////////////////////////////////////////////////////////////

#include <Inventor/SoLists.h>
#include <Inventor/SoTypedObject.h>
#include <Inventor/caches/SoBaseContextCache.h>

class SoElement;
class SoState;

SoEXTENDER class INVENTOR_API SoCache : public SoBaseContextCache, public SoTypedObject
{

 public:

  /**
   * Constructor. Takes the state in effect when the cache is used;
   * it is assumed that the state is pushed before the cache is
   * created.
   */
  SoCache(SoState *state);

  /**
   * Adds an element to elements-used list if not already there
   */
  void addElement(const SoElement *elt);

  /**
   * Adds a dependency of this instance on another cache instance.
   * The default method takes care of adding dependencies from the
   * child cache.
   */
  virtual void addCacheDependency(const SoState *state, SoCache *cache);

  /**
   * Returns TRUE if cache is valid with respect to the given state
   */
  virtual SbBool isValid(const SoState *state) const;

  /**
   * Assuming isValid() returns FALSE, this will return the first state
   * element that is invalid (auto-caching uses this in its
   * heuristic).  Returns NULL if the cache is not invalid because
   * of an element or if the cache is valid.
   */
  const SoElement* getInvalidElement(const SoState *state) const;

  /**
   * Assuming isValid() returns FALSE, this will return the first cached
   * element that is invalid (auto-caching uses this in its
   * heuristic).  Returns NULL if the cache is not invalid because
   * of an element or if the cache is valid.
   */
  const SoElement* getCachedInvalidElement(const SoState *state) const;

  /**
   * Return is the given element is valid
   * True if the element is not monitored
   */
  virtual SbBool isValid(const SoState* state, const SoElement* elf) const;

  /**
   * Make this cache invalid
   */
  inline void invalidate() const { m_invalidated = true; }

  /**
   * return the class type.
   */
  static SoType getClassTypeId();

  /**
   * Returns type id.
   */
  virtual SoType getTypeId() const;

 SoINTERNAL public:
  /**
   * register class.
   */
  static void initClass();

  /**
   * Unregister class.
   */
  static void exitClass();

  /**
   * This function is called by when a context has been requested
   * to delete the context through the SoDeviceContext::dispose() method.
   * As some object might be still attached, it triggers all attached object
   * release() method to ask them to detach (and then get it refcount to zero).
   */
  virtual void release(SoDeviceContextSharedGroup *ctx);

  /*
   * mark this cache to indicate that there is some delayed path below
   * It helps MultiPassManager to know if a cache has to be invalidated
   * when the Path is removed from a Delayed PassPathList
   * to force its traversal and to allow registerCurPath to reoccur
   */
  void flagDelayedPath()
  { m_hasDelayedPath=true; }

  /* Returns the number of delayed path that are below this cache */
  bool hasDelayedPath() const
  { return m_hasDelayedPath; }

  virtual const std::vector<const SoElement*>& getUsedElements() const { return m_elementsUsed; }

  static bool isDebuggingCache()
  { return s_debugCaches; }

  static bool isDebuggingCache2()
  { return s_debugCaches2; }

  /**
   * Takes the state in effect when the cache is used;
   * it is assumed that the state is pushed before the cache is
   * reseted.
   */
  virtual void reset(SoState *state);

 protected:

   /**
    * called by unref before calling destructor
    * if it returns false then it means that delete should not be called
    * as some object were not freed because of context dependency.
    */
   virtual bool notifyDelete() const
   { return true; };

  /**
   * Destructor.
   * called by unref() if notifyDelete returned true.
   * else will be called through Dispose call
   */
  virtual ~SoCache();

  /**
   * Return true if invalidate() was called
   */
  bool isExplicitlyInvalidate() const
  { return m_invalidated; };

   /**
   * Gets first invalid element in cache or state.
   * stateElt == true  => returns the first element in state invalid w/r to the one having same type in cache
   * stateElt == false => returns the first element in cache invalid w/r to the one having same type in state
   */
  const SoElement* getInvalidElementCore(const SoState *state, bool stateElt) const;

 private:

  std::vector<unsigned char> m_elementsUsedFlags;     // Which elements on list
  std::vector<const SoElement*> m_elementsUsed;           // Elements used in cache

  short               m_depth;                  // Depth of state
  bool                m_hasDelayedPath;         // number of delayed path below this cache
  mutable bool        m_invalidated;            // True if invalidate called

  // Class type
  static SoType s_classTypeId;

  // handle internal IV_DEBUG_CACHES and IV_DEBUG_CACHES2 debug env vars status
  static bool s_debugCaches;
  static bool s_debugCaches2;
};

#endif /* _SO_CACHE */


