/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef SO_CONTEXTED_OBJECT_CACHE_H
#define SO_CONTEXTED_OBJECT_CACHE_H

#include <Inventor/SbBasic.h>
#include <Inventor/SoDB.h>
#include <Inventor/threads/SbThreadStorage.h>
#include <Inventor/caches/SoBaseContextCache.h>
#include <Inventor/caches/SoContextedObjectCacheList.h>
#include <Inventor/errors/SoDebugError.h>
#include <Inventor/SbMathHelper.h>
#include <Inventor/devices/SoDeviceContextSharedGroup.h>
#include <Inventor/devices/SoGLContext.h>
#include <Inventor/STL/list>
#include <Inventor/STL/algorithm>

class SbThreadMutex;
class SoState;

/**
 * This class cache a OpenGL object for each contexts.
 */
template <typename TContextedObject>
SoINTERNAL class INVENTOR_API SoContextedObjectCache : public SoBaseContextCache
{
public:
  /**
   * Contsructor.
   */
  SoContextedObjectCache( bool sharable = true )
  : m_sharable(sharable)
  {
    m_numRenderCaches = 2;
    ref();
  }

  /**
   * Return cached texture object.
   * @param state is the current Inventor state
   * @param recreated is true if the GL object has just been (re)created.
   */
  TContextedObject* getObject(SoState* state, bool& recreated)
  {
    // get/check context
    SoGLContext* context = SoGLContext::getCurrent();
    if ( context == NULL )
    {
      SoDebugError::post("SoContextedObjectCache<TContextedObject>::getObject()", "No active context !!\n");
      return NULL;
    }

    // if no cache already exist for this thread, create one
    if ( !m_objCache )
      m_objCache = new SoContextedObjectCacheList<TContextedObject>(m_numRenderCaches, m_sharable);

    // try to get an existing cache object depending on the context sharedId 
    // as we consider that a cache is valid in any of its shared context
    int sharedId = context->getSharedId();
    TContextedObject *texObj = (m_objCache->getCache(state, 1.0f));
    recreated = false;

    // if no valid object exist then create one
    if ( !texObj )
    {
      recreated = true;
      texObj = createObject(state);
      m_objCache->setCache(sharedId, texObj, 1.0f, state);
      
      // do not forget to tell the context that we are attached to it
      // only if we are not already notified by this share group.
      if (!context->getSharedGroup()->contains(this))
        addSharedGroupDependency(context->getSharedGroup());
    }

    return texObj;
  }

  /**
   * Return cached texture object.
   * @param state is the current Inventor state
   */
  TContextedObject* getObject(SoState* state)
  {
    bool recreated;
    return getObject(state, recreated);
  }

  /**
   * Return true if a cache object exists for this context
   */
  bool hasValidObject(SoDeviceContext* context)
  {
    if (context==NULL)
      return false;
    // optim: to avoid bool casting that is in fact calling  T operator->() const twice on 
    // SbThreadStorage smart pointer
    SoContextedObjectCacheList<TContextedObject> *pObjCache ( m_objCache.operator->() );
    return ( pObjCache && ( pObjCache->getCache(context, 1.0f) != NULL ));
  }

  /**
   * Return true if a cache object exists for this context
   */
  bool hasValidObject(SoState* /*state*/)
  {
    // get current context
    SoGLContext* context = SoGLContext::getCurrent();
    return hasValidObject(context);
  }


  /** Invalidate all caches */
  void invalidate() const
  {
    m_objCache.call(&SoContextedObjectCacheList<TContextedObject>::invalidateAll, static_cast<SoState*>(NULL));

    // we release all context dependencies
    std::list<SoDeviceContextSharedGroup*>::const_iterator it = m_sharedGroupDependencyList.begin();
    while (it!=m_sharedGroupDependencyList.end())
    {
      SoDeviceContextSharedGroup* sharedgroup = (*it);
      // we have to convert to noconst to avoid API break of SoDeviceSharedGroup::unref() that miss the const
      sharedgroup->unref(const_cast<SoBaseContextCache*>(static_cast<const SoBaseContextCache*>(this)));
      ++it;
    }

    // and clear the list
    // nb: we have to cast to noconst to avoid notifyDelete API break
    (const_cast<SoContextedObjectCache<TContextedObject>*>(this))->m_sharedGroupDependencyList.clear();
    (const_cast<SoContextedObjectCache<TContextedObject>*>(this))->setContextDependent(false);
  }

  /**
   * This function is called when a context has been requested 
   * to be deleted through the SoDeviceContext::dispose() method.
   * As some object might be still attached, it triggers all attached object
   * release() method to ask them to detach (and then get its refcount to zero).
   */
  virtual void release(SoDeviceContextSharedGroup *ctx)
  {
    // remove all object that belongs to this context
    SoContextedObjectCacheList<TContextedObject> *pObjCache ( m_objCache.operator->() );
    if ( !pObjCache )
    {
      SoDebugError::post("SoContextedObject::release()","There is no cache for this thread");
    }
    else
    {
      pObjCache->invalidateContext( ctx );
    }
    // detach from context.
    removeSharedGroupDependency(ctx);
  }


protected:
   /**
    * called by unref before calling destructor
    * return false if there is still some context dependent object 
    * and then delete should not be called
    */
   virtual bool notifyDelete() const
   {
     bool ret = true;

     // invalidate all objects
     invalidate();

     return ret;
   }

  /**
   * Destructor.
   */
  virtual ~SoContextedObjectCache()
  {
    invalidate();
  }

  /**
   * Allocate a TContextedObject
   */
  virtual TContextedObject* createObject(SoState *state)
  {
    TContextedObject* obj = new TContextedObject(state);
    return obj;
  }

private:

  mutable SbThreadStorage< SoContextedObjectCacheList<TContextedObject> * > m_objCache;
  int m_numRenderCaches;
  bool m_sharable;

  
  // hold the list of SoDeviceContextSharedGroup we depends on
  // in order to handle notifyDelete() correctly
  std::list<SoDeviceContextSharedGroup*> m_sharedGroupDependencyList;

  // add dependency to the given shared group
  void addSharedGroupDependency(SoDeviceContextSharedGroup* sharedGroup)
  {
    if ( sharedGroup )
    {
      m_sharedGroupDependencyList.push_back(sharedGroup);
      sharedGroup->ref(this);
      setContextDependent(true);
    }
  }

  // remove dependency to the given shared group
  void removeSharedGroupDependency(SoDeviceContextSharedGroup* sharedGroup)
  {
    if ( sharedGroup)
    {
      std::list<SoDeviceContextSharedGroup*>::iterator it = std::find(m_sharedGroupDependencyList.begin(),m_sharedGroupDependencyList.end(),sharedGroup);
      if ( it!=m_sharedGroupDependencyList.end() )
      {
        m_sharedGroupDependencyList.erase(it);
        setContextDependent(m_sharedGroupDependencyList.size()>0);
        sharedGroup->unref(this);
      }
      else
      {
        // this should never happens, if we call removeSharedGroupDependency
        // addSharedGroupDependency have been called if not it is a bug
        assert(0);
      }
    }
  }
};

#endif //SO_CONTEXTED_OBJECT_CACHE


