/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : T. ANDRIEU (AUG 2014)
**=======================================================================*/


#ifndef  _SO_CACHE_OBJECT
#define  _SO_CACHE_OBJECT

#include <Inventor/caches/SoCache.h>
#include <Inventor/misc/SoRefCounter.h>
#include <Inventor/sensors/SoFieldSensor.h>

#if defined(_WIN32)
#pragma warning(push)
#pragma warning(disable: 4251) // 'identifier' : class 'type' needs to have dll-interface to be used by clients of class 'type2'
#endif

/**
 * This class is a cache handler that can be used to monitor the modifications made to elements or fields.
 *
 * @ingroup Caches
 *
 * @DESCRIPTION
 * In order to add new Elements or Fields to be monitored by the cache, use the appropriate #addDependency() method.
 *
 * If one of the added elements or fields has been modified since the last call to #updateCache(),
 * the #isValid() method will return false, and a subsequent call to #updateCache() will make it valid again.
 *
 * Additionnally, the handled cache can be manually invalidated using the #invalidate() method.
 */
class INVENTOR_API SoNodeDependencies : public SoRefCounter
{
public:
  SoNodeDependencies();

  ~SoNodeDependencies();

  /** Returns true if the cache is valid for the given state. */
  bool isValid(SoState* state) const;

  /**
   * Update cache according to given state.
   * After this call, the cache will be valid.
   */
  void updateCache(SoState* state);

  /**
   * Manually invalidate the cache.
   */
  void invalidate();

  /**
   * Add given element type as a dependency for the cache.
   * [OIV-WRAPPER NO_WRAP]
   */
  template <typename SO_ELEMENT>
  void addDependency()
  {
    addDependency( SO_ELEMENT::getClassTypeId(), SO_ELEMENT::getClassStackIndex() );
  }

  /**
   * Add given field as a dependency for the cache.
   */
  void addDependency( SoField& field );

  /** Add given element type as a dependency for the cache. */
  void addDependency(SoType type, int classStackIndex);

private:
  struct CachedElement
  {
    CachedElement(SoType& type_, int classStackIndex_) : type(type_), classStackIndex(classStackIndex_) {}
    SoType type; // type of element
    int classStackIndex; // stackIndex of element in state cache stack.
  };

  /**
   * List of elements this cache will depend on.
   * It is filled using the #addDependency() method and when m_cache is recreated, m_cache->addDependency() is
   * called with this list.
   */
  std::vector<CachedElement> m_elementsDependencies;

  /** List of fields this cache will depend on. */
  std::vector<SoFieldSensor*> m_fieldSensors;

  /** Fields cache validity. */
  bool m_fieldsValid;

  /** Callback used with field sensors. */
  static void fieldSensorCB( void* data, SoSensor* sensor );

  SoRef<SoCache> m_cache;

  mutable SbThreadSpinlock m_mutex;
};

#if defined(_WIN32)
#pragma warning(pop)
#endif

#endif /* _SO_CACHE_OBJECT */


