/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Thaddeus Beier (MMM yyyy)
** Modified by : Dave Immel (MMM yyyy)
** Modified by : Howard Look (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : Martial Bonaventure (Nov 2010)
**=======================================================================*/


#ifndef _SO_FACE_DETAIL_
#define _SO_FACE_DETAIL_

#include <Inventor/details/SoPointDetail.h>
class SoFaceDetailImpl;

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoFaceDetail
//
//  Detail information about vertex-based shapes made of faces. It
//  adds indices of various items that vary among faces and vertices.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Stores detail information about vertex-based shapes made of faces.
 * 
 * @ingroup details
 * 
 * @DESCRIPTION
 *   This class contains detail information about a face in a vertex-based
 *   shape made of faces. The information includes the number of points in the face,
 *   the points forming the vertices of the face, and the index of the face within
 *   the shape.
 *   
 *   Note that when an SoFaceDetail is returned from picking (in an SoPickedPoint), it
 *   will contain details for all points defining the face that was intersected.
 *   However, when an SoFaceDetail is created for a triangle produced during
 *   primitive generation (in an SoPrimitiveVertex), it will contain details for only
 *   the three vertices of the triangle.
 * 
 * @SEE_ALSO
 *    SoDetail,
 *    SoPickedPoint,
 *    SoPrimitiveVertex,
 *    SoVertexShape
 * 
 * 
 */
class INVENTOR_API SoFaceDetail : public SoDetail {

  SO_DETAIL_HEADER(SoFaceDetail);

 public:
  /**
   * Constructor.
   */
  SoFaceDetail();
  /**
   * Destructor.
   */
  virtual ~SoFaceDetail();

  /**
   * Returns the number of points in the face.
   */
  int32_t                     getNumPoints() const;

  /**
   * Returns information about the point forming the i'th vertex of the face,
   * represented as an SoPointDetail.
   */
  const SoPointDetail *       getPoint(int i) const;

  /**
   * Returns the index of the face within the shape.
   */
  int32_t                     getFaceIndex() const;

  /**
   * Returns the index of the part containing the face within the shape.
   */
  int32_t                     getPartIndex() const;

  /** @copydoc SoDetail::copy() */
  virtual SoDetail* copy() const;

 SoEXTENDER public:
  // Sets the number of points in the face and allocates room for the points
  void                        setNumPoints(int32_t num);

  // Copies a point detail from the given detail
  void                        setPoint(int32_t index, const SoPointDetail *pd);

  // Sets the face index and part index
  void                        setFaceIndex(int32_t i);
  void                        setPartIndex(int32_t i);

  // Return a pointer to the point details.
  SoPointDetail *             getPoints();

  /**
  * [OIV-WRAPPER NO_WRAP]
  */
  void    setNumPoints(long num)                              // System long
    { setNumPoints ((int32_t) num); }

  /**
  * [OIV-WRAPPER NO_WRAP]
  */
  void    setPoint(long index, const SoPointDetail *pd)       // System long
    { setPoint ((int32_t) index, pd); }

  /**
  * [OIV-WRAPPER NO_WRAP]
  */
  void    setFaceIndex(long i)                                // System long
    { setFaceIndex ((int32_t) i); }

  /**
  * [OIV-WRAPPER NO_WRAP]
  */
  void    setPartIndex(long i)                                // System long
    { setPartIndex ((int32_t) i); }

 SoINTERNAL public:
  SoFaceDetail( const SoFaceDetail& faceDetail );
  SoFaceDetail& operator=( const SoFaceDetail& faceDetail );
  void copy( const SoFaceDetail& faceDetail );
  SoPointDetail *       getPoint_(int i) const;

  static void                 initClass();
  static void                 exitClass();

 private:
   SoFaceDetailImpl*          m_impl;

};

#endif /* _SO_FACE_DETAIL_ */

