/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <Inventor/SbBase.h>
#include <Inventor/SbPImpl.h>
#include <Inventor/misc/SoRefCounter.h>
#include <Inventor/SoTypedObject.h>
#include <Inventor/SoSubTypedClass.h>
#include <Inventor/devices/SoDeviceContext.h>
#include <Inventor/threads/SbThreadSpinlock.h>
#include <Inventor/STL/limits>
#include <Inventor/renderer/RendererResourceMacro.h>

#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4251)
#endif

class SoDeviceContext;
class SoCpuBufferObject;
class SoCpuBufferObjectCache;

SO_PIMPL_BASE_PUBLIC_DECLARATION(SoBufferObject)

/**
 * Used to indicate that we want to use the whole buffer.
 *
 * @relates SoBufferObject
 */
#define SO_BUFFER_SIZE_ALL  ((std::numeric_limits<size_t>::max)())

/**
* @VSGEXT Abstract base class for buffer object management
*
* @ingroup Devices
*
* @DESCRIPTION
*
* This class provides generic functions to manage buffer objects.
*
* There are specific implementations of this class for buffer
* objects on each type of device:
* - SoCpuBufferObject: A buffer in CPU system memory.
* - SoGpuBufferObject: A buffer in GPU memory.
* - SoGLBufferObject: A buffer in GPU memory managed through the
*   OpenGL API.
*
* In many cases the abstract SoBufferObject methods allow the
* application to handle buffer objects without knowing the specific
* type of the buffer. This is very convenient when computation is
* implemented on multiple devices.
*
* Since version 8.0, some Open Inventor classes take or return data
* using an SoBufferObject (or derived class).  For example texture
* images (see SoSFImage), geometry data (see SoBufferedShape) and
* tiles of volume data (see SoLDMDataAccess).
*
* Creating a buffer object @BR
* \par
* Before creating a non-CPU buffer object you must bind (then unbind) a
* valid context for the target device. For example, to create a GPU
* buffer object (SoGpuBufferObject) you can bind the viewer's context using
* methods in the view class or you can use the class SoGLContext. See the
* example on the SoGpuBuffer page.  For an existing buffer object you can
* use the buffer's \if_dotnet Bind()/Unbind() \else bind()/unbind() \endif
* methods or get a valid context to bind using the \if_dotnet GetContext() \else
* getContext() \endif method. Binding this context is necessary before
* changing buffer properties or calling \if_dotnet SetSize(), Memcpy(),
* Memset(), etc. \else setSize(), memcpy(), memset(), etc. \endif
*
* @if_cpp
* \par
* SoBufferObject classes are reference counted objects (just like nodes, 
* paths, etc).  Therefore you cannot create an SoBufferObject on the 
* stack. And to delete an SoBufferObject you must ref and unref the 
* object.  The SoRef smart pointer is a safe and convenient way to
* create buffer objects. For example, the following code will NOT compile:
* \code
*    {
*      SoCpuBufferObject cpuObj; // ILLEGAL
*      . . .
*    }
*    // OR...
*    {
*      SoCpuBufferObject* cpuObj = new SoCpuBufferObject;
*      cpuObj->map(SoBufferObject::READ_ONLY);
*      . . .
*      cpuObj->unmap();
*      delete cpuObj;  // ILLEGAL
*    }
* \endcode
* \par
* Use the following code instead:
* \code
*    {
*      SoRef<SoCpuBufferObject> cpuObj = new SoCpuBufferObject;
*      cpuObj->map(SoBufferObject::READ_ONLY);
*      . . .
*      cpuObj->unmap();
*    }
*    // OR...
*    {
*      SoCpuBufferObject* cpuObj = new SoCpuBufferObject;
*      cpuObj->ref();
*      cpuObj->map(SoBufferObject::READ_ONLY);
*      . . .
*      cpuObj->unmap();
*      cpuObj->unref();
*    }
*  \endcode
* @endif
*
* Allocating memory in a buffer object: @BR
* \par
* Use the \if_dotnet SetSize() \else setSize() \endif method to allocate memory
* in a buffer object. (Note that you must bind a valid context before calling
* this method on non-CPU buffer objects.) Generally buffer objects have zero
* size when created. However some buffer object classes have a constructor
* that takes an existing block of memory as a parameter. In that case the
* initial size is non-zero. You must allocate memory before copying data into
* a buffer object using either the \if_dotnet Memcpy() \else memcpy() \endif method
* or a direct \if_cpp pointer \else reference \endif\. Memory is automatically
* allocated, if necessary, when a buffer object is mapped into another buffer object.
*
* Loading data and retrieving data: @BR
* 
* - Some buffer objects, e.g. SoCpuBufferObject, have a constructor that takes an
*   existing block of memory as a parameter. In this case the buffer object is
*   automatically set to the necessary size to contain the specified data. Creating
*   an SoCpuBufferObect that "wraps" the existing data is usually the first step in
*   loading data into a non-CPU buffer using \if_dotnet Memcpy() or Map().
*   \else memcpy() or map(). \endif
*
* - The \if_dotnet Memcpy() \else memcpy() \endif methods copy data into a buffer
*   object from another buffer object. For example to copy data into an OpenGL buffer
*   from a CPU buffer. Before using these methods on a non-CPU buffer object, you
*   must bind (then unbind) a valid context for the target device.  The buffer object
*   must be large enough (have enough memory allocated) to hold the data. See the
*   allocating memory topic above.
*
* - The \if_dotnet Memset() \else memset() \endif methods fill a buffer object with a
*   specified value.  Before using these methods on a non-CPU buffer object, you
*   must bind (then unbind) a valid context for the target device.  The buffer object
*   must be large enough (have enough memory allocated) to hold the data. See the
*   allocating memory topic above.
*
* - The \if_dotnet Map() \else map() \endif methods that have a buffer object
*   parameter "map" one buffer object into another, usually to allow the data to be
*   accessed on a different device.  When the access mode is READ_ONLY or READ_WRITE,
*   the contents of the source buffer are available through the target buffer object
*   after the \if_dotnet Map() \else map() \endif call. If necessary, Open Inventor
*   will automatically do an inter-device transfer of the data. When the access mode
*   is SET or READ_WRITE, the contents of the modified target buffer object are available
*   through the source buffer object after the \if_dotnet Unmap() \else unmap() \endif
*   call. If necessary, Open Inventor will automatically do an inter-device data transfer.
*
* - Some buffer object classes provide a \if_dotnet Map() \else map() \endif method
*   that maps the buffer's data into CPU memory and returns a \if_cpp pointer. \else
*   reference. \endif  If necessary, Open Inventor will automatically do an inter-device
*   transfer of the data. While the buffer is mapped the application can directly modify
*   its data using the returned \if_cpp pointer. \else reference. \endif  If the access
*   mode is SET or READ_WRITE, the modified data will be available through the buffer
*   object after the \if_dotnet Unmap() \else unmap() \endif call.  If necessary, Open
*   Inventor will automatically do an inter-device data transfer.
*
* @SEE_ALSO
*   SoCpuBufferObject,
*   SoGpuBufferObject,
*   SoGLBufferObject,
*   SoCpuDevice,
*   SoGLDevice,
*   SoBufferedShape
*
* [OIVNET-WRAPPER-CLASS ALL_DERIVABLE]
*/
class INVENTORBASE_API SoBufferObject: public SoRefCounter, public SoTypedObject
{
  SO_TYPED_CLASS_ABSTRACT_HEADER()
  SO_PIMPL_BASE_PUBLIC_HEADER(SoBufferObject)
  RENDERER_RESOURCE(SoBufferObject);

public:

  /** This enum provides the possible access modes for a mapped buffer object. */
  enum AccessMode
  {
    /** The target buffer initially has the same contents as the source buffer.
     */
    READ_ONLY,

    /** The target buffer contents are initially undefined (although they may be
     *  the same as the source buffer) but the source buffer will contain the
     *  contents of the target buffer after the unmap.
     */
    SET,

    /** The target buffer initially has the same contents as the source buffer
     *  and the source buffer will contain the contents of the target buffer
     *  after the unmap.
     */
    READ_WRITE
  };

  /**
   * Locks the buffer against concurrent calls from different threads. @BR
   * This is a blocking call.  In other words it will not return
   * immediately if another thread currently has the buffer locked.
   */
  void lockBuffer();

  /**
   * Unlocks the buffer object. @BR
   * If a thread calls lockBuffer() then it must call unlockBuffer()
   * to allow other threads to lock the buffer.
   */
  void unlockBuffer();

  /**
   * Sets the size in bytes of the buffer object. @BR
   *
   * If the requested size is the same as the current size, this method does
   * nothing and returns true.  If there is existing memory that is owned
   * by the buffer object, that memory is released. If the requested size is
   * zero, the buffer object is now empty.
   *
   * @param size The requested size in bytes.
   * @return true if the operation was successful.
   */
  virtual bool setSize( size_t size );

  /**
   * Returns the size, in bytes, of the buffer object.
   *
   * @return The size in bytes of the buffer object.
   */
  virtual size_t getSize() const;

  /**
   * Map the buffer to a system memory address and allows the mapping of a sub part
   * of the buffer object into CPU memory. 
   *
   * Notes:
   * - The unmap() method must be called before using the buffer.
   *
   * @return \if_cpp Pointer \else Reference \endif to data in the mapped region of the buffer memory.
   * [OIV-WRAPPER-RETURN-TYPE ARRAY{GetSize()}]
   */
  virtual void* map(AccessMode accessMode, size_t offset = 0, size_t count = SO_BUFFER_SIZE_ALL);

  /**
   * Unmaps the buffer from CPU address space.
   */
  virtual void unmap();

  /**
   * Maps the current buffer object into the specified buffer object. @BR
   * This is useful in order to use a buffer in multiple contexts.
   * The default is to map the entire contents of this buffer, but it is also possible
   * to map a subset of this buffer's memory using the @I startPosition @i and
   * @I mappingSize @i parameters.
   *
   * A source buffer may be mapped into multiple target buffers. However a target
   * buffer can only be mapped from one source buffer at a time. If a different source
   * buffer is currently mapped into the target buffer, it is automatically unmapped
   * and may be left in an undefined state.
   *
   * Note: If the current buffer is empty or startPosition is beyond the end of the
   * managed memory, the buffer is not mapped (and an error message is issued in debug builds).
   *
   * @param targetBufferObject The buffer object which will be the mapped version of
   *                           this buffer.
   * @param accessMode The access mode used for the mapping.
   * @param startPosition Offset (in bytes) in source buffer to map from (default is start of buffer).
   * @param mappingSize Size (in bytes) from the startPosition, if SO_BUFFER_SIZE_ALL then the whole buffer is mapped.
   *
   */
  virtual void map( SoBufferObject* targetBufferObject, AccessMode accessMode, size_t startPosition = 0, size_t mappingSize = SO_BUFFER_SIZE_ALL );

  /**
   * Unmap this buffer from the specified buffer object. @BR
   * In other words, remove the specified target buffer from the list of buffers which
   * this buffer is mapped to. If the access mode supports writing, the specified buffer is
   * sync'd with the current buffer.  An error is reported (in debug builds) if the
   * buffer is not mapped to the specified buffer
   *
   * @param bufferObject Buffer to be unmapped.
   */
  virtual void unmap( SoBufferObject* bufferObject );

  /**
   * Copies data from the specified buffer object into this buffer object. @BR
   * If the size or the offset are not valid an error is reported (SoDebugError).
   * This buffer is not resized, if it is too small an error is reported.
   *
   * @warning Source and destination overlaping support is implementation dependent, if not supported an error is reported.
   *
   * @param sourceBufferObject The buffer object to be copied.
   * @param destOffset The starting offset in the destination buffer object, useful for data subsets.
   * @param sourceOffset The starting offset in the source buffer object, useful for data subsets.
   * @param copySize The number of bytes to copy from the source buffer object (SO_BUFFER_SIZE_ALL means all).
   */
  virtual void memcpy( SoBufferObject* sourceBufferObject, size_t destOffset = 0, size_t sourceOffset = 0, size_t copySize = SO_BUFFER_SIZE_ALL );

  /**
   * This function sets the contents of (or a portion of) this buffer object to the specified value.
   *
   * The valueSize parameter provides a way to do a memset with float, short, byte, etc values.
   * The number of bytes set in this buffer object is exactly valueSize*count. The first byte changed
   * in this buffer is given by the offset argument.
   *
   * @param value is a pointer to the value to set in the buffer.
   * @param valueSize The size in bytes of the value. Default is 1 byte.
   * @param offset The offset in bytes (where to start setting values). Default is 0.
   * @param count The number of values to set.  Default is number of bytes in buffer.
   *
   * \if_cpp
   * EXAMPLE
   *   \par
   *   \code
   *   unsigned char memset_value = 0;
   *   buffer->memset( &memset_value );
   *   \endcode
   * \endif
   * \if_dotnet
   * EXAMPLE
   *   \par
   *   \code
   *   IntPtr ptr = Marshal.AllocHGlobal(sizeof(int));
   *   Marshal.WriteInt32( ptr, 0 );
   *   buffer.Memset( ptr );
   *   Marshal.FreeHGlobal( ptr );
   *   \endcode
   * \endif
   * [OIVNET-WRAPPER-ARG POINTER_VALUE,IN,IN,IN]
   */
  virtual void memset( void* value, size_t valueSize = 1, size_t offset = 0, size_t count = SO_BUFFER_SIZE_ALL );

  /**
    * Create a new buffer with the same properties as the current one. @BR
    * The new instance will have the same context or device properties, but no memory is allocated.
    */
  virtual SoBufferObject* createInstance() const = 0;

  /**
   * Free the memory allocated by the buffer object.
   */
  virtual void clearInstance() = 0;

  /**
   * Returns the device context where this buffer is valid.
   *
   * @return A device context.
   */
  SoDeviceContext* getContext() const;

  /**
   * Returns a pointer to the buffer object which is mapped by the actual object.
   */
  SoBufferObject* getMappedBufferObject() const;

  /**
   * Returns the access mode used for the buffer mapping.
   */
  AccessMode getMappedBufferObjectAccessMode();

  /**
   * Returns the position in the source buffer mapped in this buffer.
   *
   * @return Returns a position in bytes.
   */
  size_t getMappedBufferObjectPosition() const;

  /**
   * Returns the size of the mapped area in bytes.
   *
   * @return The size of the mapped area.
   */
  size_t getMappedBufferObjectSize() const;

SoINTERNAL public:

  // should not be used elsewhere than in SoMemoryObject for compatibility purpose
  void forceSize(size_t size);

  // This function does a copy of the buffer to a CPU buffer object
  virtual void copyToCpuBuffer( SoCpuBufferObject* targetBufferObject, size_t destOffset = 0, size_t sourceOffset = 0, size_t copySize = SO_BUFFER_SIZE_ALL );

  /** Like #createInstance but also allocate memory and copy the content of this buffer. */
  virtual SoBufferObject* clone() const;

  // Set mapped buffer object info
  void setMappedBufferObject(SoBufferObject* bufferObject, SoBufferObject::AccessMode accessMode,
                             size_t startPosition, size_t size);

  // Clear mapped buffer object info
  void clearMappedBufferObject();

  /**
   * Returns a unique (for all SoBufferObject) counter which is different each time the buffer object data is modified.
   * Counter is automatically incremented by calls to setSize, map, etc. Initial value is 1.
   */
  uint64_t getModificationsCounter() const;

  /**
   * Increments the modification counter of the buffer object.
   */
  void incrementModificationsCounter() const;

  /**
   * Returns the cache manager object.
   *  This object manages a cache of SoCpuBufferObject objects.
   *  SoBufferObject creates an instance of this class that is primarily used
   *  for the LDM tile cache (see SoLDMGlobalResourceParameters for more information).
   *  The buffer object cache can be accessed using the SoBufferObject static method
   *  getBufferObjectCache.  Default size is 50.
   */
  static SoCpuBufferObjectCache* getBufferObjectCache();

  enum class UsageType
  {
    VERTEX = 1,
    INDEX = 2,
    UNIFORM = 4,
    STORAGE = 8,
    GENERIC = 255,
  };

  /**
   * Specify for which type of buffer this buffer object will be used.
   */
  void setUsageType(UsageType usageType);

  /**
   *  See @setUsageType.
   */
  UsageType getUsageType() const;

protected:

  /**
   * Default constructor.
   *
   * The default constructor initializes the internal OpenInventor system and creates the
   * mutex which is used to lock/unlock the buffer.
   */
  SoBufferObject();

  /**
   * Destructor.
   * Protected as BufferObject must be deleted through ref/unref.
   */
  virtual ~SoBufferObject();

  /**
   * Stores the Device context active when the buffer object was allocated
   * @param context The device context used during the allocation.
   */
  void setDeviceContext(SoDeviceContext* context);

  /**
   * Checks that the arguments are valid for a specific memory copy.
   */
  bool checkCopyConditions(size_t sourceSize, size_t sourceOffset,
                           size_t targetSize, size_t targetOffset, size_t copySize);

  /** 
   * This method is used by custom buffer object classes to indicate
   * that the buffer has been modified so other classes can track the changes.
   */
  void touch();

private:

  /**
   * copy operator not implemented so prevent its use to avoid double delete
   * of mutex and other things
   */
  SoBufferObject(const SoBufferObject&)
  : SoRefCounter(), SoTypedObject() {}

  // Instance of the class responsible for the cache of BufferObject.
  static SoCpuBufferObjectCache* s_bufferObjectCache;

  friend class SoCpuBufferObject;
};


#ifdef _WIN32
#pragma warning(pop)
#endif
