/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Nicolas DAGUISE (Oct 2010)
**=======================================================================*/

#if !defined SO_DEVICE_CONTEXT_SHARE_GROUP_H
#define SO_DEVICE_CONTEXT_SHARE_GROUP_H

#include <Inventor/misc/SoRefCounter.h>
#include <Inventor/STL/set>
#include <Inventor/threads/SbThreadMutex.h>
#include <Inventor/devices/SoSharedGroupReferenceList.h>
#include <Inventor/SbEventHandler.h>

class SoDeviceContext;
class SoBaseContextCache;
class SoBaseContextObject;

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

/**
* @VSGEXT Shared context management class.
*
* @ingroup GLDevice
*
* @DESCRIPTION
*
* This class provides functions to manage a shared group of device contexts.
*
*/
class INVENTORBASE_API SoDeviceContextSharedGroup : public SoRefCounter
{
public:
  /** Type definition for SoDeviceContext set management. */
  typedef std::set<SoDeviceContext*> SoDeviceContextSet;

  /** Creates a new SharedGroup. */
  SoDeviceContextSharedGroup();

  /** Creates a new SharedGroup with specific id. */
  SoDeviceContextSharedGroup(int id);

  /** Adds a device context to the group. */
  void add(SoDeviceContext* ctx);

  /** Removes a device context from the group. */
  void remove(SoDeviceContext* ctx);
  
  /** Sets a specific id for this shared group. */
  void setId( int id );

  /** Gets the id of this shared group. */
  int getId() const;

  /** Gets the list of contexts managed by this shared group. */
  const SoDeviceContextSet& getContexts() const;

  /** Returns the first context found in the shared group. */
  SoDeviceContext* getPrimaryContext() const;

  /** Returns TRUE if the passed context is part of the shared group*/
  bool contains(const SoDeviceContext* ctx) const;
  
  /**
   * Increment reference count.
   *
   * Object is the ContextObject that should be triggered when this context is deleted.
   */
  void ref(SoBaseContextCache* object = NULL);

  /**
   * Decrement reference count.
   *
   * Object is the ContextObject that was referenced originally.
   *
   * If this call causes the reference count to go from 1 to 0 then the
   * context is automatically deleted and all ContextObject that previously
   * called ref(SoContextObject*) will be triggered for deletion
   * (potentially asynchronously).
   */
  void unref(SoBaseContextCache* object = NULL);


  /** Returns TRUE if the specified object is already in the list of notified objects.
   *  To add or remove an object, from this list, the ref(object) and unref(object) methods can be used.
   */
  bool contains(SoBaseContextCache* object) const;

  /**
   * Force all attached objects to release the context, by calling
   * their respective release function, and force deletion of the contained context.
   */
  virtual void dispose();

SoINTERNAL public:
  typedef int SharedGroupId;

  /** Callback triggered when shared resources must be deleted */
  void setDisposeCallback(void (*f)(SoDeviceContextSharedGroup*));

  /** Callback triggered when this class will be deleted */
  void setDestroyNotificationCallback(void (*f)(SoDeviceContextSharedGroup*));
  
  void setTargetMergedGroup(SoDeviceContextSharedGroup* sharedGroup);

  /**
   * Add an object to the list to be freed when this context is bond or deleted.
   */
  void addToPendingDeleteList(SoBaseContextObject *obj);

  /**
   * Used to check if an object pointer still exists in any existing context lists and remove it.
   */
  static void checkRefIssues(SoBaseContextObject* object);
  static void checkRefIssues(SoBaseContextCache* cache);

  /** delete objects that are in the waiting to be freed list (see addToPendingDeleteList) */
  void clearPendingDeleteList();

  void setDisposing( bool b );
  
  /** returns the threadId that allocate this context */
  SbThreadId_t getThreadId() const;

  /** the global shared group id counter */
  static int getNewSharedIdGroup();

  /** Prints all shared group and which contexts are in. */
  static void printSharedGroups();

  /** Returns true if this shared group has at least one valid context.
   *  A valid context is a context which returns true when it's isValid function returns true.
   */
  bool hasValidContext() const;

  /** Returns the first valid context found in the shared group. */
  SoDeviceContext* getPrimaryValidContext() const;

  // return debugging trace level requested
  static int getDebuggingLevel()
  { return s_debuggingLevel; }

  /** When there is no context on state, force a fixed shared group id */
  inline static int getFallbackContextGlobalSharedId()
  {
    return -2;
  }
  
  /**
   * Returns a pointer to the SoDeviceContextSharedGroup that corresponds to
   * this SharedGroupId if this Shared Group is still valid (i.e. contained in
   * the s_sharedGroupList map). Returns a null pointer otherwise.
   */
  static SoDeviceContextSharedGroup* getSharedGroupFromId( SharedGroupId id );

protected:

  /** class destructor. delete can only be done through the ref/unref process. */ 
  virtual ~SoDeviceContextSharedGroup();

  /**
   * This function force all attached object to release the context, by calling
   * their respective release function.
   */
  void forceReleaseAttachedObject();

  bool m_isDisposing;

private:

  void commonConstructor();

  // Returns the context to use for delete operation
  // - if currently binded context is in the shared group then use it
  // - else use the first available context in the group (getPrimaryValidContext())
  SoDeviceContext* getDeviceContextToBindForDelete() const;

  SoDeviceContextSet m_ctxSet;
  int m_id;
  SoDeviceContextSharedGroup* m_mergedGroup;

  /** Keep track of all created sharedGroup */
  typedef std::list<SoDeviceContextSharedGroup*> SoDeviceContexSharedGroupList;
  static SbThreadSpinlock s_sharedGroupListLock;
  static SoDeviceContexSharedGroupList s_sharedGroupList;
  static int s_sharedGroupId;

  /** handle list of object link to this context **/
  typedef std::list<SoBaseContextCache*> SoBaseContextCacheList;
  SoBaseContextCacheList m_listContextedObject;
  bool m_isClearingPendingList;

  /** head of the m_intrusivelistContextedObject used for optimization purpose */
  SoSharedGroupReferenceList m_intrusivelistContextedObject;

  /** handle list of object that must be freed when bind or delete is called */
  typedef std::set<SoBaseContextObject*> SoBaseContextObjectList;
  SoBaseContextObjectList m_waitingToBeFreed;

  /** keep track of which thread created this context */
  SbThreadId_t m_threadId;
  
  /** This lock prevents other threads to add objects to delete list (m_waitingToBeFreed) while we are processing it.
   * This can happen when different threads use shared contexts so all the threads
   * perform operations on the same share group.
   * Also this lock is used to protec the list of contexted objects becaue multiple threads can add and remove objects at the same time
   * under the same sharing conditions.
   */
  mutable SbThreadMutex m_sharedGroupMembersMutex;

  // Controlled by environment variable IV_DEBUG_SHARED_GROUP
  // 0 means disabled
  // 1 means show weird situation
  // 2 means show more trace
  static int s_debuggingLevel;

  /** Callback called when this shared group is destroyed */
  SbEventHandler<SoDeviceContextSharedGroup*> m_destroyNotificationCallback;
  SbEventHandler<SoDeviceContextSharedGroup*> m_disposeNotificationCallback;
};

// INLINE METHODS
inline void SoDeviceContextSharedGroup::setDisposing( bool b )
{
  m_isDisposing = b;
}
  
inline SbThreadId_t SoDeviceContextSharedGroup::getThreadId() const
{
  return m_threadId;
}

#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif // SO_DEVICE_CONTEXT_SHARE_GROUP_H


