/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#if !defined SOGLBUFFEROBJECT_H
#define SOGLBUFFEROBJECT_H

#include <Inventor/devices/SoInteropBufferObject.h>
#include <Inventor/sys/SoGLType.h>

SO_PIMPL_PUBLIC_DECLARATION(SoGLBufferObject)
class SbVec2i32;

/**
* @VSGEXT OpenGL buffer object class.
*
* @ingroup GLDevice
*
* @DESCRIPTION
*
* This class provides management functions for OpenGL memory buffers.
*
* NOTE: Since Open Inventor 10.4, applications should use SoGPUBufferObject,
* instead of SoGLBufferObject, in most cases. For example to store vertex
* data in GPU memory for use with SoBufferedShape or SoVertexShaderParameterBufferObject.
*
* This class can be used as a wrapper for GL memory buffer management when
* the application manages OpenGL code directly.
*
* NOTES:
* - Creating a OpenGL buffer requires a current OpenGL context. @BR
*   For example, by calling the viewer's bindNormalContext() method or
*   by creating an SoGLContext object and binding (then unbinding) it.
*
* - Modifying the properties of an OpenGL buffer require a current OpenGL context. @BR
*   This includes changing properties (e.g. setTarget()), allocating memory
*   (setSize()) and copying data into the buffer (e.g. memcpy()). Mapping a
*   buffer does not require explicitly binding a context. 
*
* - The setTarget() method must be called before any operation, as long as
*   OpenGL requires the target for that operation.
*
* - For detailed use of this class it may be helpful to review and understand
*   the use of OpenGL Buffer Objects, particularly Vertex Buffer Objects (VBO).
*
* @if_cpp
* - SoBufferObject classes are reference counted objects (just like nodes, 
*   paths, etc).  Therefore you cannot create an SoBufferObject on the 
*   stack. And to delete an SoBufferObject you must ref and unref the 
*   object.  The SoRef smart pointer is a safe and convenient way to
*   create buffer objects. See examples in SoBufferObject.
* \endif
*
* See SoBufferObject for general information about buffer objects.
*
* See SoBufferedShape for an example of storing vertices in an OpenGL buffer.
*
* @EXAMPLE
* Load data into a buffer object.
*
* Create a GPU buffer object and load data from an array in memory:
* \if_cpp
* \code
* const float vertices[][3] = {
*      1,0.5,0, 0,1,0, -1,0.5,0,
*     -1,-1,0, 1,-1,0, 1,0,0, -1,0,0,
*     -1,-1.5,0, 1,-1.5,0
* };
* const int NUM_COORDS = sizeof(vertices) / sizeof(SbVec3f);
*
* // Wrap coordinate array in a CPU buffer object
* SoRef<SoCpuBufferObject> cpuBuffer = 
*     new SoCpuBufferObject( (void*)coords, NUM_COORDS * sizeof(SbVec3f) );
*
* // Create a GPU (OpenGL) buffer and load data from CPU buffer
* SoRef<SoGLContext> glContext = new SoGLContext(true);
* glContext->bind();
*   SoRef<SoGLBufferObject> gpuBuffer = new SoGLBufferObject( SoGLBufferObject::STATIC_DRAW );
*   gpuBuffer->setTarget( SoGLBufferObject::ARRAY_BUFFER );
*   gpuBuffer->setSize  ( cpuBuffer->getSize() ); // Set the buffer size (allocate memory)
*   gpuBuffer->memcpy   ( cpuBuffer.ptr() );      // Copy data into the buffer
* glContext->unbind();
* \endcode
* Load data into an existing GPU buffer from a CPU buffer:
* \code
* SoGLContext* ctx = (SoGLContext*)gpuBuffer->getContext();
* ctx->bind();
*   gpuBuffer->setSize( cpuBuffer->getSize() ); // Set the buffer size (allocate memory)
*   gpuBuffer->memcpy ( cpuBuffer.ptr() );      // Copy data into the buffer
* ctx->unbind();
* \endcode
* Or
* \code
* cpuBuffer->map( gpuBuffer.ptr(), SoBufferObject::READ_ONLY );
* cpuBuffer->unmap();
* \endcode
* \endif
*
* \if_dotnet
* \code
* float[] vertices = new float[9 * 3] {
*      1.0f, 0.5f,0.0f, 0.0f, 1.0f,0.0f, -1.0f,0.5f,0.0f,
*     -1.0f,-1.0f,0.0f, 1.0f,-1.0f,0.0f,  1.0f,0.0f,0.0f, -1.0f,0.0f,0.0f,
*     -1.0f,-1.5f,0.0f, 1.0f,-1.5f,0.0f
* };
*
* // Wrap coordinate array in a CPU buffer object
* SbNativeArray<float> vertArray = new SbNativeArray<float>(vertices);
* SoCpuBufferObject vertBuf = new SoCpuBufferObject( (SbNativeArray<byte>)vertArray );
*
* // Create a GPU (OpenGL) buffer and load data from CPU buffer
* SoGLContext glContext = new SoGLContext(true);
* glContext.Bind();
*     SoGLBufferObject gpuBuffer = new SoGLBufferObject(SoGLBufferObject.Usages.STATIC_DRAW);
*     gpuBuffer.SetTarget( SoGLBufferObject.BufferObjectTargets.ARRAY_BUFFER );
*     gpuBuffer.SetSize( cpuBuffer.GetSize() );  // Set the buffer size (allocate memory)
*     gpuBuffer.Memcpy ( cpuBuffer );            // Copy data into the buffer
* glContext.Unbind();
* \endcode
* Load data into an existing GPU buffer from a CPU buffer:
* \code
* SoGLContext ctx = (SoGLContext)gpuBuffer.GetContext();
* ctx.Bind();
*     gpuBuffer.SetSize( cpuBuffer.GetSize() ); // Set the buffer size (allocate memory)
*     gpuBuffer.Memcpy ( cpuBuffer );           // Copy data into the buffer
* ctx.Unbind();
* \endcode
* Or
* \code
* cpuBuffer.Map( gpuBuffer, SoBufferObject.AccessModes.READ_ONLY );
* cpuBuffer.Unmap();
* \endcode
* \endif
*
* \if_java
* \code
* float[] vertices = {
*         1.0f, 0.5f,0.0f, 0.0f, 1.0f,0.0f, -1.0f,0.5f,0.0f,
*        -1.0f,-1.0f,0.0f, 1.0f,-1.0f,0.0f,  1.0f,0.0f,0.0f, -1.0f,0.0f,0.0f,
*        -1.0f,-1.5f,0.0f, 1.0f,-1.5f,0.0f
* };
*
* // Create a GPU (OpenGL) buffer and allocate memory
* SoGLContext glContext = new SoGLContext( true );
* glContext.bind();
*     SoGLBufferObject gpuBuffer = new SoGLBufferObject( SoGLBufferObject.Usages.STATIC_DRAW );
*     gpuBuffer.setTarget( SoGLBufferObject.BufferObjectTargets.ARRAY_BUFFER );
*     gpuBuffer.setSize( vertices.length * Float.SIZE/8 ); // Set the buffer size (allocate memory)
* 
* // Copy data into the buffer object
* FloatBuffer vertData = gpuBuffer.map( SoBufferObject.AccessModes.SET ).asFloatBuffer();
*     vertData.put(vertices);
* gpuBuffer.unmap();
*
* glContext.unbind();
* \endcode
* Load data into an existing GPU buffer from a CPU buffer:
* \code
* SoGLContext ctx = (SoGLContext)gpuBuffer.getContext();
* ctx.bind();
*     gpuBuffer.setSize( cpuBuffer.getSize() ); // Set the buffer size (allocate memory)
*     gpuBuffer.memcpy ( cpuBuffer );           // Copy data into the buffer
* ctx.unbind();
* \endcode
* Or
* \code
* cpuBuffer.map( gpuBuffer, SoBufferObject.AccessModes.READ_ONLY );
* cpuBuffer.unmap();
* \endcode
* \endif
*
*
* @EXAMPLE
* Access data stored in a GPU buffer object.
* \par
* \if_cpp
* \code
* float* data  = (float*)gpuBuffer->map( SoBufferObject::READ_ONLY );
* float  value = data[0];
* value = data[1];
*     . . .
* gpuBuffer->unmap();
* \endcode
* \endif
*
* \if_dotnet
* \code
* SbNativeArray<float> data = (SbNativeArray<float>)gpuBuffer.Map( SoBufferObject.AccessModes.READ_ONLY );
* float value = data[0];
* value = data[1];
*     . . .
* gpuBuffer.Unmap();
* \endcode
* \endif
*
* \if_java
* \code
* FloatBuffer data = gpuBuffer.map( SoBufferObject.AccessModes.READ_ONLY ).asFloatBuffer();
* float value = data.get(0);
* value = data.get(1);
*     . . .
* gpuBuffer.unmap();
* \endcode
* \endif
*
* @SEE_ALSO
* SoCpuBufferObject
*/
class INVENTORGL_API SoGLBufferObject : public SoInteropBufferObject
{
  SO_TYPED_CLASS_HEADER();
  SO_PIMPL_PUBLIC_HEADER(SoGLBufferObject)

public:

  /**
   * This enum declares the possible targets of the buffer.
   */
  enum BufferObjectTarget
  {
    /** 
    * The buffer is used as a pixel pack buffer, it can be used as texture.
    * For instance as a Texture Buffer Object (TBO).
    */
    PIXEL_PACK_BUFFER,

    /**
    * The buffer is used as a pixel unpack buffer, it can be used for readback operation.
    * For instance as a Pixel Buffer Object (PBO).
    */
    PIXEL_UNPACK_BUFFER,

    /**
    * The buffer is used as an array buffer, it can be used for vertices, normals, colors. 
    * For instance as a Vertex Buffer Object (VBO). Tis is the default value.
    */
    ARRAY_BUFFER,

    /**
     * The buffer is used as a pixel pack buffer, it is used to specify the indices for indexed geometries. 
     * For instance a Vertex Buffer Object (VBO).
     */
    ELEMENT_ARRAY_BUFFER,

    /**
     * The buffer is used as a shader storage buffer, it can be used to perform random access reads, writes and
     * atomic memory operations from a shader.
     */
    SHADER_STORAGE_BUFFER
  };

  /**
   * This enum declares the possible usages of the memory allocated for the buffer.
   * This is a hint to the OpenGL driver implementation as to how a buffer object's 
   * data store will be accessed. This enables the OpenGL implementation to make 
   * more intelligent decisions that may significantly impact buffer object performance. 
   * It does not, however, constrain the actual usage of the data store.
   * usage can be broken down into two parts: first, the frequency of access (modification 
   * and usage - STATIC, STREAM, DYNAMIC), and second, the nature of that access - DRAW, COPY, READ.
   */
  enum Usage
  {
    /**
    * The data store contents will be modified once and used at most a few times.
    * The data store contents are modified by the application, and used as the source 
    * for GL drawing and image specification commands.
    */
    STREAM_DRAW,

    /**
    * The data store contents will be modified once and used at most a few times.
    * The data store contents are modified by reading data from the GL, and used 
    * to return that data when queried by the application.
    */
    STREAM_READ,

    /**
    * The data store contents will be modified once and used at most a few times.
    * The data store contents are modified by reading data from the GL, and used 
    * as the source for GL drawing and image specification commands.
    */
    STREAM_COPY,

    /**
    * The data store contents will be modified once and used many times.              
    * The data store contents are modified by the application, and used as the source 
    * for GL drawing and image specification commands.
    */
    STATIC_DRAW,

    /**
    * The data store contents will be modified once and used many times.
    * The data store contents are modified by reading data from the GL, and used 
    * to return that data when queried by the application.
    */
    STATIC_READ,

    /**
    * The data store contents will be modified once and used many times.
    * The data store contents are modified by reading data from the GL, and used 
    * as the source for GL drawing and image specification commands.
    */
    STATIC_COPY,

    /**
    * The data store contents will be modified repeatedly and used many times.
    * The data store contents are modified by the application, and used as the source 
    * for GL drawing and image specification commands.
    */
    DYNAMIC_DRAW,

    /**
    * The data store contents will be modified repeatedly and used many times.
    * The data store contents are modified by reading data from the GL, and used 
    * to return that data when queried by the application.
    */
    DYNAMIC_READ,

    /**
    * The data store contents will be modified repeatedly and used many times.
    * The data store contents are modified by reading data from the GL, and used 
    * as the source for GL drawing and image specification commands.
    */
    DYNAMIC_COPY
  };

  /**
   * Constructor.
   *
   * @param usage The intended usage of this buffer. @useenum{Usage}
   */
  SoGLBufferObject( Usage usage );

  /**
   * Query if SoGLBufferObjects are available on this system.
   *
   * @return Returns true if SoGLBufferObjects are available on this system.
   */
  static bool isAvailable();

  /**
   * Specify the buffer target, which defines possible usage of the buffer.
   *
   * WARNING: The function setTarget must be called before any operation for which
   * OpenGL requires the target to be specified.
   * 
   * @param target The new target of the buffer. @useenum{BufferObjectTarget}
   */
  void setTarget( BufferObjectTarget target );

  /**
   * Returns the current buffer target.
   *
   * @return The actual buffer target. @useenum{BufferObjectTarget}
   */
  BufferObjectTarget getTarget() const;

  /**
   * Set the size of the buffer in bytes.
   *
   * @param size New size in bytes of the buffer.
   *
   * Notes:
   * - Causes memory to be allocated for the buffer.
   * - A valid OpenGL context (see SoGLContext) must be bound to perform this operation.
   * - If the buffer is already allocated, the memory is freed and re-allocated with the new size. @BR
   *   Therefore if you increase the size of the buffer, the original data is lost!
   */
  virtual bool setSize( size_t size );

  /**
   * Map the current buffer object into the specified buffer object.
   *
   * @param targetBufferObject The buffer object which will be the mapped version of
   *                           this buffer.
   * @param accessMode The access mode used for the mapping. @useenum{AccessMode}
   * @param startPosition offset in source buffer to map from (default is start of buffer).
   * @param mappingSize size from the startPosition, if SO_BUFFER_SIZE_ALL then the whole buffer is mapped.
   * 
   */
  virtual void map( SoBufferObject* targetBufferObject, AccessMode accessMode, size_t startPosition = 0, size_t mappingSize = SO_BUFFER_SIZE_ALL );

  /**
   * Unmap the specified buffer object.
   */
  virtual void unmap( SoBufferObject* bufferObject );

  /**
   * Bind the current buffer to the specified target so it's usable by OpenGL operations.
   *
   * Notes:
   * - Most applications will not need to use this method, because the other methods in this
   *   class (map, memcpy, etc) bind the buffer to its target automatically. It may be useful
   *   when making direct calls to OpenGL. It corresponds to the glBindBuffer() call.
   * - A valid OpenGL context (see SoGLContext) must be bound before calling this method.
   */
  void bind();

  /**
   * Unbind the buffer.
   *
   * Notes:
   * - A valid OpenGL context (see SoGLContext) must be bound before calling this method.
   * - Most applications will not need to use this method, because the other methods in this
   *   class (map, memcpy, etc) unbind the buffer automatically. It may be useful when making
   *   direct calls to OpenGL. It corresponds to the glBindBuffer() call with an id of 0.
   */
  void unbind();

  /** 
   * This function extends the map(AccessMode) method by allowing the mapping of a sub part
   * of the buffer object into CPU memory. 
   * On some systems this method has better performance.
   *
   * Notes:
   * - It is not necessary to bind an OpenGL context before calling this method.
   * - The unmap() method must be called before using the buffer for any OpenGL operation,
   *   otherwise the operation will produce an error.
   *
   * @return \if_cpp Pointer \else Reference \endif to data in the mapped region of the OpenGL buffer.
   * [OIV-WRAPPER-RETURN-TYPE ARRAY{GetSize()}]
   */
  virtual void* map( AccessMode accessMode, size_t offset = 0, size_t count = SO_BUFFER_SIZE_ALL );

  /**
   * Unmaps the buffer using the regular unmap function.
   */
  virtual void unmap();

  /**
   * Query if the buffer is valid in the current context.
   *
   * Notes:
   * - It is not necessary to bind an OpenGL context to call this method.
   * - The buffer object will be invalid until memory is allocated,
   *   either explicitly (see #setSize()) or implicitly (e.g. mapping
   *   from another buffer).
   * - The buffer object will be invalid after calling #clearInstance().
   */
  bool isValid();

  /**
   * Returns the OpenGL id of the buffer.
   */
  GLuint getId() const;

  /**
    * Create a new buffer with the same properties as the current one.
    * The new instance will have the same context and device properties, but
    * no memory is allocated so the new buffer is initially invalid.
    */
  virtual SoBufferObject* createInstance() const;

  /**
   * Free the memory allocated in the buffer object.
   * 
   * Notes:
   * - It is not necessary to bind an OpenGL context to call this method.
   * - After this call the buffer is invalid (until memory is allocated again).
   */
  virtual void clearInstance();

  /**
   * Copy data from a buffer into this GL buffer.
   *
   * Notes:
   * - It is not necessary to bind an OpenGL context before calling this method.
   * - The current buffer object must be large enough to hold the data that will be
   *   copied. See #setSize() to allocate memory.
   * - If the size or the offset are not valid an error is reported (SoDebugError).
   * - This buffer is not resized, if it is too small an error is reported.
   *
   * This function is a specialized function for speed-up. @BR
   * 
   * @param source The buffer object to be copied.
   * @param destOffset The starting offset in the destination buffer object, useful for data subsets.
   * @param sourceOffset The starting offset in the source buffer object, useful for data subsets.
   * @param copySize The number of bytes to copy from the source buffer object (SO_BUFFER_SIZE_ALL means all).
   */
  void memcpy(SoBufferObject* source, size_t destOffset = 0, size_t sourceOffset = 0, size_t copySize = SO_BUFFER_SIZE_ALL);
 
SoINTERNAL public:

  void copyToCpuBuffer( SoCpuBufferObject* targetBufferObject, size_t destOffset = 0, size_t sourceOffset = 0, size_t copySize = SO_BUFFER_SIZE_ALL );

  /** Return true if the GL buffer can be binded in the current context */
  bool isBindable();

  /** Copy OpenGL texture in GLBufferObject */
  void copyFromGLTexture(GLuint tex, const SbVec2i32& size, bool alpha);

  struct Configuration
  {
    size_t size;
    BufferObjectTarget target;
    Usage usage;
  };
  
protected:
  /**
   * Destructor.
   */
  virtual ~SoGLBufferObject();

  /** default constructor */
  SoGLBufferObject();

  bool checkContext();

};

#endif // SOGLBUFFEROBJECT_H
