/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Thibaut Andrieu (Jan 2011)
**=======================================================================*/

#if !defined _SO_ELLIPSE_SCREEN_DRAWER_H_
#define _SO_ELLIPSE_SCREEN_DRAWER_H_


#include <Inventor/drawers/SoPolyLineScreenDrawer.h>
#include <Inventor/fields/SoSFEnum.h>

/**
 * @VSGEXT Interactively draw an ellipse in normalized screen space.
 *
 * @ingroup drawers
 *
 * @DESCRIPTION
 *   This class can be used to interactively draw an ellipse on screen.
 *
 *   When the ellipse is finished, an event is raised to notify the application
 *   (see SoPolyLineScreenDrawer::onFinish ).  Points are returned as normalized
 *   coordinates in the range -1..1, which is conveniently the default view volume
 *   for an SoOrthographicCamera node.
 *
 *   The line color is specified by the #color field in the parent class SoPolyLineScreenDrawer.
 *
 *   Please see SoPolyLineScreenDrawer for general notes and code example.
 *
 *   Ellipse specific notes:
 *
 *   - The ellipse can be specified by the 2 corners of its bounding box (default) or 
 *     by its center and one corner of its bounding box (see #CreationMethod).
 *
 *   - The #simplificationThreshold field does not apply to this node. The number of
 *     points in the ellipse is specified using the #nbPoint field.
 *
 *   - The #isClosed field does apply to this node (but FALSE is not very useful).
 *
 *   - If user finishes an ellipse, then the next click will automatically clear
 *     the polyline (begin a new ellipse).
  *
 *   - SbEventHandlers:
 *     - OnStart : Triggered on LeftMouseDown input event.
 *     - OnMove : Triggered on MouseMove input events (after Start).
 *     - OnFinish : Triggered on LeftMouseUp input event.
*
 *   \if_cpp
 *   [C++] @BR Screen drawer classes must be initialized by calling SoInteraction::init(). High level
 *   viewer init methods, like SoWin::init(), automatically do this. However, if the application
 *   is calling SoDB::init() directly, then it is usually necessary to also call SoInteraction::init()
 *   before using the screen drawer classes.
 *   \endif
 *
 * @USAGE
 * - Press and drag left mouse to draw the ellipse.
 * - Release to finish the ellipse.
 * - Press Escape to cancel construction of the ellipse.
 *
 * @FILE_FORMAT_DEFAULT
 *    EllipseScreenDrawer {
 *    @TABLE_FILE_FORMAT
 *       @TR point        @TD []
 *       @TR color        @TD 1 0 0
 *       @TR simplificationThreshold      @TD 0
 *       @TR isClosed     @TD TRUE
 *       @TR method       @TD CORNER_CORNER
 *       @TR nbPoint      @TD 16
 *    @TABLE_END
 *    }
 *
 * @SEE_ALSO
 *    SoLassoScreenDrawer,
 *    SoPolyLineScreenDrawer,
 *    SoPolygonScreenDrawer,
 *    SoRectangleScreenDrawer,
 */
class INVENTOR_API SoEllipseScreenDrawer : public SoPolyLineScreenDrawer 
{
  SO_NODE_HEADER( SoEllipseScreenDrawer );

public:
  /** Constructor */
  SoEllipseScreenDrawer();

  /** 
   * Methods to create ellipse. 
   */
  enum CreationMethod
  {
    /** Initial point is a corner, current point is a corner. */
    CORNER_CORNER,
    /** Initial point is the center, current point is a corner. */
    CENTER_CORNER
  };

  /** Method to create ellipse.
   *  @useenum{CreationMethod}. Default is CORNER_CORNER. 
   */
  SoSFEnum method;

  /** Number of points in generated ellipse. 
   *  Default is 16. Must be > 0. 
   */
  SoSFUInt32 nbPoint;

  /** @copydoc SoPolyLineScreenDrawer::reset() */
  virtual void reset();

SoINTERNAL public:

  /** Register in database */
  static void initClass();

  /** Unregister from database */
  static void exitClass();

SoEXTENDER_Documented protected:

  /** Finish ellipse. */
  virtual void onKeyDown( SoHandleEventAction* action );

  /** Update ellipse. */
  virtual void onMouseDragging( SoHandleEventAction* action );

  /** Start creation of new ellipse. */
  virtual void onMouseDown( SoHandleEventAction* action );

  /** Finish creation of ellipse. */
  virtual void onMouseUp( SoHandleEventAction* action );

  /** Is an ellipse is creating. */
  inline bool isCreating() const { return m_initPoint[0] != -1 && m_initPoint[1] != -1; }

protected:

  /** Destructor. */
  virtual ~SoEllipseScreenDrawer();

private:

  /** 
   * Retrieve ellipse parameters from current method and current initPoint. 
   * @param currentPoint The current mouse move position.
   * @param xRadius half ellipse x axis.
   * @param yRadius half ellipse y axis.
   * @param center ellipse center.
   */
  void getEllipseParameters( const SbVec2f& currentPoint, 
    float& xRadius, 
    float& yRadius, 
    SbVec2f& center ) const;

  /** 
   * Initial point of ellipse (usually the mouseDown event). See #CreationMethod.
   * If (-1, -1), ellipse is not creating 
   */
  SbVec2f m_initPoint;

};


#endif // _SO_ELLIPSE_SCREEN_DRAWER_H_


