/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Thibaut Andrieu (Jan 2011)
**=======================================================================*/

#if !defined _SO_POLY_LINE_SCREEN_DRAWER_H_
#define _SO_POLY_LINE_SCREEN_DRAWER_H_

#include <Inventor/drawers/SoScreenDrawer.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFUShort.h>
#include <Inventor/fields/SoSFUInt32.h>
#include <Inventor/fields/SoSFColor.h>
#include <Inventor/fields/SoMFVec2f.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/SbEventHandler.h>
#include <Inventor/SbEventArg.h>

/**
 * @VSGEXT Interactively draw a polyline in normalized screen space.
 *
 * @ingroup drawers
 *
 * @DESCRIPTION
 *   This class is a base class used to allow the user to dynamically draw 
 *   line-based shapes (lasso, circle, rectangle, ...) on screen.  The
 *   geometry is defined in normalized screen space ([-1, 1]x[-1, 1]). 
 *
 *   Applications will typically use one of the derived classes, for example,
 *   SoEllipseScreenDrawer, SoRectangleScreenDrawer, SoPolygonScreenDrawer or SoLassoScreenDrawer.
 *
 *   The line color is specified by the #color field (default is red).
 *
 *   SoPolyLineScreenDrawer classes can automatically reduce the number of points
 *   in the shape drawn by the user.  This is especially useful with SoLassoScreenDrawer.
 *   See the #simplificationThreshold field.
 *
 *   Sub-classes can implement the onMouseDown, onMouseMove, etc... methods, 
 *   to add and edit points.
 *
 * Notes:
 *   - Viewer: @BR
 *       - Screen drawers, like draggers, depend on SoHandleEventAction to get events. Therefore
 *         when using one of the Open Inventor viewer classes, screen drawers will only work when
 *         the viewer is in "selection" mode (arrow cursor). (In "viewing" mode, mouse events are
 *         handled by the viewer to control the camera.) @BR @BR
 *       - The escape key (ESC) can be used to cancel creation of the polyline.
 *         However, currently it also changes the viewer mode (this is a known problem).
 *
 *   - Event handling: @BR
 *     The drawer will automatically begin a new polyline when it sees a mouse button 1 down event.
 *     However the drawer does not call setHandled(), so other nodes in the scene graph, e.g.
 *     SoEventCallback or a dragger, may also handle this event.
 *
 *   - Finish notification: @BR
 *       - An event is raised to notify the application when the line is finished.
 *         (see \if_dotnet OnFinish \else onFinish \endif).
 *         This event is called at the end of line finalization (after simplification etc).
 *         See SbEventHandler for methods to set a callback to be notified when this
 *         event is raised. @BR @BR
 *       - In this callback the application can retrieve coordinates from the #point field.
 *         These points are in Open Inventor normalized device coordinates (-1 to 1), not pixel coordinates. @BR @BR
 *       - The drawer does not call the SoHandleEventAction's setHandled() method
 *         for the event that finished the polyline. The application can call setHandled()
 *         in this callback. @BR @BR
 *       - With some drawers it is possible to get a call with zero points if the user
 *         pressed ESC to cancel the interaction.  Applications should check for this case. @BR @BR
 *       - With some drawers it is also possible to get a call with a non-zero number of points,
 *         but all points are the same coordinate. E.g. SoRectangleScreenDrawer stores four
 *         points even if the mouse down/up events are at the same location. @BR @BR
 *       - Typically the application will want to remove the polyline from the screen in this
 *         callback. See the next note.
 *
 *   - Part of the scene graph: @BR
 *       - Note that the polyline remains on the screen (because it is part of the scene graph)
 *         until \if_dotnet Clear() \else clear() \endif is called or the user starts a new interaction. @BR @BR
 *       - Changing the color field or the isClosed field will immediately change the appearance of a
 *         polyline currently on the screen.
 *
 *   - Simplification: @BR
 *     Simplification is done (if > 0) when the user releases the mouse button and also if
 *     the simplificationThreshold is changed.  But note that the old points are thrown away,
 *     so you can't re-simplify with a different value.
 *
 *   - isClosed field: @BR
 *     If true (default), the polyline will be closed while being drawn (not just when the user releases the mouse button).
 *     The default in this class is false, but most sub-classes set it to true.
 *
 *   - Coordinates: @BR
 *     This node and its subclasses use screen coordinates ranging from -1 to 1.  (Because
 *     this is the default view volume for an SoOrthographicCamera node.)  Be careful because
 *     other classes in Open Inventor use normalized screen coordinates ranging from 0 to 1.
 *     In particular the SbViewportRegion normalize() methods and the SbViewVolume methods
 *     that project from 3D to 2D and vice-versa.
 *
 *   - There should only be one SoScreenDrawer derived node in the sceneGraph (to avoid conflicts 
 *     in mouse events).
 *
 *   \if_cpp
 *   - [C++] @BR Screen drawer classes must be initialized by calling SoInteraction::init(). High level
 *     viewer init methods, like SoWin::init(), automatically do this. However, if the application
 *     is calling SoDB::init() directly, then it is usually necessary to also call SoInteraction::init()
 *     before using the screen drawer classes.
 *   \endif
 *
 *  @EXAMPLE
 *  Create a screen drawer, set the callback and add it to the scene graph:
 *  \if_cpp
 *    \code
 *    SoRectangleScreenDrawer* drawer = new SoRectangleScreenDrawer();
 *      drawer->onFinish.add( lineDrawerCallback );
 *      root->addChild( drawer );
 *    \endcode
 *  \endif
 *  \if_dotnet
 *    \code
 *    SoRectangleScreenDrawer drawer = new SoRectangleScreenDrawer();
 *      drawer.OnFinish += LineDrawerCallback;
 *      root.AddChild(drawer);
 *    \endcode
 *  \endif
 *  \if_java
 *    \code
 *    SoRectangleScreenDrawer drawer = new SoRectangleScreenDrawer();
 *    drawer.onFinish.addEventListener(new LineDrawerListener());
 *    root.addChild( drawer );
 *    \endcode
 *  \endif
 *  A finish callback that gets the number of points created by the screen drawer:
 *  \if_cpp
 *    \code
 *    void lineDrawerCallback( SoPolyLineScreenDrawer::EventArg& eventArg )
 *    {
 *      SoPolyLineScreenDrawer *drawer = eventArg.getSource();
 *      SoHandleEventAction *action = eventArg.getAction();
 *
 *      int numPoints = drawer->point.getNum();
 *      if (numPoints > 0) {
 *        // Use points
 *        SbVec2f pt = drawer->point[0];
 *      }
 *      drawer->clear();  // Remove polyline from screen
 *      action->setHandled();
 *    }
 *    \endcode
 *  \endif
 *  \if_dotnet
 *    \code
 *    public void LineDrawerCallback( object sender, SoPolygonScreenDrawer.EventArg eventArg )
 *    {
 *        SoPolyLineScreenDrawer drawer = eventArg.Source;
 *        SoHandleEventAction action = eventArg.Action;
 *
 *        int numPoints = drawer.point.Count;
 *        if (numPoints > 0)
 *        {   // Use points
 *            SbVec2f point0 = drawer.point[0];
 *        }
 *        drawer.Clear();  // Remove polyline from screen
 *        action.SetHandled();
 *    }
 *    \endcode
 *  \endif
 *  \if_java
 *    \code
 *    class LineDrawerListener implements SbEventListener<EventArg>
 *    {
 *      @Override
 *      public void onEvent(EventArg eventArg)
 *      {
 *        SoPolyLineScreenDrawer drawer = eventArg.getSource();
 *        SoHandleEventAction action = eventArg.getAction();
 *
 *        int numPoints = drawer.point.getNum();
 *        if ( numPoints > 0 )
 *        {
 *          // Use points
 *          SbVec2f point0 = drawer.point.getValueAt(0);
 *        }
 *        drawer.clear(); // Remove polyline from screen
 *        action.setHandled();
 *      }
 *    }
 *    \endcode
 *  \endif
 *
 * @SEE_ALSO
 *    SoEllipseScreenDrawer,
 *    SoLassoScreenDrawer,
 *    SoPolygonScreenDrawer,
 *    SoRectangleScreenDrawer,
 *    SoScreenDrawer
 */
class INVENTOR_API SoPolyLineScreenDrawer : public SoScreenDrawer 
{
  SO_NODE_HEADER( SoPolyLineScreenDrawer );

public:
  /** Constructor */
  SoPolyLineScreenDrawer();

  /**
   * Threshold (in pixels) used to simplify line when it is finished. 
   * A value of 0 means no simplification. Default value is 5 pixels. 
   */
  SoSFUInt32 simplificationThreshold;

  /** 
   * Make the line counter-clockwise when it is finalized.
   * This can change order of points. Default is TRUE.
   * @note Some shapes (like "8") do not have a defined clockwiseness.
   * In these cases, the result is undefined.
   *
   * @FIELD_SINCE_OIV 9.0
   */
  SoSFBool doCCW;

  /** Color of line. Default value is red. */
  SoSFColor color;

  /** Points of line. Default is empty. */
  SoMFVec2f point;

  /** 
   * Close the line during display (connect last point to first point).
   * Default is FALSE (but most sub-classes automatically set it to TRUE).
   * @warning This only affects the display.
   */
  SoSFBool isClosed;

  /**
   * Stipple pattern.
   * This specifies how dashed or dotted lines will be drawn.
   * The pattern is a 16-bit series of 0s and 1s and is repeated
   * as necessary to stipple a given line. A 1 indicates that
   * drawing occurs, and a 0 that it does not, on a
   * pixel-by-pixel basis, starting with the low-order bits of the pattern.
   * Values can range from 0 (invisible) to 0xffff (solid). Default is 0xffff.
   *
   * The line pattern can be stretched using the #linePatternScaleFactor field.
   */
  SoSFUShort linePattern;

  /**
   * Stipple pattern scale factor. It stretches the line pattern
   * (see #linePattern) by multiplying each subseries of consecutive 1s and 0s.
   * Scale factors are clamped to lie between 1 and 255. Default is 1.
   */
  SoSFInt32 linePatternScaleFactor;

  /**
   * Width of lines.
   * The default line width is 0, meaning to use the default OpenGL value (1).
   * Line widths greater than zero are considered to be specified in printer's points,
   * where 1 inch = 72 printer's points. However the default pixels-per-inch value (managed
   * by SbViewportRegion) is also 72. So effectively line width is specified in pixels
   * unless the application sets a different pixels-per-inch value.
   */
  SoSFFloat lineWidth;

  /**
   * Structure given to callback when an event is raised.
   * Please refer to #onStart, #onMove, ... events.
   * [OIV-WRAPPER-CLASS AUTO_PROPERTY]
   */
  struct EventArg : public SbEventArg
  {
    /** Default constructor */
    EventArg( SoHandleEventAction* action, SoPolyLineScreenDrawer* drawer )
    : m_action( action ), m_drawer( drawer )
    {}

    /** Destructor */
    ~EventArg() { }

    /** Returns the handle event action related to the drawing. */
    SoHandleEventAction* getAction() const { return m_action; }

    /** Returns the drawer. */
    SoPolyLineScreenDrawer* getSource() const { return m_drawer; }

  private:
    /** Handle event action related to the drawing. */
    SoHandleEventAction* m_action;

    /** Screen drawer related to the drawing. */
    SoPolyLineScreenDrawer* m_drawer;
  };

  /**
   * Event raised when starting to draw a polyline.
   * See SbEventHandler for methods to set a callback to be notified when this
   * event is raised.  The callback will be called with an EventArg from which
   * it can query the current action and drawer objects. @BR
   * Note that the callback parameter must be declared @I const @i.
   */
  SbEventHandler<const EventArg&> onStart;

  /**
   * Event raised during polyline drawing.
   * See SbEventHandler for methods to set a callback to be notified when this
   * event is raised.  The callback will be called with an EventArg from which
   * it can query the current action and drawer objects.
   * Note that the callback parameter must be declared @I const @i.
   */
  SbEventHandler<const EventArg&> onMove;

  /**
   * Event raised when the line is finished.
   * See SbEventHandler for methods to set a callback to be notified when this
   * event is raised.  The callback will be called with a line drawer specific
   * EventArg from which you can query the current action and drawer objects.
   * Note that the callback parameter in this case is @I not @i const.
   */
  SbEventHandler<EventArg&> onFinish;

  /** Convenience method to clear the points in the line. */
  virtual void clear() { point.deleteValues(0); }

  /**
   * Clears the points in the line and resets internal state to initial values.
   * This cancels any current drawing.
   *
   * @M_SINCE 10.7.3
   */
  virtual void reset();

SoINTERNAL public:

  /** Register in database */
  static void initClass();

  /** Unregister from database */
  static void exitClass();

  /** Convenient method to add a point in screen space ([-1, 1]x[-1, 1]) */
  void addPoint( const SbVec2f& point );

  /** 
   *Convenient method to remove the id'th point. if id > point->getNum(), nothing is 
   * done. 
   */
  void removePoint( unsigned id );

  /** 
   *Convenient method to set new value for the id'th point. 
   * If id >= point->getNum(), nothing is done. 
   */
  void setPoint( unsigned id, const SbVec2f& newPoint);

  /** 
   * Finalize creation of line: simplify it and ensure it is CCW.
   * @warning The finalized line is not closed (first point != last point). 
   */
  void finalize( SoHandleEventAction* action );

protected:

  /** Destructor */
  virtual ~SoPolyLineScreenDrawer();

private:

  /** simplify the specified polyline using specified epsilon. */
  static void simplify( std::vector<SbVec2f>& polyline, float epsilon );

   /** Return true if line is CCW. */
  static bool isCCW( const std::vector<SbVec2f>& polyline );

  /** Make specified polyline line in CCW order. */
  static void makeCCW( std::vector<SbVec2f>& polyline );

private:

  /** Root of scenegraph that contains the line. */
  SoSeparator* m_lineRoot;

  // Engine used to convert from MFVec2 to MFVec3. it is used only internally by 
  // this class to draw line in screen space and should not be used by everyone 
  // else, so it is defined in in SoPolyLineScreenDrawer.cxx file.

  /** Engine used to convert from MFVec2 to MFVec3. */
  class MFVec2ToMFVec3;
  /** Engine used to convert from MFVec2 to MFVec3. */
  MFVec2ToMFVec3* m_vec2ToVec3;
};


#endif // _SO_POLY_LINE_SCREEN_DRAWER_H_


