/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Thibaut Andrieu (Jan 2011)
**=======================================================================*/

#if !defined _SO_SCREEN_DRAWER_H_
#define _SO_SCREEN_DRAWER_H_

#include <Inventor/nodes/SoNode.h>
#include <Inventor/SbVec.h>
#include <Inventor/fields/SoSFNode.h>
#include <Inventor/SbBox.h>

class SoHandleEventAction;
class SoOrthographicCamera;
class SoSeparator;
class SbViewportRegion;

/** 
 * @VSGEXT Base class to draw a scene graph in screen space.
 *
 * @ingroup drawers
 * 
 * @DESCRIPTION
 *   This node displays the scene graph specified by the #sceneGraph field in screen space 
 *   normalized in the range ([-1, 1]x[-1, 1]x[-1, 1]) relative to the current viewport.
 *
 *   The specified scene graph is automatically rendered using an SoAnnotation node,
 *   an SoPickStyle node with style = UNPICKABLE, and an SoOrthographicCamera node.
 *
 *   Classes derived from SoScreenDrawer are normally used to implement dynamically
 *   drawing a 2D shape, for example a rectangle, in response to mouse events.
 *   Applications will typically use one of these derived classes, for example,
 *   SoEllipseScreenDrawer, SoRectangleScreenDrawer, SoPolygonScreenDrawer or SoLassoScreenDrawer.
 *
 *   Please see SoPolyLineScreenDrawer for important notes about these classes and a code example.
 *
 *   Applications may also implement their own custom screen drawer classes.
 *   The convenient onMouseDown(), onMouseMove(), ... methods can be 
 *   implemented by sub-classes to customize behaviors for specific events, but the
 *   application can also reimplement the handleEvent() method to handle any event.
 *
 *   There should only be one SoScreenDrawer derived node in the sceneGraph (to avoid conflicts 
 *   in mouse events).
 *
 *   NOTE: This node and its subclasses use screen coordinates ranging from -1 to 1.  (Because
 *   this is the default view volume for an SoOrthographicCamera node.)  Be careful because
 *   other classes in Open Inventor use normalized screen coordinates ranging from 0 to 1.
 *   In particular the SbViewportRegion normalize() method and the SbViewVolume methods that
 *   project from 3D to 2D and vice-versa.
 *
 *   \if_cpp
 *   [C++] @BR Screen drawer classes must be initialized by calling SoInteraction::init(). High level
 *   viewer init methods, like SoWin::init(), automatically do this. However, if the application
 *   is calling SoDB::init() directly, then it is usually necessary to also call SoInteraction::init()
 *   before using the screen drawer classes.
 *   \endif
 *
 * @SEE_ALSO
 *    SoEllipseScreenDrawer
 *    SoLassoScreenDrawer
 *    SoPolygonScreenDrawer
 *    SoPolyLineScreenDrawer
 *    SoRectangleScreenDrawer
 */
class INVENTOR_API SoScreenDrawer : public SoNode
{
  SO_NODE_HEADER( SoScreenDrawer );

public:

  /** Scene graph to render. */
  SoSFNode sceneGraph;

  /** Constructor */
  SoScreenDrawer();

  /**
   * Resets internal state to initial values. This cancels any current drawing.
   *
   * @M_SINCE 10.7.3
   */
  virtual void reset();

SoINTERNAL public:

  /** Register in database */
  static void initClass();

  /** Unregister from database */
  static void exitClass();
  
  /** catch some fields changes */
  virtual void notify( SoNotList *list );

SoEXTENDER public:
  /**
  * Traversal routine for SoHandleEventAction - this will call
  * action->setHandled() if the event is handled
  */
  virtual void handleEvent( SoHandleEventAction* action );

  // Implement actions to redirect to m_root

  /** @copydoc SoNode::doAction */
  virtual void doAction( SoAction *action );

  /** @copydoc SoNode::callback */
  virtual void callback( SoCallbackAction *action );

  /** @copydoc SoNode::GLRender */
  virtual void GLRender( SoGLRenderAction *action );

  /** @copydoc SoNode::rayPick */
  virtual void rayPick( SoRayPickAction *action );

  /** @copydoc SoNode::getBoundingBox */
  virtual void getBoundingBox( SoGetBoundingBoxAction *action );

  /** @copydoc SoNode::getPrimitiveCount */
  virtual void getPrimitiveCount( SoGetPrimitiveCountAction *action );

  /** Returns position normalized in range -1..1 relative to specified viewport. */
  SbVec2f normalizePoint(const SbVec2f& point, const SbViewportRegion& viewport);

SoEXTENDER_Documented protected:

  /** Called on keyDown event. Should be implemented by sub-classes. */
  virtual void onKeyDown( SoHandleEventAction* ) {};

  /** Called on mouseDown event. Should be implemented by sub-classes. */
  virtual void onMouseDown( SoHandleEventAction* ) {};

  /** Called on mouseMove event. Should be implemented by sub-classes. */
  virtual void onMouseMove( SoHandleEventAction* ) {};

  /** Called on double click event. Should be implemented by sub-classes. */
  virtual void onMouseDblClick( SoHandleEventAction* ) {};

  /** Called on mouseDrag event. Should be implemented by sub-classes. */
  virtual void onMouseDragging( SoHandleEventAction* ) {};

  /** Called on mouseUp event. Should be implemented by sub-classes. */
  virtual void onMouseUp( SoHandleEventAction* ) {};

  /** Returns last mouse down position in pixels. */
  inline const SbVec2f& getLastMouseDown() const { return m_lastMouseDown; }

protected:

  /** Destructor */
  virtual ~SoScreenDrawer();

private:

  /** 
   * Root of sceneGraph to render. Contains an orthographic camera and the user 
   * sceneGraph. All actions will be redirected to this node.
   */
  SoSeparator* m_root;

  /** 
   * Node containing input user scene graph. It has at most one child (sceneGraph) and
   * its parent is m_root.
   */
  SoSeparator* m_userGraphContainer;

  /** Orthographic normalized camera (viewProj matrix is identity) */
  SoOrthographicCamera* m_camera;

  /** true if left mouse button is currently down. */
  bool m_mouseLeftDown;

  /** Last mouse down position in pixel. */
  SbVec2f m_lastMouseDown;

};


#endif // _SO_SCREEN_DRAWER_H_


