/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_ACCUMULATED_ELEMENT
#define  _SO_ACCUMULATED_ELEMENT

#ifdef _WIN32
#pragma warning( push )
#pragma warning(disable:4251)
#endif

/**
* 
*   Abstract base class for each state element whose value
*   may be accumulated when it is set
*
*   @ingroup elements
*
*   @DESCRIPTION
*   This is the abstract base class for each state element whose value
*   may be accumulated when it is set. (This is rare.) Examples are
*   transformations and profiles.
* 
*   Subclasses may need to override the push() method to copy values
*   from the next instance in the stack (using getNextInStack() ),
*   if the new one has to accumulate values on top of the old ones.
* 
*   This class defines the matches() method to compare lists of
*   node-id's. The node-id's represent the states of all nodes that
*   changed the value of the element. SoAccumulatedElement provides
*   methods that maintain lists of node-id's of all nodes that affect
*   an instance. Subclasses must call these methods to make sure the
*   id's are up to date, if they plan to use the standard matches()
*   method. Otherwise, they can define matches() differently, if they
*   wish.
*
*   @SEE_ALSO
*   Referenced classes go here.
*
*/

#include <Inventor/elements/SoSubElement.h>
#include <Inventor/STL/vector>

SoEXTENDER_Documented class INVENTOR_API SoAccumulatedElement : public SoElement {

  SO_ELEMENT_ABSTRACT_HEADER(SoAccumulatedElement);

 public:

 /**
  *  Pushes element. Allows for side effects to occur. Default
  *  methods do nothing.
  */
  virtual void push( SoState* );

  /**
   *  Prints element (for debugging).
   */
  virtual void print(FILE *fp) const;

protected:

  /**
  *  Returns TRUE if the element matches another element, based on node-id's.
  */
  virtual SbBool matches(const SoElement* elt) const;

  /**
  * Create and return a copy of this element; this will copy the
  * nodeId list properly.
  */
  virtual SoElement* copyMatchInfo() const;

 SoINTERNAL public:
  // Initializes the SoAccumulatedElement class
  static void         initClass();
  static void         exitClass();

 protected:
  // Clears out the list of node id's
  void clearNodeIds();

  // Adds the id of the given node to the current list
  void addNodeId(const SoNode *node);

  // Replace an existing id by the id of the given node
  void replaceNodeId(const SoNode *nodeToReplace,const SoNode* newNode);

  // Sets the node id list to JUST the id of the given node
  void setNodeId(const SoNode *node);

  // Override normal capture method to capture elements at other
  // depths that we are accumulating with.
  virtual void captureThis(SoState *state) const;

  // Destructor
  virtual ~SoAccumulatedElement();

  // ordered list of nodeID
  // we do not use a set for memory optim reason (reserve not available in set)
  typedef std::vector<uint64_t> SoNodeIdList;
  
  // reset the current nodeIdList with the given one
  void setNodeIdList( const SoNodeIdList& nodeIdList );
  
  // returns the current nodeIdList
  inline const SoNodeIdList& getNodeIdList() const
  { return m_nodeIdList; }

 private:
  /** Add nodeid to the list keeping order */

  // This stores the list of node id's.
  mutable SoNodeIdList m_nodeIdList;
  bool accumulatesWithParentFlag;

};

#ifdef _WIN32
#pragma warning( pop )
#endif

#endif /* _SO_ACCUMULATED_ELEMENT */


