/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#ifndef _SO_BLEND_ELEMENT_H
#define _SO_BLEND_ELEMENT_H

#include <Inventor/SbColorRGBA.h>
#include <Inventor/elements/SoElement.h>
#include <Inventor/elements/SoSubElement.h>
#include <Inventor/SbEnums.h>

/**
 * @VSGEXT Stores the current blending mode and parameters.
 *
 * @ingroup elements
 *
 * @DESCRIPTION
 * This element stores the current blending mode, including source and target factors for RGB and Alpha values, equation and color.
 *
 */
class INVENTOR_API SoBlendElement : public SoElement
{
  SO_ELEMENT_HEADER( SoBlendElement );

public:
  /**
   * BlendFactor as defined by OpenGL's glBlendFunc.
   */
  enum BlendFactor
  {
    ZERO = SbEnums::BLEND_ZERO,
    ONE = SbEnums::BLEND_ONE,
    SRC_COLOR = SbEnums::BLEND_SRC_COLOR,
    ONE_MINUS_SRC_COLOR = SbEnums::BLEND_ONE_MINUS_SRC_COLOR,
    DST_COLOR = SbEnums::BLEND_DST_COLOR,
    ONE_MINUS_DST_COLOR = SbEnums::BLEND_ONE_MINUS_DST_COLOR,
    SRC_ALPHA = SbEnums::BLEND_SRC_ALPHA,
    ONE_MINUS_SRC_ALPHA = SbEnums::BLEND_ONE_MINUS_SRC_ALPHA,
    DST_ALPHA = SbEnums::BLEND_DST_ALPHA,
    ONE_MINUS_DST_ALPHA = SbEnums::BLEND_ONE_MINUS_DST_ALPHA,
    CONSTANT_COLOR = SbEnums::BLEND_CONSTANT_COLOR,
    ONE_MINUS_CONSTANT_COLOR = SbEnums::BLEND_ONE_MINUS_CONSTANT_COLOR,
    CONSTANT_ALPHA = SbEnums::BLEND_CONSTANT_ALPHA,
    ONE_MINUS_CONSTANT_ALPHA = SbEnums::BLEND_ONE_MINUS_CONSTANT_ALPHA,
    SRC_ALPHA_SATURATE = SbEnums::BLEND_SRC_ALPHA_SATURATE,
    SRC1_COLOR = SbEnums::BLEND_SRC1_COLOR,
    ONE_MINUS_SRC1_COLOR = SbEnums::BLEND_ONE_MINUS_SRC1_COLOR,
    SRC1_ALPHA = SbEnums::BLEND_SRC1_ALPHA,
    ONE_MINUS_SRC1_ALPHA = SbEnums::BLEND_ONE_MINUS_SRC1_ALPHA
  };

  /**
   * Blend Equation as defined by OpenGL's glBlendEquation.
   */
  enum BlendEquation
  {
    FUNC_ADD = SbEnums::BLEND_FUNC_ADD,
    FUNC_SUBTRACT = SbEnums::BLEND_FUNC_SUBTRACT,
    FUNC_REVERSE_SUBTRACT = SbEnums::BLEND_FUNC_REVERSE_SUBTRACT,
    MIN = SbEnums::BLEND_FUNC_MIN,
    MAX = SbEnums::BLEND_FUNC_MAX
  };

  /**
   * BlendElementParameters packages all the different blending parameters.
   *
   * By default it has each parameter set at its default value, and provides comparison operators.
   *
   */
  struct INVENTOR_API BlendElementParameters {
    BlendElementParameters()
      : enabled(false)
      , sourceRGBFactor( SoBlendElement::ONE )
      , destinationRGBFactor( SoBlendElement::ZERO )
      , sourceAlphaFactor( SoBlendElement::ONE )
      , destinationAlphaFactor( SoBlendElement::ZERO )
      , equation( SoBlendElement::FUNC_ADD )
      , color( SbColorRGBA( 0, 0, 0, 0 ) )
    {
    }

    bool operator == (const BlendElementParameters& other) const;
    bool operator != (const BlendElementParameters& other) const;

    bool enabled;
    BlendFactor sourceRGBFactor;
    BlendFactor destinationRGBFactor;
    BlendFactor sourceAlphaFactor;
    BlendFactor destinationAlphaFactor;
    BlendEquation equation;
    SbColorRGBA color;
  };

  /**
   * Sets the blending parameters.
   */
  static void set( SoState* state, const BlendElementParameters& parameters );

  /**
   * Enable or disable blending.
   */
  static void setBlending( SoState* state, bool enabled );

  /**
   * Sets the blending function.
   * Convenience function to shorten code only setting the blending function withput separating components.
   */
  static void setFunction( SoState* state, const BlendFactor source, const BlendFactor destination );

  /**
   * Retrieves the current blending parameters.
   */
  static void get( SoState* state, BlendElementParameters& parameter );

  /**
  *  Prints element (for debugging).
  */
  virtual void print( FILE* fp ) const;

  /**
  *  Override push() method to keep GL up to date.
  */
  virtual void push( SoState* state );

  /**
  *  Override pop() method to keep GL up to date.
  */
  virtual void pop( SoState* state, const SoElement* prevTopElement );

protected:
  virtual ~SoBlendElement();

  /** Initializes element. */
  virtual void init( SoState* state );

  /**
   * Sets the blending parameters
   */
  virtual void setElt( const BlendElementParameters& parameter );

  /**
  *  Returns TRUE if blend parameters match.
  */
  virtual SbBool matches( const SoElement* element ) const;

  /**
  *  Create and return a copy of this element.
  */
  virtual SoElement* copyMatchInfo() const;

SoINTERNAL public :

    /** register in database. */
  static void initClass();

  /** unregister from database. */
  static void exitClass();

protected:
  /** current blending parameters **/
  BlendElementParameters m_parameters;
};

#endif // _SO_BLEND_ELEMENT_H
