/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_ELEMENT
#define  _SO_ELEMENT

#include <Inventor/SbPImpl.h>
#include <Inventor/SoType.h>
#include <Inventor/SoTypedObject.h>
#include <Inventor/misc/SoState.h>
#include <Inventor/errors/SoDebugError.h>
#include <assert.h>

class SoNotList;


#ifndef HIDDEN_FROM_DOC
//Max expected key, smaller than full range to optimize use in bitset
#define SO_ELEMENT_MAX_KEY 512
#endif

SO_PIMPL_PUBLIC_DECLARATION(SoGLCacheList)

/**
 *   Abstract base class for all state elements.
 *
 * @ingroup elements
 *
 *   @DESCRIPTION
 *   This is the abstract base class for all state elements. This class
 *   defines the following features for all of its derived classes:
 *
 *       @B Type identifiers: @b Each class of element has a unique (static)
 *       SoType identifier. The correct type id is also stored in each
 *       instance for easy access.
 *
 *       @B Copying: @b Elements are copied into the list of elements used
 *       in an SoCache. Performing any operation other than matches()
 *       on a copied element is not guaranteed to work (and will not
 *       work for things like Coordinates, if the coordinate they were
 *       pointing to has been deleted).
 *
 *       @B Stack indices: @b Since a subclass of an element class needs to
 *       be in the same state stack as the class from which it is
 *       derived, stack indices are inherited.
 *
 *       @B Capturing: @b Each time an element's value is accessed, that
 *       element is added to the elements-used list of all currently
 *       open caches.
 *
 *   @SEE_ALSO
 *
 * [OIV-WRAPPER-CLASS ALL_DERIVABLE]
 */

SoEXTENDER_Documented class INVENTOR_API SoElement: public SoTypedObject {

 public:

  /**
  *  Pushes element. Allows for side effects to occur. Default
  *  methods do nothing.
  */
  virtual void push(SoState *state);

  /**
  *  Pops element. Allows for side effects to occur. Default
  *  methods do nothing.
  */
  virtual void pop(SoState *state, const SoElement *prevTopElement);

  /**
  *  Prints element (for debugging).
  */
  virtual void print(FILE *fp) const;

  /**
  *  Returns type identifier for SoElement class.
  */
  static SoType getClassTypeId();

  /**
  *  Returns type identifier for element instance.
  */
  virtual SoType getTypeId() const       { return typeId; }

protected:

  /**
  *  Common initialization. Directly called by the constructor.
  *  Default method does nothing.
  * [OIV-WRAPPER NO_WRAP]
  */
  virtual void commonInit();

  /**
  *  Initializes element. Called for first element of its kind in
  *  stack. Default method does nothing.
  */
  virtual void init(SoState *state);

  /**
  *  Returns TRUE if the element matches another element (of the
  *  same class, presumably) with respect to cache validity.  If you
  *  write a matches() method, you must also write a copy() method.
  */
  virtual SbBool matches(const SoElement* elt) const = 0;

  /**
  *  Create a copy that we can put in a cache used list and call
  *  matches() on later.
  */
  virtual SoElement* copyMatchInfo() const = 0;

  /**
   * Returns next instance in specific element stack
   */
  SoElement* getNextInStack() const { return nextInStack; }

 SoINTERNAL public:
  typedef uint16_t SoElementKeyType;
  // Create a new key
  static SoElementKeyType createElementKey();

  /**
  *  Returns key identifier for element instance.
  */
  virtual SoElementKeyType getElementKey() const
  {
    return badElementKey();
  }

  /**
   * Returns an always-illegal Key. Useful for returning errors.
   */
  static SoElementKeyType badElementKey();

  // Initialize ALL Inventor element classes
  static void initElements();
  static void exitElements();

  /**  Return true if setter and getter of this element needs to be recorded by SoCacheElement */
  inline bool isTrackingSetterGetterNodes() const
  {
    return m_trackSetterGetterNodes;
  }
  
  virtual int getClassStackIndexInternal() const = 0;

  // Initializes the SoElement class
  static void initClass();
  static void exitClass();

  // get max element key value
  static int maxElementKey() { return s_nextElementKey; }

  // Returns the number of stack indices allocated
  static int getNumStackIndices();

  // Returns the id for the element with the given stack index
  static SoType getIdFromStackIndex(int stackIndex);

  // Returns the stack index for an element instance
  inline int getStackIndex() const   { return stackIndex; }
  
  // Sets stackIndex in instance
  void setStackIndex(int index)        { stackIndex = index; }

  // Sets stuff in an element instance
  void setDepth(int dpth)              { depth = dpth; }
  void setNext(SoElement *nxt)         { next = nxt; }
  void setNextInStack(SoElement *nxt)  { nextInStack = nxt; }
  void setNextFree(SoElement *nxt)     { nextFree = nxt; }

  // Returns stuff from element instance
  int getDepth() const               { return depth; }
  SoElement* getNext()  const        { return next; }

  static const SoElement* getStackElement(SoState *state, const SoType& type, int stackIndex);

  /**
   * Returns a read-only pointer to the top instance in the given element stack.
   * Note: visibility is internal for wrapping purpose
   *
   * [OIVJAVA-WRAPPER VISIBILITY{Protected}]
   * [OIVNET-WRAPPER VISIBILITY{Internal}]
   */
  static const SoElement* getConstElement(SoState *state, const SoType& type, int stackIndex);

  /**
   * Returns a read-only pointer to the top Element's type instance in the current state.
   * [OIVJAVA-WRAPPER CUSTOM_CODE]
   */
  template<typename T>
  static const T* getConstElement(SoState *state)
  {
    return static_cast<const T*>(getConstElement(state, T::getClassTypeId(), T::getClassStackIndex()));
  }

  // Sets/Gets the pop priority.
  // Higher priority will be popped first.
  void setPopPriority(unsigned short priority) {m_popPriority = priority ;}
  unsigned short getPopPriority() const { return m_popPriority ; }

  // Destructor
  virtual ~SoElement();

  // This is called when a referenced SoBase (path, field, node, whatever) is
  // deleted. All subclasses must implement this to do the right
  // thing in case of reference to any SoBase object.
  virtual void dyingReference() {} ;

  /**
  *  Returns type identifier for element instance.
  *  It returns the same info than getTypeId() but without virtual
  *  call performance penalty. Internal usage since 9.3
  */
  inline SoType getInstanceTypeId() const
  {
    // if this assert is triggered it means that SO_SUB_ELEMENT macro
    // are not use/defined correctly: setTypeId(classTypeId) has not
    // been called in element constructor (see constructor code above)
    assert(typeId!=SoType::badType());
    return typeId;
  }

#if SoDEPRECATED_BEGIN(9000)

  /**
   * Returns a read-only pointer to the top instance in the given
   * element stack
   */
  SoDEPRECATED_METHOD(9000,"Use the type safe templated version.")
  static const SoElement* getConstElement(SoState *state, int stackIndex);

  SoDEPRECATED_METHOD(9000,"Use the type version.")
  static const SoElement* getStackElement(SoState *state, int stackIndex);

#endif /** @DEPRECATED_END */

 protected:
  /** Constructor. use typeId.createInstance to create elements. */
  SoElement();

  /**
   * Returns an instance of an element from the stack. This instance is writeable.
   * This returns NULL if no writable instance can be returned.
   * [OIV-WRAPPER CUSTOM_CODE]
   */
  template<typename T>
  static T* getElement(SoState *state)
  {
    return static_cast<T*>(getElement(state, T::getClassTypeId(), T::getClassStackIndex()));
  }

  /**
   * Returns an instance of an element from the stack. This instance is writeable.
   * This returns NULL if no writable instance can be returned.
   */
  static SoElement* getElement(SoState *state, const SoType& type, int stackIndex);

  // Does whatever is necessary in state to capture this element for
  // caching purposes. Should be called by subclasses whenever
  // any value in the element is accessed.
  void capture(SoState *state) const
  {
    if ( state->isCacheOpen() )
      captureThis(state);
  }

  // Really captures this element, once it has been determined that
  // a cache is open to capture it
  virtual void captureThis(SoState *state) const;

  // Sets typeId in instance
  void setTypeId(SoType id) { typeId = id; }


  // Returns next free element in a specific element stack
  SoElement* getNextFree() const { return nextFree; }

  inline void setPopped(SbBool flag) { m_popped = (flag == TRUE); }
  inline SbBool isPopped() const { return m_popped ; }

  /**
   * [OIVJAVA-WRAPPER CUSTOM_CODE]
   * [OIVNET-WRAPPER OBSOLETE]
   */
  static int getClassStackIndex();

  /**
   * Returns an instance of an element from the stack with the given
   * index in the given state. This instance is writeable. This
   * returns NULL if no writable instance can be returned.
   */
#if SoDEPRECATED_BEGIN(9000)

  SoDEPRECATED_METHOD(9000,"Use the type safe templated version.")
  static SoElement* getElement(SoState *state, int stackIndex);

#endif /** @DEPRECATED_END */

 SoINTERNAL protected:
  // Creates and returns a new stack index
  // used internally whille registering an element in the database
  static int createStackIndex(const SoType& id);

  /** If false, SoCacheElement won't record setter and getter of this element (Default true) */
  inline void trackSetterGetterNodes(bool flag)
  {
    m_trackSetterGetterNodes = flag;
  }

 private:
  /**
   * Insert an element at the given depth in the element list
   * Return the new first element
   */
  SoElement* insertAtDepth(SoElement* elt, int depth);

  SoElement* replaceElement(SoState* state, SoElement* oldElt, SoElement* newElt);

  void updateNextInStackAndFree(SoElement* oldElt, SoElement* newElt);

  // nextInStack and nextFree are the pointers in a doubly-linked
  // list for each stack in the state.  nextInStack points toward
  // the bottom of the stack and nextFree points toward the top of
  // the stack.  Once a stack has been fully allocated, pushing
  // becomes simply following the nextFree pointer and popping
  // becomes following the nextInStack pointer.

  SoElement           *nextInStack;   // Next element of same type on stack
  SoElement           *nextFree;      // Next free element
  SoElement           *next;          // Next element in all-element

  static int         classStackIndex;  // Stack index for SoElement class

  // Stack index stuff
  static int         nextStackIndex;  // Next stack index to allocate

  static SoTypeList  *stackToType;    // Correlates stack index to type id
  int                 stackIndex;     // Stack index for instance
  int                 depth;          // Stack depth of element

  // Type identifier stuff
  static SoType      classTypeId;     // Type identifier for SoElement class
  SoType              typeId;         // Type identifier for instance

  // Other per-instance stuff
  // stack
  unsigned short    m_popPriority ; // Stores the pop priority.
  bool              m_popped ;      // Indicates if the element has been popped.

  /**
   * True if setter and getter of this element needs to be recorded by SoCacheElement 
   * Default is true. (eg: False for elements inheriting SoCacheElement)
   */
  bool m_trackSetterGetterNodes;

  friend class SoState;
  friend class SoCache;
  friend class SoBoundingBoxCache;
  friend class inventor::impl::SoGLCacheListImpl;
  friend class SoGLViewingMatrixElement;

  //Insure increment of key while initing element class
  static SoElementKeyType s_nextElementKey;
};

#endif /* _SO_ELEMENT */


