/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_MODEL_MATRIX_ELEMENT
#define  _SO_MODEL_MATRIX_ELEMENT

#include <Inventor/SbLinear.h>
#include <Inventor/elements/SoAccumulatedElement.h>

/**
*   Stores the current model matrix.
*
* @ingroup elements
*
*   @DESCRIPTION
*   This element stores the current model matrix - the cumulative
*   transformation applied to subsequent shapes. Because the matrix is
*   cumulative, this class is derived from SoAccumulatedElement. The
*   set() method replaces the current matrix, while all the others
*   (mult(), translateBy(), etc.) multiply into it. Node id's of the
*   nodes that affect the element are accumulated properly.
*
*   This element also stores the current view-volume culling
*   transformation, which is normally the view*projection matrices
*   (set by cameras), but which may be modified by sophisticated
*   culling schemes.
*
*   @SEE_ALSO
*   SoMatrixTransform, SoRotation, SoRotationXYZ, SoScale,
*   SoTransform, SoTranslation, SoUnits
*/

SoEXTENDER_Documented class INVENTOR_API SoModelMatrixElement : public SoAccumulatedElement {

  SO_ELEMENT_HEADER(SoModelMatrixElement);

 public:

  /**
  *  Overrides push() method to copy values from next instance in the stack.
  */
  virtual void        push(SoState *state);

  /**
  *  Sets the model matrix to the identity matrix.
  */
  static void         makeIdentity(SoState *state, SoNode *node);

  /**
  *  Sets the model matrix to the given matrix.
  *  This method will (indirectly) call glLoadMatrix with the
  *  new matrix unless the 'sendToGL' parameter is explicitly FALSE.
  */
  static void         set(SoState *state, SoNode *node,
                          const SbMatrix &matrix, SbBool sendToGL = TRUE);

  /**
  *  Multiplies the given matrix into the model matrix.
  */
  static void         mult(SoState *state, SoNode *node,
                           const SbMatrix &matrix);

  /**
  *  This multiplies a matrix that performs the specified
  *  transformation into the model matrix.
  */
  static void         translateBy(SoState *state, SoNode *node,
                                  const SbVec3f &translation);
  /**
  *  This multiplies a matrix that performs the specified
  *  transformation into the model matrix.
  */
  static void         rotateBy(SoState *state, SoNode *node,
                               const SbRotation &rotation);
  /**
  *  This multiplies a matrix that performs the specified
  *  transformation into the model matrix.
  */
  static void         scaleBy(SoState *state, SoNode *node,
                              const SbVec3f &scaleFactor);

  /**
  *  This method is used by the TransformSeparator node.
  *  WARNING!  For proper caching behavior, the matrix returned by
  *  ::pushMatrix() must be used ONLY as a later argument to
  *  ::popMatrix(), and you must NOT modify the matrix between the
  *  push and the pop.  If you need the matrix for any other reason,
  *  you MUST use the ::get() routine.
  */
  static SbMatrix     pushMatrix(SoState *state);
  /**
  *  This method is used by the TransformSeparator node.
  *  WARNING!  For proper caching behavior, the matrix returned by
  *  ::pushMatrix() must be used ONLY as a later argument to
  *  ::popMatrix(), and you must NOT modify the matrix between the
  *  push and the pop.  If you need the matrix for any other reason,
  *  you MUST use the ::get() routine.
  */
  static void         popMatrix(SoState *state, const SbMatrix &m);

  /**
  *  Sets the transformation that defines the volume that
  *  view-volume culling should be tested against.
  */
  static void         setCullMatrix(SoState *state, SoNode *node,
                                    const SbMatrix &matrix);

  /**
  *  This method gets the model*cullMatrix combined matrix (which is
  *  cached by this element).
  */
  static const SbMatrix &     getCombinedCullMatrix(SoState *state);

  /**
  *  Returns current model matrix from the state.
  */
  static const SbMatrix &     get(SoState *state);

  /**
  *  Returns current model matrix from the state, sets given flag to TRUE
  *  if matrix is known to be identity.
  */
  static const SbMatrix &     get(SoState *state, SbBool &isIdent);

  /**
  *  Prints element (for debugging).
  */
  virtual void        print(FILE *fp) const;

protected:

  /** Initializes element */
  virtual void init(SoState *state);

  /**
  *  Override standard matches to invalidate on cull-test.
  */
  virtual SbBool matches(const SoElement* elt) const;

  /** Copy method, copies flags */
  virtual SoElement* copyMatchInfo() const;

 SoINTERNAL public:
  // Initializes the SoModelMatrixElement class
  static void         initClass();
  static void         exitClass();
  const SbMatrix&     getModelMatrix() const { return modelMatrix; }
  SbBool              isIdent() const { return flags.isModelIdentity; }

  /** see @setCullMatrix */
  static const SbMatrix& getCullMatrix(SoState* state);

  /** see @setCullMatrix */
  const SbMatrix& getCullMatrix();

  /**
   * Similar to pushMatrix public fonction but adding forward
   * traversal model matrix in addition to model matrix.
   * [OIV-WRAPPER NO_WRAP]
   */
  static void pushForwardMatrix(SoState *state, SbMatrix& model, SbMatrix& ftmodel);

  /**
   * Similar to popMatrix public fonction but adding forward
   * traversal model matrix in addition to model matrix.
   * [OIV-WRAPPER NO_WRAP]
   */
  static void popForwardMatrix(SoState *state, const SbMatrix &m, const SbMatrix& f);

  /**
   * Returns current forward traversal model matrix from the state.
   * Note that this matrix is relevant only if the action traverses
   * nodes inside a forward traversal.
   * (i.e., action->isForwardTraversing() returns true).
   */
  static const SbMatrix& getForwardTraversalMatrix(SoState *state);

  /**
  * Returns last saved model matrix at forward traversal entrance.
  * (i.e. the model matrix updated until first custom node traversal)
  */
  static const SbMatrix& getPreForwardTraversalMatrix(SoState *state);

  /**
   *  Sets the forward traversal transformation to identity.
   */
  static void resetForwardTraversalMatrix(SoState *state);

 protected:
  // Sets the matrix in an instance to identity
  virtual void        makeEltIdentity();

  // Sets the matrix in an instance to given matrix
  virtual void        setElt(const SbMatrix &matrix, SbBool sendToGL = TRUE);

  // Multiplies into the matrix in an instance
  virtual void        multElt(const SbMatrix &matrix);

  // Each of these performs the appropriate operation on the matrix
  // in an instance
  virtual void        translateEltBy(const SbVec3f &translation);
  virtual void        rotateEltBy(const SbRotation &translation);
  virtual void        scaleEltBy(const SbVec3f &scaleFactor);

  // virtual methods for push/pop so GL element can Do The Right Thing:
  virtual SbMatrix    pushMatrixElt();
  virtual void        popMatrixElt(const SbMatrix &m);

  // same as previous but taking into account forward traversal transformation
  /** [OIV-WRAPPER NO_WRAP] */
  virtual void        pushForwardMatrixElt(SbMatrix& m, SbMatrix& f);
  /** [OIV-WRAPPER NO_WRAP] */
  virtual void        popForwardMatrixElt(const SbMatrix &m, const SbMatrix& f);

  virtual ~SoModelMatrixElement();

  SbMatrix            modelMatrix;
 private:
  SbMatrix            cullMatrix;
  SbMatrix            modelCullMatrix;
  SbMatrix            forwardTraversalMatrix; // transformation inside a foward traversal
  SbMatrix            preForwardTraversalMatrix;     // transformation until first foward traversal
  struct Flags {
    unsigned int    isModelIdentity : 1;
    unsigned int    haveCullMatrix  : 1; // TRUE if cullMatrix set
    unsigned int    haveModelCull   : 1; // TRUE if model*cull computed
  };
  Flags flags;

  friend class SoLocalBBoxMatrixElement;
};

#endif /* _SO_MODEL_MATRIX_ELEMENT */


