/*=================================================================================
*** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        ***
***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                 ***
***                                                                            ***
***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS   ***
***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR   ***
***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                       ***
***                                                                            ***
***                        RESTRICTED RIGHTS LEGEND                            ***
***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS  ***
***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN  ***
***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT  ***
***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN  ***
***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         ***
***                                                                            ***
***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                    ***
***                              BORDEAUX, FRANCE                              ***
***                            ALL RIGHTS RESERVED                             ***
=================================================================================*/

#pragma once

#include <Inventor/SbLinear.h>
#include <Inventor/elements/SoReplacedElement.h>
#include <Inventor/errors/SoDebugError.h>

/**
 * Allows read-only access to the top element in the state.
 *
 * @ingroup elements
 *
 * @DESCRIPTION
 * This class allows read-only access to the top element in the state
 * to make accessing several values in it more efficient. Individual
 * values must be accessed through this instance.
 *
 * @SEE_ALSO
 * SoTangent, SoVertexProperty
 */
SoEXTENDER_Documented class INVENTOR_API SoTangentElement : public SoReplacedElement
{
  SO_ELEMENT_HEADER( SoTangentElement );

public:
  /**
   * Sets the current tangents.
   * [OIV-WRAPPER-ARG IN,IN,NO_WRAP{(tangents != NULL? tangents->Length: 0)},ARRAY]
   */
  static void set( SoState* state, SoNode* node, int32_t numTangents, const SbVec3f* tangents );

  /**
   * Returns the top (current) instance of the element in the state.
   */
  static const SoTangentElement* getInstance( SoState* state )
  {
    return getConstElement<SoTangentElement>( state );
  }

  /**
   * Returns the number of tangent vectors in an instance.
   */
  int32_t getNum() const
  {
    return m_numTangents;
  }

  /**
   * Returns the indexed tangent from an element.
   */
  const SbVec3f& get( int index ) const
  {
    if ( index < 0 || index >= m_numTangents )
    {
#if defined(_DEBUG)
      SoDebugError::post( "SoTangentElement::get",
                          "Index (%d) is out of range 0 - %d",
                          index, m_numTangents - 1 );
#endif
      return s_defaultTangent;
    }
    return m_tangents[index];
  }

  /**
   * Prints element (for debugging).
   */
  virtual void print( FILE* fp ) const;

protected:
  /** Initializes element */
  virtual void init( SoState* state );

SoINTERNAL public:
  // Initializes the SoTangentElement class
  static void initClass();
  static void exitClass();

protected:
  int32_t m_numTangents;
  const SbVec3f* m_tangents;

  virtual ~SoTangentElement();

private:
  /**
   * This stores a pointer to the default tangent so that we can set
   * "tangents" to point to it if no other tangent has been set.
   */
  static SbVec3f s_defaultTangent;
};
