/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_DEBUG_ERROR
#define  _SO_DEBUG_ERROR

#include <Inventor/errors/SoError.h>
#include <cstdarg>

#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4251)
#endif

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoDebugError
//
//  This class is used for all errors reported from the debugging
//  version of the Inventor library. These errors are typically
//  programmer errors, such as passing a NULL pointer or an
//  out-of-range index. The post() method takes the name of the
//  Inventor method that detected the error, to aid the programmer in
//  debugging.
//
//////////////////////////////////////////////////////////////////////////////

/**
* Debug error handling.
*
* @ingroup errors
*
* @DESCRIPTION
*   SoDebugError is used for all errors reported from Open Inventor library.
*   These errors are typically programmer errors, such as passing a null
*   reference or an out-of-range index. The post() method takes
*   the name of the Open Inventor method that detected the error, to aid the
*   programmer in debugging.
*
*   See SoError for full discussion of error handling in Open Inventor.
*
* @SEE_ALSO
*    SoMemoryError,
*    SoReadError,
*    SoGLError
*/
class INVENTORBASE_API SoDebugError : public SoError {

 public:
   /** Error severity */
  enum Severity {
    /** Error */
    SERROR=0,     // Error
#if !defined(_WIN32) 
    /**
     * @deprecated Use SERROR instead.
     *  Error
     * @DEPRECATED_SINCE_OIV 9.3
     */
    ERROR=SERROR,
#endif
    /**
     *  Just a warning
     */
    WARNING=1,
    /**
     *  No error, just information
     */
    INFO=2
  };

  /**
   * Sets handler callback for SoDebugError class.
   *
   * @B Note:@b the handler callback is reset to the default handler
   * callback by method SoWin::finish.
   */
  static void setHandlerCallback(SoErrorCB *cb, void *data);
  /**
   * Returns handler callback for SoDebugError class.
   * [OIV-WRAPPER-NO-WRAP]
   */
  static SoErrorCB *getHandlerCallback();
  /**
   * Returns handler data for SoDebugError class.
   * [OIV-WRAPPER-NO-WRAP]
   */
  static void *getHandlerData();

  /**
   * Returns type identifier for SoDebugError class.
   */
  static SoType getClassTypeId();

  /**
   * Returns type identifier for error instance
   */
  virtual SoType getTypeId() const;

  /**
   * Returns severity of error (for use by handlers).
   */
  SoDebugError::Severity getSeverity() const { return severity; }

  /**
   * Posts an error
   */
  SoPRINTFPREFIX(2,3) static void post(const char *methodName, const char *formatString ...);

  /**
   * Posts a warning
   */
  SoPRINTFPREFIX(2,3) static void postWarning(const char *methodName, const char *formatString ...);

  /**
   * Posts an informational (non-error) message
   */
  SoPRINTFPREFIX(2,3) static void postInfo(const char *methodName, const char *formatString ...);

 SoINTERNAL public:
  // Initializes SoDebugError class
  static void initClass();
  static void exitClass();

  static SoErrorCB* getDefaultHandlerCB() { return defaultHandlerCB; }

  /** 
   * Posts an error but only once during the lifetime of the application 
   * @M_SINCE 9.3.1
   */
  SoPRINTFPREFIX(2,3) static void postOnce(const char *methodName, const char *formatString ...);

  /** 
   * Posts a warning but only once during the lifetime of the application 
   * @M_SINCE 9.3.1
   */
  SoPRINTFPREFIX(2,3) static void postWarningOnce(const char *methodName, const char *formatString ...);

  /** 
   * Posts an info but only once during the lifetime of the application 
   * @M_SINCE 9.3.1
   */
  SoPRINTFPREFIX(2,3) static void postInfoOnce(const char *methodName, const char *formatString ...);

 protected:
  // Returns handler callback (and data) to use for a given instance
  virtual SoErrorCB *getHandler(void *&data) const;
  virtual bool isInit() const;

 private:
  /** 
   * Return the SbString result of the formatting of formatString with the given argList.
   */
  static SbString getFormattedString(const char *formatString, va_list argList);

  /** Posts a message of type message error but only once during the lifetime of the application  
   *  if alwaysEmit is set to false.
   */
  static void postMessage(Severity messageType, const char *methodName,
                          const char *formatString, va_list argList, bool alwaysEmit);

  /** When calling postOnce, message errors are added to this set */
  static std::set<SbString> s_emittedErrors;

  static  SoType classTypeId; // Type id of SoDebugError class
  static  SoErrorCB *handlerCB; // Handler callback for class
  static  void *cbData; // User data for callback
  static SbBool wasInitted; // TRUE if handler initialized

  Severity severity;    // Severity of error

};

#ifdef _WIN32
#pragma warning(pop)
#endif

#endif /* _SO_DEBUG_ERROR */


