/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_MEMORY_ERROR
#define  _SO_MEMORY_ERROR

#include <Inventor/errors/SoError.h>

class SbThreadMutex;
class SoMemoryError;

/**
 * Memory cleanup handling callback.
 *
 * @memberof SoMemoryError
 *
 * SoMemoryCleanupCB is called when Open Inventor is not able to allocate at 
 * least @I requestedSize @i bytes of memory.
 * This notifies the application that available memory is too low. The application
 * can try to release some memory to allow the allocation to succeed.
 * If the callback returns true then Open Inventor will retry its allocation call.
 *
 * @SEE_ALSO
 *    SoMemoryError::setMemoryCleanupCallback
 *
 * [OIV-WRAPPER NAME{MemoryCleanupCallback}]
 */
typedef bool SoMemoryCleanupCB(const size_t requestedByteSize, void* data);

/**
 * Memory error handling.
 * 
 * @ingroup errors
 * 
 * @DESCRIPTION
 *   SoMemoryError is used for errors reported due to lack of memory.
 *   See SoError for full discussion of error handling in Open Inventor.
 *
 *   The post() method takes a string describing the item that was to be
 *   allocated, such as "cylinder vertex array".
 *
 *   setMemoryCleanupCallback() can be used to set a handler function
 *   that will be called when a memory allocation fails.  The function
 *   can release some memory to allow the allocation to succeed. 
 * 
 * @SEE_ALSO
 *    SoDebugError, SoReadError, SoGLError
 * 
 */
class INVENTORBASE_API SoMemoryError : public SoError {

 public:

  /**
   * Sets handler callback for SoMemoryError class.
   *
   * @B Note:@b the handler callback is reset to the default handler
   * callback by method SoWin::finish.
   */
  static void setHandlerCallback(SoErrorCB *cb, void *data);

  /**
   * Returns handler callback for SoMemoryError class.
   * [OIV-WRAPPER-NO-WRAP]
   */
  static SoErrorCB* getHandlerCallback();

  /**
   * Returns handler data for SoMemoryError class.
   * [OIV-WRAPPER-NO-WRAP]
   */
  static void* getHandlerData();

  /**
   * Sets memory cleanup handler callback for SoMemoryError class.
   * [OIV-WRAPPER NAME{setMemoryCleanupHandler}]
   */
  static void setMemoryCleanupCallback(SoMemoryCleanupCB *cb, void * data);

  /**
   * Returns type identifier for SoMemoryError class.
   */
  static SoType getClassTypeId();

  /**
   * Returns type identifier for an instance of this class.
   */
  virtual SoType getTypeId() const;

  /**
   * Posts an error
   */
  static void post(const SbString& whatWasAllocated);

  /**
   * Posts an error
   */
  static void post(const char* whatWasAllocated);

 SoINTERNAL public:
  // Initializes SoMemoryError class
  static void initClass();
  static void exitClass();
  static SoErrorCB* getDefaultHandlerCB() { return defaultHandlerCB; }
  static bool getMemorySpace(const size_t size);

 protected:
  // Returns handler callback (and data) to use for a given instance
  virtual SoErrorCB * getHandler(void *&data) const;
  virtual bool isInit() const;

 private:
  // Type id of SoMemoryError class
  static SoType classTypeId;
  // Handler callback for class
  static SoErrorCB *handlerCB;
  // User data for callback
  static void *cbData;
  // TRUE if handler initialized
  static SbBool wasInitted;

  static SoMemoryError* s_memoryErrorObject;
  static SbThreadMutex* s_threadMutex;
  static SbString* s_memoryErrorString;

  // This allow to avoid infinite loop in this code
  // when user set an error callback that need allocation itself.
  static bool s_memoryFreeSpaceLoop;
  static SoMemoryCleanupCB* s_memoryCleanupCB;
  static void* s_memoryCleanupData;

};

#endif /* _SO_MEMORY_ERROR */

