/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : David Mott (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2018 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/

#ifndef  _SO_EVENT_
#define  _SO_EVENT_

#include <Inventor/SbBasic.h>
#include <Inventor/SbViewportRegion.h>
#include <Inventor/SbLinear.h>
#include <Inventor/SbTime.h>
#include <Inventor/SoTypedObject.h>

class SbTrackerInfo;

/**
* Base class for all events.
*
* @ingroup events
*
* @DESCRIPTION
*   SoEvent is the base class for events in the Open Inventor event model. An event
*   typically represents a user action, such as a mouse button being pressed or a
*   keyboard key being released. SoEvent contains general information found in all
*   Open Inventor events, including the time the event occurred, the position of the
*   locater when the event occurred, and the state of the modifier keys when the
*   event occurred.
*
*   The virtual method SoEvent::getTrackerInfo() can be used to determine if an
*   event object has associated 3D tracker information. Traditional 2D events like
*   SoMouseButtonEvent and SoLocation2Event return NULL. 3D events like
*   SoTrackerEvent and SoControllerButtonEvent return their associated
*   tracker information (SbTrackerInfo).
*
*   Add an SoEventCallback node to the scene graph to handle Open Inventor events.
*
*   Events are delivered to the scene graph using an SoHandleEventAction.
*
*   The Open Inventor viewer classes automatically convert system events to Open
*   Inventor event objects and send those event objects to the scene graph using
*   an instance of SoSceneManager. SoSceneManager automatically creates and 
*   applies an SoHandleEventAction.
*
*   Similarly, the RemoteViz RenderArea class converts events coming from the
*   RemoteViz client to Open Inventor event objects and sends those events to the
*   scene graph using an instance of SoSceneManager.
*
*   Applications may create synthetic events and send them to the scene graph.
*   Get the SoSceneManager object from the render area and call processEvent().
*
* @SEE_ALSO
*    SoButtonEvent,
*    SoControllerButtonEvent,
*    SoKeyboardEvent,
*    SoLocation2Event,
*    SoMotion3Event,
*    SoMouseButtonEvent,
*    SoSpaceballButtonEvent,
*    SoTrackerEvent,
*    SoHandleEventAction,
*    SoEventCallback,
*    SoSelection,
*    SoInteraction,
*    SoWinDevice,
*    SoWinRenderArea,
*    SoTouchEvent,
*    SoGestureEvent
*
*
*/
class INVENTOR_API SoEvent: public SoTypedObject {
 public:

  /**
   * Constructor.
   */
  SoEvent();
  /**
   * Destructor.
   */
  virtual ~SoEvent();

  /**
   * Returns the type id for this event instance.
   */
  virtual SoType      getTypeId() const;

  /**
   * Returns the type id for the SoEvent class.
   */
  static SoType       getClassTypeId();

  /**
   * Sets the time at which the event occurred.
   */
  virtual void        setTime(SbTime t)               { timestamp = t; }
  /**
   * Gets the time at which the event occurred.
   */
  SbTime              getTime() const                 { return timestamp; }

  /**
   * Sets the window pixel location of the cursor when the event occurred. The
   * position is relative to the lower left corner of the window in which the event
   * occurred.
   */
  virtual void        setPosition(const SbVec2s &p)   { position = p;
                                                        positionFloat[0] = (float)position[0];
                                                        positionFloat[1] = (float)position[1];}

  /**
   *  Float version of #setPosition(const SbVec2s &p). It can be used when a desktop is magnified
   *  on a wall of screens using ScaleViz with a tracker device calibrated for this wall.
   */
  virtual void        setPosition(const SbVec2f &p)   { positionFloat = p;
                                                        position[0] = (short)positionFloat[0];
                                                        position[1] = (short)positionFloat[1];}


  /**
   * Returns the window pixel location of the cursor when the event occurred as integer values.
   * The position is relative to the lower left corner of the window in which the event
   * occurred.
   *
   * Position should be set for all events, but that may not be possible for
   * some events in some display environments. If position was not available,
   * the method returns 0,0.
   */
  const SbVec2s &     getPosition() const             { return position; }

  /**
   * Returns the window pixel location of the cursor when the event occurred as float values.
   * The position is relative to the lower left corner of the window in which the event
   * occurred.
   *
   * Position should be set for all events, but that may not be possible for
   * some events in some display environments. If position was not available,
   * the method returns 0,0.
   */
  const SbVec2f &     getPositionFloat() const             { return positionFloat; }

  /**
   * Gets the viewport pixel location of the cursor when the event occurred, relative
   * to the origin of the specified viewport region, as integer values.
   *
   * Position should be set for all events, but that may not be possible for
   * some events in some display environments. If position was not available,
   * the method returns the viewport origin.
   */
  const SbVec2s &     getPosition(const SbViewportRegion &vpRgn) const;

  /**
   * Gets the viewport pixel location of the cursor when the event occurred, relative
   * to the origin of the specified viewport region, as float values.
   *
   * Position should be set for all events, but that may not be possible for
   * some events in some display environments. If position was not available,
   * the method returns the viewport origin.
   */
  const SbVec2f &     getPositionFloat(const SbViewportRegion &vpRgn) const;

  /**
   * Gets the normalized location of the cursor when the event occurred, relative to
   * the specified viewport region. The returned value will lie between 0.0 and 1.0.
   *
   * Position should be set for all events, but that may not be possible for
   * some events in some display environments. If position was not available,
   * the method returns 0,0.
   */
  const SbVec2f & getNormalizedPosition(const SbViewportRegion &vpRgn) const;

  /**
   * Sets whether the shift key was down when the event occurred.
   */
  void                setShiftDown(SbBool isDown)     { shiftDown = isDown; }
  /**
   * Sets whether the CTRL key was down when the event occurred.
   */
  void                setCtrlDown(SbBool isDown)      { ctrlDown = isDown; }
  /**
   * Sets whether the ALT key was down when the event occurred.
   */
  void                setAltDown(SbBool isDown)       { altDown = isDown; }
  /**
   * Sets whether mouse Button 1 was down when the event occurred.
   */
  void                setButton1Down(SbBool isDown)       { button1Down = isDown; }

  /**
   * Returns true if the shift key was down when the event occurred.
   */
  SbBool              wasShiftDown() const            { return shiftDown; }
  /**
   * Returns true if the CTRL key was down when the event occurred.
   */
  SbBool              wasCtrlDown() const             { return ctrlDown; }
  /**
   * Returns true if the ALT key was down when the event occurred.
   */
  SbBool              wasAltDown() const              { return altDown; }
  /**
   * Returns true if mouse button 1 was down when the event occurred.
   */
  SbBool              wasButton1Down() const              { return button1Down; }

  /**
   * Gets whether an event object has associated tracker information.
   * [OIV-WRAPPER-RETURN-TYPE-VALUE]
   */
  virtual const SbTrackerInfo *getTrackerInfo() const { return NULL; }

  /**
   * Gets whether an event object has associated tracker information.
   * [OIV-WRAPPER-RETURN-TYPE-VALUE]
   * [OIVJAVA-WRAPPER NO_WRAP]
   */
  inline SbTrackerInfo* getTrackerInfo();

 SoINTERNAL public:
  // Initializes base event class
  static void         initClass();
  static void exitClass();

  // Initialize ALL Inventor event classes
  static void         initClasses();
  static void exitClasses();

 private:
  // all of these are set according to when the event occurred
  SbTime              timestamp;   // time the event occurred
  SbBool              shiftDown;   // TRUE if shift key was down
  SbBool              ctrlDown;    // TRUE if ctrl key was down
  SbBool              altDown;     // TRUE if alt key was down
  SbBool              button1Down; // TRUE if first mouse button was down

  SbVec2s             position;         // locator position when event occurred
  SbVec2f             positionFloat;    // locator position when event occurred
  SbVec2s             viewportPos;      // position relative to viewport
  SbVec2f             viewportPosFloat;  // position relative to viewport
  SbVec2f             normalizedPos;    // normalized position

  static SoType      classTypeId; // base typeId for all events
};

SbTrackerInfo* SoEvent::getTrackerInfo()
{
  const ::SoEvent* constEvent = this;
  return const_cast< ::SbTrackerInfo* >(constEvent->getTrackerInfo());
}

#endif /* _SO_EVENT_ */

