/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Nick Thompson (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/



#ifndef  _SO_FIELD_DATA_
#define  _SO_FIELD_DATA_

#include <Inventor/misc/SoBasic.h>
#include <Inventor/SbPList.h>
#include <Inventor/SbString.h>

class SoField;
class SoInput;
class SoFieldContainer;
class SoOutput;

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoFieldData
//
//  The SoFieldData class holds data about fields of an object: the
//  number of fields the object has, the names of the fields, and a
//  pointer to a prototype instance of the field. (Typically, this
//  prototype field is contained within a prototype instance of the
//  node or engine.)
//
//  All objects of a given class share the same field data. Therefore,
//  to get information about a particular field in a particular object
//  instance, it is necessary to pass that instance to the appropriate
//  method.
//
//////////////////////////////////////////////////////////////////////////////

SoINTERNAL class INVENTOR_API SoFieldData {
 public:
  // Constructor
  SoFieldData() {}

  // Copy constructors
  SoFieldData(const SoFieldData &);
  SoFieldData(const SoFieldData *);

  // Constructor that takes number of fields as a hint
  SoFieldData(int numFields) : fields(numFields) {}

  // Destructor
  ~SoFieldData();

  // Adds a field to current data, given default value object, name of
  // field and pointer to field within default value object. This is
  // used to define fields of a FieldContainer (node or engine) class.
  void  addField(SoFieldContainer *defObject,
                               const char *fieldName,
                               const SoField *field);

  // Copy values and flags of fields from one object to another (of
  // the same type). If copyConnections is TRUE, any connections to
  // the fields are copied as well
  void                overlay(SoFieldContainer *to,
                              const SoFieldContainer *from,
                              SbBool copyConnections) const;

  // Returns number of fields
  int                 getNumFields() const    { return fields.getLength(); }

  // Returns name of field with given index
  const SbName &      getFieldName(int index) const;

  // Returns name of field with given adress
  SbString getFieldName(const SoFieldContainer *object, const SoField* field) const;

  // Returns pointer to field with given name
  SoField* getField(const SoFieldContainer *object, const SbString& name) const;

  // Returns pointer to field with given index within given object instance
  SoField *           getField(const SoFieldContainer *object,
                               int index) const;

  // Returns index of field, given the field and the instance it is in
  int                 getIndex(const SoFieldContainer *fc,
                               const SoField *field) const;

  // Enum name/value mechanism
  void                addEnumValue(const char *typeName,
                                   const char *valName, int val);
  void                getEnumData(char *typeName, int &num,
                                  int *&vals, SbName *&names);
  void                getEnumValueName(char *typeName, int val, SbName *&name);

 SoINTERNAL public:

  // Reads into fields according to SoInput. The third
  // parameter indicates whether an unknown field should be reported
  // as an error; this can be FALSE for nodes that have children.
  // The last argument will be set to TRUE if a description of the
  // fields (using the "inputs" or "fields" keywords) was read.
  SbBool              read(SoInput *in, SoFieldContainer *object,
                           SbBool errorOnUnknownField,
                           SbBool &notBuiltIn) const;

  // This function is used when the field name has already been
  // read, and just the value needs to be read in.  It is used by
  // the above read() method and to read in GlobalFields.  It
  // returns TRUE if there were no problems reading, and will return
  // FALSE in 'foundName' if the given name is not the name of one
  // of the fields in the fieldData.
  SbBool              read(SoInput *in, SoFieldContainer *object,
                           const SbName &fieldName,
                           SbBool &foundName) const;

  // Writes all fields of object according to SoOutput
  void                write(SoOutput *out,
                            const SoFieldContainer *object) const;

  // Copies all fields from given field data into this one, creating
  // new fields for them
  void                copy(const SoFieldData *from);

  // Returns TRUE if the values in the fields of one container are
  // the same as those in another (of the same type)
  SbBool              isSame(const SoFieldContainer *c1,
                             const SoFieldContainer *c2) const;

  // Read field type information
  SbBool              readEventIn(
                                  SoInput *in, SoFieldContainer *object ) const;

  SbBool              readEventOut(
                                   SoInput *in, SoFieldContainer *object ) const;

  // Read field type information
  SbBool              readVField(
                                 SoInput *in, SoFieldContainer *object ) const;
  // Read field type information
  SbBool              readVExposedField(
                                        SoInput *in, SoFieldContainer *object ) const;

  // Read field type information
  SbBool              readFieldDescriptions(
                                            SoInput *in, SoFieldContainer *object,
                                            int numDescriptionsExpected) const;

  // Write field type information
  void                writeFieldDescriptions(
                                             SoOutput *out,
                                             const SoFieldContainer *object) const;

 private:
  // Used by binary reading to read in all the fields:
  SbBool              readFields(
                                 SoInput *in, SoFieldContainer *object,
                                 int numFieldsWritten) const;

  SbPList             fields;         // List of fields (SoFieldEntry)
  SbPList             enums;          // List of enums (SoEnumEntry)

  /**
   * SoFieldEntry holds the name and offset of a field in an SoFieldData.
   * SoEnumEntry holds the type name and name/value pairs for an enum type.
   * These are internal to Inventor. A list of each of these is stored
   * in the SoFieldData.
   */
  struct SoFieldEntry
  {
    SbName name;     // Name of field
    int64_t offset;  // Offset of field within object
    bool noObj;
  };

  struct SoEnumEntry
  {
  SbName typeName;  // Name of enum type
  int num;  // number of values
  int arraySize;  // size of arrays
  int *vals;  // array of values
  SbName *names;  // array of names

  SoEnumEntry(const SbName &name);
  SoEnumEntry(const SoEnumEntry &o);
  ~SoEnumEntry();

  static int growSize;  // amount to grow arrays
  };

};

#endif /* _SO_FIELD_DATA_ */


