/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Aug 2013)
**=======================================================================*/

#ifndef  _SO_MF_INSTANCE_PARAMETER_
#define  _SO_MF_INSTANCE_PARAMETER_

#include <Inventor/fields/SoSubField.h>
#include <Inventor/nodes/SoInstanceParameter.h>


/** 
* @VSGEXT
*   Multiple-value field containing any number of SoInstanceParameter nodes.
*
* @ingroup fields
*
* @DESCRIPTION
*   This field contains a set of SoInstanceParameter instances\if_cpp , correctly
*   maintaining their reference counts \endif\.
*
*   An SoMFInstanceParameter is written to file as one or more SoInstanceParameters.
*   When more than one value is
*   present, all of the values are enclosed in square brackets and separated by
*   commas.
*
*   Convenience methods, for example #setColors(), are provide to more easily
*   set commonly used instance parameters.
*
* @SEE_ALSO
*   SoMultipleInstance
*
* @NODE_SINCE_OIV 9.5
*/
class INVENTOR_API SoMFInstanceParameter : public SoMField {

  SO_MFIELD_REQUIRED_HEADER(SoMFInstanceParameter);
  SO_MFIELD_CONSTRUCTOR_HEADER(SoMFInstanceParameter);
  SO_MFIELD_VALUE_HEADER(SoMFInstanceParameter, SoInstanceParameter *, SoInstanceParameter *);
  SO_MFIELD_SETVALUESPOINTER_HEADER(SoInstanceParameter *);

public:
  /**
   * Returns the first parameter with the specified name. Returns NULL if not found.
   */
  SoInstanceParameter* findParameterByName(const SbString& name) const;

  /**
   * Sets the value of this parameter from the input array.
   * @param values The color values as SbColor.
   * @param nbValues The number of values contained in the input array.
   * @param divisor The number of instances that share the same value.
   * @param policy Whether to copy the input values or not. NO_COPY_AND_DELETE and NO_COPY_AND_FREE are treated as NO_COPY.
   */
  SoInstanceParameter* setColors( SbColor* values,
                                  size_t nbValues,
                                  uint32_t divisor = 1,
                                  SoMemoryObject::CopyPolicy policy = SoMemoryObject::NO_COPY );

  /**
   * Sets the value of this parameter from the input array.
   * @param values The transparency values as float.
   * @param nbValues The number of values contained in the input array.
   * @param divisor The number of instances that share the same value.
   * @param policy Whether to copy the input values or not. NO_COPY_AND_DELETE and NO_COPY_AND_FREE are treated as NO_COPY.
   */
  SoInstanceParameter* setTransparencies( float* values,
                                          size_t nbValues,
                                          uint32_t divisor = 1,
                                          SoMemoryObject::CopyPolicy policy = SoMemoryObject::NO_COPY );

  /**
   * Sets the value of this parameter from the input array.
   * @param values The scale values as SbVec3f.
   * @param nbValues The number of values contained in the input array.
   * @param divisor The number of instances that share the same value.
   * @param policy Whether to copy the input values or not. NO_COPY_AND_DELETE and NO_COPY_AND_FREE are treated as NO_COPY.
   */

  SoInstanceParameter* setScales( SbVec3f* values,
                                  size_t nbValues,
                                  uint32_t divisor = 1,
                                  SoMemoryObject::CopyPolicy policy = SoMemoryObject::NO_COPY );

  /**
   * Sets the value of this parameter from the input array.
   * @param values The rotation values as SbVec4f (quaternions).
   * @param nbValues The number of values contained in the input array.
   * @param divisor The number of instances that share the same value.
   * @param policy Whether to copy the input values or not. NO_COPY_AND_DELETE and NO_COPY_AND_FREE are treated as NO_COPY.
   */
  SoInstanceParameter* setRotations( SbVec4f* values,
                                     size_t nbValues,
                                     uint32_t divisor = 1,
                                     SoMemoryObject::CopyPolicy policy = SoMemoryObject::NO_COPY );

  /**
   * Sets the value of this parameter from the input array.
   * @param values The rotation values as SbRotation.
   * @param nbValues The number of values contained in the input array.
   * @param divisor The number of instances that share the same value.
   * @param policy Whether to copy the input values or not. NO_COPY_AND_DELETE and NO_COPY_AND_FREE are treated as NO_COPY.
   */
  SoInstanceParameter* setRotations( SbRotation* values,
                                     size_t nbValues,
                                     uint32_t divisor = 1,
                                     SoMemoryObject::CopyPolicy policy = SoMemoryObject::NO_COPY );

  /**
   * Sets the value of this parameter from the input array.
   * @param values The visibility values as int32_t.
   * @param nbValues The number of values contained in the input array.
   * @param divisor The number of instances that share the same value.
   * @param policy Whether to copy the input values or not. NO_COPY_AND_DELETE and NO_COPY_AND_FREE are treated as NO_COPY.
   */
  SoInstanceParameter* setVisibilities( int32_t* values,
                                        size_t nbValues,
                                        uint32_t divisor = 1,
                                        SoMemoryObject::CopyPolicy policy = SoMemoryObject::NO_COPY );

  /**
   * Sets the value of this parameter from the input array.
   * @param values The translation values as SbVec3f.
   * @param nbValues The number of values contained in the input array.
   * @param divisor The number of instances that share the same value.
   * @param policy Whether to copy the input values or not. NO_COPY_AND_DELETE and NO_COPY_AND_FREE are treated as NO_COPY.
   */
  SoInstanceParameter* setTranslations( SbVec3f* values,
                                        size_t nbValues,
                                        uint32_t divisor = 1,
                                        SoMemoryObject::CopyPolicy policy = SoMemoryObject::NO_COPY );

  /**
   * Remove the instance parameter with the specified name.
   * Returns true if successful, false if not found.
   */
  bool removeParameterByName(const SbString& name);

  /**
   * Returns a pointer to the internally maintained array that can be modified.
   * The values in the array may be changed, but values cannot be added or removed. It is illegal to call any other
   * editing methods between startEditing() and finishEditing() (e.g. set1Value(), setValue(), etc.).
   *
   * Fields, engines or sensors connected to this field and sensors are not notified that this field has changed until
   * finishEditing() is called. Calling finishEditing() always sets the isDefault() flag to FALSE and informs engines
   * and sensors that the field changed, even if none of the values actually were changed.
   * [OIVNET-WRAPPER-RETURN-TYPE ARRAY{Count}]
   */
  SoInstanceParameter** startEditing();

  /** Indicates that batch edits have finished. @see startEditing(). */
  void finishEditing();

  SoINTERNAL public:
  static void initClass();
  static void exitClass();

  // Update a copied field to use the copy of each uniform shader parameter if there is one
  virtual void fixCopy(SbBool copyConnections);

  // Override this to also check the stored uniform shader parameters
  virtual SbBool referencesCopy() const;

private:
  // Override this to maintain write-references in uniform shader parameters
  virtual void countWriteRefs(SoOutput *out) const;

  // Changes value in field without doing other notification stuff.
  // Keeps track of references and auditors.
  void setVal(int index, SoInstanceParameter *newValue);

  /**
   * Return true if param has the given name
   */
  static bool hasGivenName(SbString name, const SoShaderParameter* param);

  SoInstanceParameter** m_oldValues;
  int m_oldNum;
};


#endif /* _SO_MF_INSTANCE_PARAMETER_ */
