/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_MF_VEC3F_
#define  _SO_MF_VEC3F_

#include <Inventor/fields/SoSubField.h>
#include <Inventor/SbLinear.h>

//////////////////////////////////////////////////////////////////////////////
//
//  SoMFVec3f subclass of SoMField.
//
//////////////////////////////////////////////////////////////////////////////

/**
* Multiple-value field containing any number of three-dimensional vectors.
* 
* @ingroup fields
* 
* @DESCRIPTION
*   A multiple-value field that contains any number of three-dimensional vectors.
*   
*   SoMFVec3fs are written to file as one or more triples of floating point values
*   separated by whitespace.
*   
*   When more than one value is present, all of the values are enclosed in square
*   brackets and separated by commas; for example:
*   
*   \verbatim
     [ 0 0 0, 1.2 3.4 5.6, 98.6 -4e1 212 ]
    \endverbatim
*
*   @B Using the setValues() method: @b
*
*   Note: If you use the setValues() method to set a larger number of values than
*   the field currently contains, Open Inventor will automatically allocate more
*   memory. The total number of values in the field is increased as expected.
*   However the converse is not true.  If you use setValues() to set a @I smaller@i
*   number of values than the field currently contains, you are simply overriding
*   some of the values in the field.  The total number of values in the field does
*   @I not@i change.  If you want to replace the current contents of the field with
*   a smaller number of values, first call setNum( 0 ), then call setValues().
*
*   @B Using an Application-Supplied Array @b
*   
*   \if_cpp
*   The setValuesPointer() methods allow Open Inventor to directly use an array
*   of values supplied by the application. The application data is not copied
*   and the memory will not be deleted by Open Inventor unless enableDeleteValues()
*   is called. 
*   NOTE: When setValuesPointer( 0, 0 ) is called, the previously set buffer and
*   number of items are @I not@i reset. Call the deleteValues( 0 ) method
*   instead to reset the field's content.
*   \else 
*   The \if_dotnet SetValuesBuffer() \else setValuesBuffer() \endif methods allow Open Inventor to directly use an array
*   of values supplied by the application. The application data is not copied.
*   \endif
*   
*   When using application data directly the values may be modified by changing
*   the application memory and calling touch() to notify Open Inventor of the change.
*   The values may also be modified using the usual methods (set1Value(), etc.)
*   
*   However, note that some methods (setNum(), deleteValues(),
*   insertSpace(), setValues(), set1Value(), setValue()) may
*   force Open Inventor to allocate a larger block of memory to hold all the values.
*   In this case, Open Inventor will allocate memory internally and copy the data.
*   Modifying values in the application memory will not affect the field if Open Inventor has
*   allocated new memory.
*   \if_cpp The application is still responsible for freeing its memory (using free()) unless 
*   enableDeleteValues() has been called.\endif 
*
*   Example:
*   \if_cpp
*   \par
*   \code
*   // Allocate memory for vertices
*   SbVec3f* vertices = new SbVec3f[NUM_VERTICES]; 
*
*   // Assign values to the vertices (application-specific code)
*   //...
*
*   // Set the field's value to vertices in application memory
*   SoVertexProperty* vertexProperty = new SoVertexProperty();
*   vertexProperty->vertex.setValuesPointer( NUM_VERTICES, vertices ); 
*
*   // Change application data and notify Open Inventor
*   vertices[5].setValue( 0, 1, 2 );
*   vertexProperty->vertex.touch(); 
*   \endcode
*   \endif 
*
*   \if_dotnet
*   \par
*   \code
*   // Allocate memory for vertices
*   SbVec3f[] vertices = new SbVec3f[NUM_VERTICES]; 
*   
*   // Assign values to the vertices (application-specific code)
*   //...
*   
*   // Set the field's value to vertices in application memory
*   SoVertexProperty vertexProperty = new SoVertexProperty();
*   vertexProperty.vertex.SetValuesBuffer( vertices ); 
*   
*   // Change application data and notify Open Inventor
*   vertices[5].SetValue( 0, 1, 2 );
*   vertexProperty.vertex.Touch(); 
*   \endcode
*   \endif 
*
*   \if_java
*   \par
*   \code
*   // Allocate memory for vertices
*   int numBytes = NUM_VERTICES * 4; 
*   ByteBuffer vertexBuffer = ByteBuffer.allocateDirect( numBytes );
*   vertexBuffer.order( ByteOrder.nativeOrder() );
*   FloatBuffer vertexFBuffer = vertexBuffer.asFloatBuffer();
*
*   // Assign values to the vertices (application-specific code)
*   // . . .
*
*   // Set the field's value to vertices in application memory
*   SoVertexProperty vertexProperty = new SoVertexProperty();
*   vertexProperty.vertex.setValuesBuffer( vertexBuffer );
*            
*   // Change application data and notify Open Inventor
*   vertexFBuffer.put( 4, 2.0f );
*   vertexProperty.vertex.touch();
*   \endcode
*   \endif 
* 
*/
class INVENTOR_API SoMFVec3f : public SoMField {
  // Use standard field stuff
  SO_MFIELD_HEADER(SoMFVec3f, SbVec3f, const SbVec3f &);

  SO_MFIELD_SETVALUESPOINTER_HEADER(float);
  SO_MFIELD_SETVALUESPOINTER_HEADER(SbVec3f);
  SO_FIELD_SUPPORT_MEMOBJ();

    public:
  
  //
  // Some additional convenience functions:
  //
  /**
   * \if_java
   * Sets values from the specified array of floats.
   * The size of the array must be a multiple of 3.
   * \else
   * Sets values from array of arrays of 3 floats.
   * \endif
   * [OIVNET-WRAPPER-ARG INDEX{0,},IN,IN]
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValues(start, num)}]
   * [OIVJAVA-WRAPPER-ARG IN,NO_WRAP{(num)},IN]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{int num = xyz != null ? xyz.length/3 : 0; onSetValues(start, num)}]
   */
  void setValues(int start, int num, const float xyz[][3]);
#ifdef _WIN32
  /**
   * Need overload to work around MS compiler bug (fixed in VC++ 5.0)
   * [OIV-WRAPPER-NO-WRAP]
   */
  void  setValues(int start, int num, float xyz[][3]);
#endif
 
  /**
   * Sets one value from 3 floats.
   * [OIVNET-WRAPPER-ARG INDEX{0,},IN,IN,IN]
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSet1Value(index)}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSet1Value(index)}]
   */
  void  set1Value(int index, float x, float y, float z);
  
  /**
   * Sets one value from 3 floats in array.
   * [OIVNET-WRAPPER-ARG INDEX{0,},IN]
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSet1Value(index)}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSet1Value(index)}]
   */
  void  set1Value(int index, const float xyz[3]);
  
  /**
   * Sets to one value from 3 floats.
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */
  void  setValue(float x, float y, float z);
  
  /**
   * Sets to one value from 3 floats in array.
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */
  void  setValue(const float xyz[3]);
  
  /**
   * Sets one value from double precision vector.
   * [OIVNET-WRAPPER-ARG INDEX{0,},IN]
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSet1Value(index)}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSet1Value(index)}]
   */
  void  set1Value(int index, const SbVec3d &vec3d);
  
  /**
   * Sets to one value from double precision vector.
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */
  void  setValue(const SbVec3d &vec3d);
  
  /**
   * Sets values from array of double precision vectors.
   * [OIVNET-WRAPPER-ARG INDEX{0,},NO_WRAP{vec3d->Length},ARRAY]
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValues(start, vec3d->Length)}]
   * [OIVJAVA-WRAPPER-ARG IN,NO_WRAP{num},ARRAY]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{int num = vec3d != null ? vec3d.length : 0; onSetValues(start, num)}]
   */
  void  setValues(int start, int num, const SbVec3d *vec3d);
  
 SoINTERNAL public:
  static void initClass();
  static void exitClass();
  virtual void writeBinaryValues(SoOutput*, int , int ) const;
  virtual SbBool readBinaryValues( SoInput*, int, int );

private:
  // Write the values out as a block of data
  virtual void  writeBinaryValues(SoOutput *out) const;
  virtual SbBool        readBinaryValues(SoInput *in, int numToRead);

};

#endif /* _SO_MF_VEC3F_ */

