#pragma once

/*====================================================================================
*** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
***                                                                                ***
***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
***                                                                                ***
***                        RESTRICTED RIGHTS LEGEND                                ***
***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
***                                                                                ***
***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
***                        BORDEAUX, FRANCE                                        ***
***                      ALL RIGHTS RESERVED                                       ***
**==================================================================================*/

#include <Inventor/helpers/SbGraphicsCapabilities.h>
#include <Inventor/SbString.h>
#include <Inventor/SbVec.h>
#include <Inventor/SbBase.h>


/**
 * @ingroup InventorHelpers
 *
 * List of common graphics capabilities of the device used by a SoGLContext.
 *
 * This structure aims at making algorithms scalable according to the capabilities
 * of the device used for execution. For instance, a texture can be resized if its
 * size is greater than supported hardware size, such as:
 *
 * \if_cpp
 *   \par
 *   \code
 *    const SbGraphicsCapabilities& caps = SoGLContext::getGraphicsCapabilities();
 *    if (texSize > caps.maxTextureSize)
 *      resizeTexture(caps.maxTextureSize);
 *   \endcode
 * \endif
 *
 * @SEE_ALSO SbGraphicsCapabilities, SoGLContext, SoGLDevice
 */
class INVENTORBASE_API SbGPUCapabilities : public SbGraphicsCapabilities
{
public:

  /** enumeration of OpenGL profiles */
  enum Profile
  {
    COMPATIBILITY = 0x1,
    CORE = 0x2
  };

  /** enumeration of GPU vendor */
  enum GpuVendor
  {
    INTEL_GPU,
    AMD_GPU,
    NVIDIA_GPU,
    UNKNOWN_GPU
  };

  /** OpenGL version. Ex: 440 for OpenGL 4.4*/
  int getVersion() const;

  /** GLContext profile. See Profile enumeration. */
  Profile getProfile() const;

  /** GPU vendor */
  GpuVendor getGpuVendor() const;

  /** Renderer name. Format depends on GPU */
  const SbString& getRenderer() const;

  /** GL_MAX_VIEWPORT_DIMS */
  const SbVec2i32& getMaxViewportSize() const;

  /** GL_MAX_TEXTURE_SIZE */
  int getMaxTextureSize() const;

  /** GL_MAX_3D_TEXTURE_SIZE */
  int getMax3DTextureSize() const;

  /** GL_MAX_VERTEX_ATTRIBS */
  int getMaxVertexAttribs() const;

  /** GL_MAX_TEXTURE_UNITS (i.e. max texture units for fixed pipeline) */
  int getMaxFixedPipelineTextureUnits() const;

  /** GL_MAX_SAMPLES (i.e. max number of samples for multisampling) */
  int getMaxSamples() const;

  /** List capabilities in human readable format */
  INVENTORBASE_API friend std::ostream& operator<<(std::ostream& os, const SbGPUCapabilities& capabilities);

protected:

  // protect from suspicious instancing
  SbGPUCapabilities();

  // OpenGL version
  int m_version;

  // GLContext profile
  Profile m_profile;

  // GPU Vendor
  GpuVendor m_vendor;

  // Name of renderer
  SbString m_renderer;

  // GL_MAX_VIEWPORT_DIMS
  SbVec2i32 m_maxViewportSize;

  // GL_MAX_TEXTURE_SIZE
  int m_maxTextureSize;

  // GL_MAX_3D_TEXTURE_SIZE
  int m_max3DTextureSize;

  // GL_MAX_VERTEX_ATTRIBS
  int m_maxVertexAttribs;

  // GL_MAX_TEXTURE_UNITS (i.e. max texture units for fixed pipeline)
  int m_maxFixedPipelineTextureUnits;

  // GL_MAX_SAMPLES
  int m_maxSamplesMultiSampling;
};
