/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : C. OGNIER (Jan 2005)
**=======================================================================*/

#ifndef _SODDSIMAGERW_
#define _SODDSIMAGERW_


#include <Inventor/image/SoRasterImageRW.h>
#include <Inventor/threads/SbThreadMutex.h>

/**
 * @VSGEXT Class for decoding a DDS raster image.
 * 
 * @ingroup Image
 *
 * @DESCRIPTION
 * This class is used for decoding a DDS (DirectDraw Surface) raster image.
 *
 * DDS files are faster to load and use less memory than other image file formats,
 * because the compressed data is sent directly to the graphics device (no
 * decompression is done on the CPU).
 *
 * Formats: @BR
 * This class is able to read DDS files compressed with DXT1 with 1-bit Alpha,
 * DXT3, and DXT5 compression. The other formats which can be contained by
 * a DDS file (e.g. DXT2 and DXT4) are not currently supported.
 *
 * Mipmaps: @BR
 * This format can also embed mipmaps (levels of resolution), 
 * but all levels (from higher resolution to a 1*1 pixel image) are 
 * not necessarily required.
 * The SoComplexity::textureQuality field has no effect on 
 * mipmap generation if the DDS file already embeds mipmaps.
 *
 * NOTE: File encoding and writing is not currently supported.
 *
 * See SoRasterImageRW for more information and code examples.
 * 
 * @SEE_ALSO
 *    SoRasterImageFile,
 *    SoRasterImageRW,
 *    SbRasterImage,
 *    SoComplexity
 * 
 * 
 */
class INVENTOR_API SoDDSImageRW : public SoRasterImageRW {

    SO_IMAGE_RASTER_RW_HEADER()

public:

  /**
   * Constructor.
   */
  SoDDSImageRW();

  /**
   * Destructor.
   */
  virtual ~SoDDSImageRW();

  /**
   * Opens the reader/writer in the specified open mode.
   */
  virtual SbBool open(SoRasterImageIO* rasterImageIO, OpenMode openMode);

  /**
   * Writes and encodes the given data in the specific format. <BR>
   * @B Not currently supported for this format.@b
   */
  virtual SbBool write(SbRasterImage* rasterImage, unsigned int xPos = 0, unsigned int yPos = 0);

  /**
   * Reads, decodes, and fills the @I rasterImage@i object.
   * If @I infoOnly@i is TRUE, then the buffer will not be read, the parameter @I rasterImage@i will be
   * set with raster size and raster number of components, the buffer will be NULL.
   */
  virtual SbBool read(SbRasterImage* rasterImage, SbBool infoOnly = FALSE);

  /**
   * Specifies the write direction when using multiple buffers.
   * Returns FALSE if buffers are written from top to bottom.
   * Returns TRUE if buffers are written from bottom to top.
   */
  virtual SbBool isMultipleBufferInverted() const;

  /**
   * Returns the write capability of the raster format.
   */
  virtual SoRasterImageRW::WriteCapability getWriteCapability() const;

  /**
   * Returns the read capability of the raster format.
   */
  virtual SoRasterImageRW::ReadCapability getReadCapability() const;


protected:
  virtual void createSuffixList();

  void readAndConvert(void* dd, FILE* fp);

private:
  static SbThreadMutex s_mutex;

  /**
   * Size of a pixel in bytes.
   */
  static const size_t s_pixelSize;

  /**
   * Number of pixels on each side of a block.
   */
  static const size_t s_numPixelsPerBlockSide;

  /**
   * Indicates if the image must be flipped or not.
   */
  bool m_isFlipNeeded;

  /**
   * Provides size of the buffer which must be allocated to contain the data read in the DDS file.
   */
  size_t getBufferSize(size_t topLevelTextureSize, size_t numMipMaps) const;

};

#endif
