/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : C. OGNIER (Jun 2003)
**=======================================================================*/



#ifndef _SORASTERIMAGERW_
#define _SORASTERIMAGERW_

#include <Inventor/nodes/SoNode.h>
#include <Inventor/image/SoRasterImageIO.h>
#include <Inventor/image/SbRasterImage.h>
#include <Inventor/errors/SoDebugError.h>
#include <Inventor/image/SoSubRasterImageRW.h>
#include <Inventor/SbPImpl.h>

class SoRasterImageFile; // FORWARD

SO_PIMPL_BASE_PUBLIC_DECLARATION(SoRasterImageRW);

/**
* @VSGEXT Abstract base class for encoding and decoding raster images.
*
* @ingroup Image
*
* @DESCRIPTION
* SoRasterImageRW is the base class used for encoding and decoding raster images.
*
* The following file formats are supported for raster image input (i.e., read):
* - BMP (Windows only)
* - DDS
* - GIF
* - HDRI (with SoTextureCubeMap only)
* - JPEG 2000
* - JPEG
* - PNG
* - SGI RGBA
* - TIFF
*
* The following formats are supported for raster image output (i.e., write):
* - All the above except GIF (read only), plus
* - PS (PostScript), which is available for writing only.
*
* Note that only the following image file formats can contain transparency
* (alpha channel) information:
* - DDS
* - GIF
* - PNG
* - SGI RGBA
* - TIFF
*
* There is a subclass of SoRasterImageRW for each of the above formats.
*
* The convenience class SoRasterReaderSet can be used to determine the
* appropriate reader class for an image file by attempting to read the
* file using each of the built-in reader classes.
*
* A subclass of SoRasterImageIO, e.g. SoRasterImageFile, must be used as
* the source of a read operation or the destination of a write operation.
*
* Note that it is not necessary to explicitly load image files for
* common operations like texture mapping. SoTexture2, for example,
* automatically loads the image file specified in its filename field.
*
* SoRasterImageRW classes support two methods of writing out an image.
* The simplest one is writing out a complete image already assembled
* in memory.  This is the default method and is efficiently supported
* by all image formats.  SoRasterImageRW also supports assembling an
* image from multiple sub-images or "tiles".  The SoOffscreenRenderArea
* class, for example, uses this feature when the requested image size
* is too large for OpenGL to render as a single image.  To use this
* feature call the enableMultipleWriting method with TRUE.  Note that
* some image formats allow the image to be written (encoded) incrementally,
* so tiles provided in scan-line order can written immediately, with no
* intermediate storage. Other image formats cannot be encoded until the
* complete image is assembled in memory.
*
* Image formats that allow tiles to be written incrementally have the
* writeCapability WRITE_SCANLINES. This is the most memory efficient
* way to write large images.  Note that some image formats are encoded
* from top to bottom and some are encoded from bottom to top.  The
* method isMultipleBufferInverted returns TRUE if the format should be
* encoded from bottom to top.
*
* Image formats that require a complete image before encoding have the
* writeCapability WRITE_FULL_IMAGE.  In this case the SoRasterImageRW
* subclass will automatically allocate enough memory to hold the
* complete image.  This may require a very large block of contiguous
* memory!  The image is not actually written until the writeFooter
* method is called.
*
* @EXAMPLE
* Load an image from a PNG format file (error checking is omitted for clarity):
* \if_cpp
* \code
*   SoRasterImageFile inputFile( "test.png" );
*
*   SoPNGImageRW imageReader;
*   imageReader.open( &inputFile, SoRasterImageRW::OPEN_READ );
*
*   SbRasterImage image;
*   imageReader.read( &image );
*   imageReader.close();
* \endcode
* \endif
* \if_dotnet
* \code
*   SoRasterImageFile inputFile = new SoRasterImageFile( "test.png" );
*
*   SoPNGImageRW imageReader = new SoPNGImageRW();
*   imageReader.Open( inputFile, SoRasterImageRW.OpenModes.OPEN_READ );
*
*   SbRasterImage image = new SbRasterImage();
*   imageReader.Read(image);
*   imageReader.Close();
* \endcode
* \endif
* \if_java
* \code
*   SoRasterImageFile inputFile = new SoRasterImageFile( "test.png" );
*
*   SoPNGImageRW imageReader = new SoPNGImageRW();
*   boolean ok = imageReader.open( inputFile, SoRasterImageRW.OpenModes.OPEN_READ );
*
*   SbRasterImage image = new SbRasterImage();
*   imageReader.read( image );
*   imageReader.close();
* \endcode
* \endif
*
* @EXAMPLE
* Write an image to a JPEG format file (error checking is omitted for clarity):
* \if_cpp
* \code
*   SoRasterImageFile outputFile( "test.jpg" );
*
*   SoJPEGImageRW imageWriter;
*   imageWriter.open( &outputFile, SoRasterImageRW::OPEN_WRITE );
*
*   SbRasterImage image . . .; // Previously created
*   SbVec2i32 size = image.getSize_i32();
*   imageWriter.writeHeader( size );
*   imageWriter.write( &image );
*   imageWriter.writeFooter();
*   imageWriter.close();
* \endcode
* \endif
* \if_dotnet
* \code
*   SoRasterImageFile outputFile = new SoRasterImageFile("test.jpg");
*
*   SoJPEGImageRW imageWriter = new SoJPEGImageRW();
*   imageWriter.Open(outputFile, SoRasterImageRW.OpenModes.OPEN_WRITE);
*
*   SbRasterImage image . . .; // Previously created
*   SbVec2i32 size = image.GetSize_i32();
*   imageWriter.WriteHeader( size );
*   imageWriter.Write( image );
*   imageWriter.WriteFooter();
*   imageWriter.Close();
* \endcode
* \endif
* \if_java
* \code
*   SoRasterImageFile outputFile = new SoRasterImageFile( "test.jpg" );
*
*   SoJPEGImageRW imageWriter = new SoJPEGImageRW();
*   imageWriter.open( outputFile, SoRasterImageRW.OpenModes.OPEN_WRITE );
*
*   SbRasterImage image . . .; // Previously created
*   SbVec2i32 size = image.getSizei32();
*   imageWriter.writeHeader( size );
*   imageWriter.write( image );
*   imageWriter.writeFooter();
*   imageWriter.close();
* \endcode
* \endif
*
* @SEE_ALSO
*    SoBMPImageRW,
*    SoDDSImageRW,
*    SoGIFImageRW,
*    SoJP2ImageRW,
*    SoJPEGImageRW,
*    SoPGXImageRW,
*    SoPNGImageRW,
*    SoPNMImageRW,
*    SoPSImageRW,
*    SoSGIRGBImageRW,
*    SoSUNImageRW,
*    SoTIFFImageRW,
*    SoRasterReaderSet
*
*/
class INVENTOR_API SoRasterImageRW 
{

  SO_IMAGE_RASTER_RW_ABSTRACT_HEADER()
  SO_PIMPL_BASE_PUBLIC_HEADER(SoRasterImageRW);

public:

  /** Write Capability enumeration values. */
  enum WriteCapability {
    /**
     * Can write an image incrementally (e.g., by scan-line).
     */
    WRITE_SCANLINES,
    /**
     * Can only write a complete image.
     */
    WRITE_FULL_IMAGE,
    /**
     * No writing available.
     */
    WRITE_UNAVAILABLE
  };

  /** Read Capability enumeration values. */
  enum ReadCapability {
    /**
     * Read functionality available.
     */
    READ_AVAILABLE,
    /**
     * No reading available.
     */
    READ_UNAVAILABLE
  };

  /** open Mode enumeration values. */
  enum OpenMode {
    /**
     * Opened for reading.
     */
    OPEN_READ,
    /**
     * Opened for writing.
     */
    OPEN_WRITE
  };

  /**
   * Destructor.
   */
  virtual ~SoRasterImageRW();

  /**
   * Opens the reader/writer in the specified open mode.
   * Returns true if successful.
   */
  virtual SbBool open(SoRasterImageIO* rasterImageIO, OpenMode openMode);

  /**
   * Closes the reader/writer.
   */
  virtual void close();

  /**
   * Checks if the specified file can be read.
   * Returns true if successful.
   */
  virtual SbBool checkRead( SoRasterImageFile* imageFile );


  /**
   * Writes and encodes the given data in the specific format.
   * Returns true if successful.
   * If you are using the multiple writing mode, then you can give the
   * offset with (xPos,yPos) you want to write to.
   */
  virtual SbBool write(SbRasterImage* rasterImage, unsigned int xPos = 0, unsigned int yPos = 0) = 0;

  /**
   * Writes and encodes the header for this specific format.
   * Returns true if successful.
   * The parameter is the size of the whole raster image to be saved.
   * Use for sizes with at least one side greater than 32767.
   */
  virtual SbBool writeHeader( const SbVec2i32& size );

#if SoDEPRECATED_BEGIN(9700)
  /**
   * Writes and encodes the header for this specific format.
   * Returns true if successful.
   * The parameter is the size of the whole raster image to be saved.
   * Use for sizes with at least one side greater than 32767.
   * [OIV-WRAPPER-NO-WRAP]
   */
  SoDEPRECATED_METHOD( 9700, "Use writeHeader(const SbVec2i32& size) instead." )
  virtual SbBool writeHeader( SbVec2i32& size );
#endif /** @DEPRECATED_END */

  /**
   * Writes and encodes the footer for this specific format.
   * Returns true if successful.
   */
  virtual SbBool writeFooter();


  /**
   * Read the current open image into rasterImage.
   * Returns true if successful. Call the open() method before calling this method.
   *
   * If @I infoOnly@i is TRUE, then the buffer will not be read; the parameter @I rasterImage@i will be
   * set with raster size and raster number of components, the buffer will be NULL.
   */
  virtual SbBool read(SbRasterImage* rasterImage, SbBool infoOnly = FALSE) = 0;


  /**
   *  Read the specified region of current open image into rasterImage.
   *  Returns true if successful. Call the open() method before calling this method.
   *
   *  After calling this method successfully, the size of the SbRasterImage is the
   *  requested region size. The requested region may extend outside the source image.
   *  In this case, undefined pixels are set to zero. The region may be as small as a
   *  single pixel, e.g. (1,1,1,1), but xmin must be <= xmax and ymin <= ymax.
   */
  virtual SbBool readRegion(SbRasterImage* rasterImage, const SbBox2i32& region);


  /**
   * Returns the list of file suffixes supported.
   * \if_cpp Return value is a pointer to an array containing numSuffixes SbString objects.
   * This is internal object memory. Do NOT delete or modify. \endif
   * [OIV-WRAPPER-ARG NO_WRAP]
   * [OIV-WRAPPER-RETURN-TYPE ARRAY{numSuffixes}] 
   */
  virtual SbString* getSuffixes(int& numSuffixes) const;


  /**
   * Returns the write order when using multiple buffers.
   * Returns FALSE if buffers are written from top to bottom.
   * Returns TRUE if buffers are written from bottom to top.
   * Returns TRUE by default.
   */
  virtual SbBool isMultipleBufferInverted() const;


  /**
   * Returns the write capability of the raster format.
   * Returns WRITE_SCANLINES by default.
   */
  virtual SoRasterImageRW::WriteCapability getWriteCapability() const;


  /**
   * Returns the read capability of the raster format.
   * Returns READ_AVAILABLE by default.
   */
  virtual SoRasterImageRW::ReadCapability getReadCapability() const;


  /**
   * Enable writing image with multiple calls to write method.
   * Default is FALSE.
   */
  virtual void enableMultipleWriting(SbBool state);


  /**
   * Returns TRUE if multiple buffer writing is enabled.
   */
  virtual SbBool isMultipleWritingEnabled() const;

#if SoDEPRECATED_BEGIN(8000)

  /**
   * Writes and encodes the header for this specific format.
   * Returns true if successful.
   * The parameter is the size of the whole raster image to be saved.
   * The size is limited to 32767 by 32767
   * [OIV-WRAPPER-ARG CONST]
   */
  SoDEPRECATED_METHOD_NOWARN(8000,"Use writeHeader(SbVec2i32& size) instead.")
  virtual SbBool writeHeader(SbVec2s& size);

#endif /** @DEPRECATED_END */

SoINTERNAL public:

  /**
   * Read the number of images in the file
   *
   * @return success
   */
  virtual SbBool getImagesNumber(size_t& imagesNumber);


  /**
   * Reads, decodes, and fills the @I rasterImage@i parameter with the image at the given index
   * If @I infoOnly@i is TRUE, then the buffer will not be read; the parameter @I rasterImage@i will be
   * set with raster size and raster number of components, the buffer will be NULL.
   */
  virtual SbBool read(SbRasterImage* rasterImage, size_t imageIndex, SbBool infoOnly = FALSE);


  /**
   * Read the specified region at sub image index of currently opened image,
   * The returned rasterImage correspond to requested region. After calling
   * readRegion(), rasterImage->getSize() == region.getSize().
   * Region outside of image are filled with 0.
   *
   * @return success
   */
  virtual SbBool readRegion(SbRasterImage* rasterImage, size_t imageIndex, const SbBox2i32& region);

  /**
   * Return origin of given image, ie the position of lower left corner of lower
   * left pixel.
   * Return (0, 0, 0) if origin is not defined in image file or if
   * imageId >= getImagesNumber().
   */
  virtual SbVec3f getOrigin(const size_t& imageId);

  /**
   * Return size of pixel along X and Y.
   * Return (1, 1) if pixel size is not defined in image file or if
   * imageId >= getImagesNumber().
   */
  virtual SbVec2f getPixelSize(const size_t& imageId);

  /**
   * Return thickness of image.
   * Return 1.0 if thickness is not defined in image file or if
   * imageId >= getImagesNumber().
   */
  virtual float getThickness(const size_t& imageId);

  /**
   * Return normalized X vector of image, ie vector along lines of image.
   * Return (1, 0, 0) if not defined in image file or if
   * imageId >= getImagesNumber().
   */
  virtual SbVec3f getX(const size_t& imageId);

  /**
   * Return normalized Y vector of image, ie vector along rows of image.
   * Return (0, 1, 0) if not defined in image file or if
   * imageId >= getImagesNumber().
   */
  virtual SbVec3f getY(const size_t& imageId);

  /**
   * Return normalized Z vector of image, ie vector normal to image.
   * It is generally equal to getX().cross(getY()).
   * Return (0, 0, 1) if not defined in image file or if
   * imageId >= getImagesNumber().
   */
  virtual SbVec3f getZ(const size_t& imageId);

  /** Returns true if the writer supports alpha channel. */
  virtual bool supportsAlpha() const;

  /**
   * Write a RasterImage to the given file.
   * The file format is derived from the filename extension.
   */
  static bool write(const SbString& filename, const SbRasterImage& rasterImage);

protected:

  // This method must fill m_suffixes and update m_numSuffixes
  virtual void createSuffixList() = 0;
  SoRasterImageRW();

  SoRasterImageIO* m_rasterImageIO;
  SbBool           m_multipleWriting;
  SbVec2i32        m_rasterSize;
  unsigned int     m_multipleWritingNumComponents;
  SbBool           m_firstMultipleImages;

  SbString*        m_suffixes;
  int              m_numSuffixes;
};

#endif
